<?php
// app/baseKRIZAN/Bootstrap/Bootstrap.php

namespace baseKRIZAN\Bootstrap;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Error\UnifiedErrorHandler;

/**
 * Main Bootstrap class that manages the bootstrap process.
 * 
 * This class implements a modular approach to the bootstrap process
 * using the Initializer pattern for component initialization.
 */
class Bootstrap
{
    /**
     * Singleton instance
     * 
     * @var self|null
     */
    private static ?self $instance = null;
    
    /**
     * Container for dependency injection
     * 
     * @var Container
     */
    private Container $container;
    
    /**
     * Indicates if bootstrap has been executed
     * 
     * @var bool
     */
    private bool $bootstrapped = false;
    
    /**
     * Loaded initializers
     * 
     * @var array<string, object>
     */
    private array $initializers = [];
    
    /**
     * Initializer configuration by bootstrap mode
     * 
     * @var array<string, array<string>>
     */
    private array $initializersByMode = [
        'minimal' => [
            'EnvironmentInitializer',
            'ConfigurationInitializer',
            'LoggingAndErrorInitializer',
            'DatabaseInitializer',
            'CacheInitializer',
            'SessionInitializer',
            'SecurityInitializer',
            'MiddlewareInitializer'
        ],
        'api' => [
            'EnvironmentInitializer',
            'ConfigurationInitializer',
            'LoggingAndErrorInitializer',
            'DatabaseInitializer',
            'CacheInitializer',
            'EventInitializer',
            'SessionInitializer',
            'SecurityInitializer',
            'TemplateInitializer',
            'ModuleInitializer',
            'MiddlewareInitializer', 
            'RouteInitializer',
            'ValidationInitializer',
            'DependencyInitializer',
            'ApplicationInitializer'
        ],
        'console' => [
            'EnvironmentInitializer',
            'ConfigurationInitializer',
            'LoggingAndErrorInitializer',
            'DatabaseInitializer',
            'CacheInitializer',
            'EventInitializer',
            'ModuleInitializer',
            'ValidationInitializer',
            'DependencyInitializer',
            'ApplicationInitializer'
        ],
        'full' => [
            'EnvironmentInitializer',
            'ConfigurationInitializer',
            'LoggingAndErrorInitializer',
            'DatabaseInitializer',
            'CacheInitializer',
            'EventInitializer',
            'SessionInitializer',
            'SecurityInitializer',
            'AssetInitializer',
            'TemplateInitializer',
            'ModuleInitializer',
            'MiddlewareInitializer',
            'CoreModuleInitializer',
            'RouteInitializer',
            'ValidationInitializer',
            'DependencyInitializer',
            'ApplicationInitializer'
        ]
    ];
    
    /**
     * Current request
     * 
     * @var Request|null
     */
    private ?Request $currentRequest = null;
    
    /**
     * Current request type (web, api, console, minimal)
     * 
     * @var string
     */
    private string $requestType = 'web';
    
    /**
     * Gets singleton instance of the application
     * 
     * @param Container|null $externalContainer Container to use instead of a new one
     * @return self Application instance
     */
    public static function getInstance(Container $externalContainer = null): self
    {
        if (self::$instance === null) {
            self::$instance = new self($externalContainer);
        } elseif ($externalContainer !== null && self::$instance->container !== $externalContainer) {
            // If a new container is passed, update existing instance
            self::$instance->container = $externalContainer;
        }
        
        return self::$instance;
    }
    
    /**
     * Constructor - private due to singleton pattern
     * 
     * @param Container|null $externalContainer Container to use
     */
    private function __construct(?Container $externalContainer = null)
    {
        // Use passed container or create a new one
        $this->container = $externalContainer ?? new Container();
    }
    
    /**
     * Executes bootstrap process for the entire application
     * 
     * @param string $mode Bootstrap mode ('full', 'minimal', 'api', 'console')
     * @return void
     */
    public function bootstrap(string $mode = 'full'): void
    {
        if ($this->bootstrapped) {
            return;
        }
        
        try {
            // Save request type
            $this->requestType = $mode;
            
            // Register initializers for selected mode
            $this->registerInitializers($mode);
            
            // Initialize all initializers
            $this->runInitialization();
            
            // Finalize all initializers
            $this->runFinalization();
            
            $this->bootstrapped = true;
        } catch (\Throwable $e) {
            $this->handleBootstrapError($e);
            
            // Re-throw exception
            throw $e;
        }
    }
    
    /**
     * Registers initializers for the specified mode
     * 
     * @param string $mode Bootstrap mode
     * @return void 
     */
    private function registerInitializers(string $mode): void
    {
        // Get initializer list for the specified mode
        $initializers = $this->initializersByMode[$mode] ?? $this->initializersByMode['full'];
        
        // Register each initializer
        foreach ($initializers as $initializerClass) {
            $this->registerInitializer($initializerClass);
        }
    }
    
    /**
     * Registers a single initializer
     * 
     * @param string $initializerClass Initializer class name
     * @return void
     */
    private function registerInitializer(string $initializerClass): void
    {
        // Add namespace if not explicitly given
        if (strpos($initializerClass, '\\') === false) {
            $initializerClass = "\\baseKRIZAN\\Bootstrap\\Initializers\\{$initializerClass}";
        }
        
        // Skip if class doesn't exist
        if (!class_exists($initializerClass)) {
            $this->logBootstrapMessage("Initializer class does not exist: {$initializerClass}");
            return;
        }
        
        // Create initializer instance
        $initializer = new $initializerClass($this->container);
        
        // Store instance for later initialization
        $this->initializers[$initializerClass] = $initializer;
        
        // Log registration if possible
        $this->logBootstrapMessage("Registered initializer: {$initializerClass}");
    }
    
    /**
     * Runs initialize method for all registered initializers
     * 
     * @return void
     */
    private function runInitialization(): void
    {
        foreach ($this->initializers as $initializer) {
            $initializer->initialize();
        }
    }
    
    /**
     * Runs finalize method for all registered initializers
     * 
     * @return void
     */
    private function runFinalization(): void
    {
        foreach ($this->initializers as $initializer) {
            if (method_exists($initializer, 'finalize')) {
                $initializer->finalize();
            }
        }
    }
    
    /**
     * Logs bootstrap message if logger is available
     * 
     * @param string $message Message to log
     * @param array $context Additional context
     * @return void
     */
    private function logBootstrapMessage(string $message, array $context = []): void
    {
        if ($this->container->has('logger')) {
            $logger = $this->container->get('logger');
            $logger->bootstrap($message, $context);
        }
    }
    
    /**
     * Handles errors during bootstrap process
     * 
     * @param \Throwable $e Error that occurred
     * @return void
     */
    private function handleBootstrapError(\Throwable $e): void
    {
        // Ensure basic logging is available
        $storageLogPath = APP_ROOT . '/storage/logs';
    
        if (!is_dir($storageLogPath)) {
            mkdir($storageLogPath, 0777, true);
        }
        
        // Log the bootstrap error
        $errorMessage = '[' . date('Y-m-d H:i:s') . '] [BOOTSTRAP ERROR] ' . $e->getMessage() . 
                    ' in ' . $e->getFile() . 
                    ' on line ' . $e->getLine() . 
                    PHP_EOL . 'Trace: ' . $e->getTraceAsString();
        
        error_log($errorMessage, 3, $storageLogPath . '/bootstrap_errors.log');
        
        // If we have logger, use it
        if ($this->container->has('logger')) {
            $logger = $this->container->get('logger');
            $logger->error('Bootstrap error', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
        }
        
        // Let UnifiedErrorHandler handle the rest
        UnifiedErrorHandler::getInstance()->handleException(
            $e, 
            UnifiedErrorHandler::getInstance()->determineResponseFormat()
        );
    }
    
    /**
     * Sets current request
     * 
     * @param Request $request Current request
     * @return void
     */
    public function setCurrentRequest(Request $request): void
    {
        $this->currentRequest = $request;
        $this->container->register('request', $request, 'request');
    }
   
    /**
     * Processes request with given path and method
     * 
     * @param string $path Request path
     * @param string $method HTTP method
     * @return void
     */
    public function handleRequest(string $path, string $method): void
    {
        // Log request processing start
        $this->logBootstrapMessage('Request handling started', [
            'path' => $path,
            'method' => $method,
            'request_type' => $this->requestType
        ]);
        
        // Run EntryPoint
        if ($this->container->has('entryPoint')) {
            $this->container->get('entryPoint')->run();
        } else {
            // Create and run EntryPoint if not registered
            $entryPoint = new \baseKRIZAN\Http\EntryPoint(
                $path,
                $method,
                $this->container->get('routes'),
                $this->container
            );
            $entryPoint->run();
        }
        
        // Log request processing completion
        $this->logBootstrapMessage('Request handling completed', [
            'path' => $path,
            'method' => $method
        ]);
    }

    /**
     * Method called at end of request to clean up resources
     * 
     * @return void
     */
    public function shutdown(): void
    {
        $this->logBootstrapMessage('Bootstrap shutdown initiated');
        
        // Clean request scope services
        $this->cleanupScope('request');
        
        // Call shutdown handlers for registered services
        $this->callShutdownHandlers();
        
        $this->logBootstrapMessage('Bootstrap shutdown completed');
    }
 
    /**
     * Cleans instances for given scope after request completion
     * 
     * @param string $scope Scope to clean ('request' or 'all')
     * @return void
     */
    public function cleanupScope(string $scope = 'request'): void
    {
        if ($scope === 'all') {
            // Clean all instances except logger
            $logger = null;
            if ($this->container->has('logger')) {
                $logger = $this->container->get('logger');
            }
            
            $this->container->clearScope('singleton');
            $this->container->clearScope('request');
            
            // Restore logger
            if ($logger) {
                $this->container->register('logger', $logger, 'singleton');
            }
        } else {
            // Clean only request scope
            $this->container->clearScope($scope);
        }
    }
    
    /**
     * Calls shutdown handlers for registered services
     * 
     * @return void
     */
    private function callShutdownHandlers(): void
    {
        // For each service with a shutdown method, call it
        foreach ($this->container->getServices() as $id => $service) {
            if (is_object($service) && method_exists($service, 'shutdown')) {
                try {
                    $service->shutdown();
                } catch (\Throwable $e) {
                    // Log error but continue with other services
                    $this->logBootstrapMessage("Error shutting down service: $id", [
                        'error' => $e->getMessage()
                    ]);
                }
            }
        }
    }
   
    /**
     * Gets container
     * 
     * @return Container
     */
    public function getContainer(): Container
    {
        return $this->container;
    }
}