<?php
// app/baseKRIZAN/Bootstrap/Initializers/CoreModuleInitializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

/**
 * Initializer for core modules initialization
 */
class CoreModuleInitializer extends Initializer
{
    /**
     * Initializes core modules
     * 
     * @return void
     */
    public function initialize(): void
    {
        $logger = $this->getLogger();
        
        // Ensure EventDispatcher is available
        $this->ensureService('eventDispatcher', 'EventInitializer');
        
        // Initialize LUKA module
        $this->registerLukaModule($logger);
        
        // Initialize BORNA module
        $this->registerBornaModule($logger);
        
        $logger->bootstrap('Core modules registered');
    }
    
    /**
     * Registers LUKA module if available
     * 
     * @param mixed $logger Logger for logging
     * @return void
     */
    private function registerLukaModule($logger): void
    {
        if (class_exists('\\baseKRIZAN\\LUKA\\LukaManager')) {
            try {
                $lukaEnabled = \baseKRIZAN\Config\Config::get('luka.enabled', false);
                
                if ($lukaEnabled) {
                    $this->container->register('lukaManager', function($container) use ($logger) {
                        return \baseKRIZAN\LUKA\LukaManager::getInstance($logger, $container);
                    });
                    
                    $logger->bootstrap("LUKA module registered and will be initialized through LukaManager");
                } else {
                    $logger->bootstrap("LUKA module disabled in config, skipping initialization");
                }
            } catch (\Throwable $e) {
                $logger->error("Failed to register LUKA module", [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }
    }
    
    /**
     * Registers BORNA module if available
     * 
     * @param mixed $logger Logger for logging
     * @return void
     */
    private function registerBornaModule($logger): void
    {
        if (class_exists('\\baseKRIZAN\\BORNA\\BornaManager')) {
            try {
                $bornaEnabled = \baseKRIZAN\Config\Config::get('borna.enabled', false);
                
                if ($bornaEnabled) {
                    $this->container->register('bornaManager', function($container) use ($logger) {
                        return \baseKRIZAN\BORNA\BornaManager::getInstance($logger, $container);
                    });
                    
                    $logger->bootstrap("BORNA module registered and will be initialized through BornaManager");
                } else {
                    $logger->bootstrap("BORNA module disabled in config, skipping initialization");
                }
            } catch (\Throwable $e) {
                $logger->error("Failed to register BORNA module", [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }
    }
    
    /**
     * Initializes core modules after all services are available
     * 
     * @return void
     */
    public function finalize(): void
    {
        $logger = $this->getLogger();
        
        // Initialize LUKA module
        if ($this->container->has('lukaManager')) {
            try {
                $lukaManager = $this->container->get('lukaManager');
                $enabled = $lukaManager->isEnabled();
                
                if ($enabled && $this->container->has('middlewareRegistry')) {
                    // Let Manager class handle middleware registration
                    $registry = $this->container->get('middlewareRegistry');
                    $lukaManager->registerMiddleware($registry);
                }
                
                $logger->bootstrap("LUKA module initialization " . 
                    ($enabled ? "completed" : "disabled"));
            } catch (\Throwable $e) {
                $logger->error("Failed to initialize LUKA module", [
                    'error' => $e->getMessage()
                ]);
            }
        }
        
        // Initialize BORNA module
        if ($this->container->has('bornaManager')) {
            try {
                $bornaManager = $this->container->get('bornaManager');
                $enabled = $bornaManager->isEnabled();
                
                if ($enabled && $this->container->has('middlewareRegistry')) {
                    // Let Manager class handle middleware registration
                    $registry = $this->container->get('middlewareRegistry');
                    $bornaManager->registerMiddleware($registry);
                }
                
                $logger->bootstrap("BORNA module initialization " . 
                    ($enabled ? "completed" : "disabled"));
            } catch (\Throwable $e) {
                $logger->error("Failed to initialize BORNA module", [
                    'error' => $e->getMessage()
                ]);
            }
        }
        
        $logger->bootstrap('Core modules initialization completed');
    }
}