<?php
// app/baseKRIZAN/Bootstrap/Initializers/Initializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

use baseKRIZAN\Services\Container;

/**
 * Abstract class that all initializers will extend
 */
abstract class Initializer
{
    /**
     * Container for dependency injection
     * 
     * @var Container
     */
    protected Container $container;
    
    /**
     * Logger instance cache
     * 
     * @var mixed
     */
    protected $loggerInstance = null;
    
    /**
     * Constructor
     * 
     * @param Container $container Container for dependency injection
     */
    public function __construct(Container $container)
    {
        $this->container = $container;
        // Do NOT initialize logger here - it can cause circular dependencies
    }
    
    /**
     * Initializes services in the container
     * Each initializer must implement this method
     * 
     * @return void
     */
    abstract public function initialize(): void;
    
    /**
     * Finalizes services after initialization of all initializers
     * Optional method that initializer can implement
     * 
     * @return void
     */
    public function finalize(): void
    {
        // Optional method - default implementation is empty
    }
    
    /**
     * Gets the logger if available
     * 
     * @return mixed Logger instance or null
     */
    protected function getLogger()
    {
        // Return cached logger if we have it
        if ($this->loggerInstance !== null) {
            return $this->loggerInstance;
        }
        
        // Check if logger exists in container
        if (!$this->container->has('logger')) {
            // Only initialize logger if we're not already in LoggingAndErrorInitializer
            $currentClass = get_class($this);
            if ($currentClass !== 'baseKRIZAN\\Bootstrap\\Initializers\\LoggingAndErrorInitializer' &&
                class_exists('\\baseKRIZAN\\Bootstrap\\Initializers\\LoggingAndErrorInitializer')) {
                (new \baseKRIZAN\Bootstrap\Initializers\LoggingAndErrorInitializer($this->container))->initialize();
            }
        }
        
        // Cache logger to avoid circular references
        if ($this->container->has('logger')) {
            $this->loggerInstance = $this->container->get('logger');
            return $this->loggerInstance;
        }
        
        return null;
    }
    
    /**
     * Logs bootstrap message
     * 
     * @param string $message Message to log
     * @param array $context Context data
     * @return void
     */
    protected function log(string $message, array $context = []): void
    {
        $logger = $this->getLogger();
        if ($logger) {
            $logger->bootstrap($message, $context);
        }
    }
    
    /**
     * Ensures a service is available in the container
     * 
     * @param string $service Service name
     * @param string $initializer Initializer class name
     * @return bool True if service is now available
     */
    protected function ensureService(string $service, string $initializer): bool
    {
        if ($this->container->has($service)) {
            return true;
        }
        
        if (strpos($initializer, '\\') === false) {
            $initializer = "\\baseKRIZAN\\Bootstrap\\Initializers\\{$initializer}";
        }
        
        // Prevent circular initialization
        $currentClass = get_class($this);
        if ($currentClass === $initializer) {
            return false;
        }
        
        if (class_exists($initializer)) {
            (new $initializer($this->container))->initialize();
            return $this->container->has($service);
        }
        
        return false;
    }
    
    /**
     * Ensures multiple services are available in the container
     * 
     * @param array $services Array of service names or [service => initializer] pairs
     * @return bool True if all services are now available
     */
    protected function ensureMultipleServices(array $services): bool
    {
        $success = true;
        
        foreach ($services as $key => $value) {
            if (is_numeric($key)) {
                // Simple service name
                $service = $value;
                $initializer = $this->guessInitializerFromService($service);
            } else {
                // [service => initializer] pair
                $service = $key;
                $initializer = $value;
            }
            
            $serviceSuccess = $this->ensureService($service, $initializer);
            $success = $success && $serviceSuccess;
        }
        
        return $success;
    }
    
    /**
     * Guesses the initializer class name based on service name
     * 
     * @param string $service Service name
     * @return string Initializer class name
     */
    private function guessInitializerFromService(string $service): string
    {
        $mappings = [
            'dbConnection' => 'DatabaseInitializer',
            'pdo' => 'DatabaseInitializer',
            'sessionManager' => 'SessionInitializer',
            'eventDispatcher' => 'EventInitializer',
            'cache' => 'CacheInitializer',
            'authentication' => 'SecurityInitializer',
            'apiAuth' => 'SecurityInitializer',
            'csrfProtection' => 'SecurityInitializer',
            'templateRenderer' => 'TemplateInitializer',
            'logger' => 'LoggingAndErrorInitializer',
            'errorHandler' => 'LoggingAndErrorInitializer',
            'middlewareRegistry' => 'MiddlewareInitializer',
            'middlewareHandler' => 'MiddlewareInitializer',
        ];
        
        return $mappings[$service] ?? ucfirst($service) . 'Initializer';
    }
}