<?php
// app/baseKRIZAN/Bootstrap/Initializers/LoggingAndErrorInitializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Error\UnifiedErrorHandler;

/**
 * Unified initializer for logging and error handling
 */
class LoggingAndErrorInitializer extends Initializer
{
    /**
     * Initializes logging and error handling components
     * 
     * @return void
     */
    public function initialize(): void
    {
        // Initialize logger first
        $this->initializeLogging();
        
        // Initialize error handling after logging
        $this->initializeErrorHandling();
    }
    
    /**
     * Initializes logging components
     * 
     * @return void
     */
    private function initializeLogging(): void
    {
        $this->container->register('logger', function($container) {
            // Load required configurations
            $logPath = \baseKRIZAN\Config\Config::get('paths.logs', APP_ROOT . '/storage/logs');
            $developmentMode = \baseKRIZAN\Config\Config::get('environment') === 'development';
            $rotationDays = (int)(\baseKRIZAN\Config\Config::get('logging.rotation_days') ?? 30);
            
            // Prepare category configuration for Logger
            $categoryConfig = $this->prepareLogCategories();
            
            // Create Logger instance with configuration
            return new Logger(
                $logPath, 
                $developmentMode,
                $categoryConfig,
                $rotationDays
            );
        });
        
        $logger = $this->getLogger();
        $logger->bootstrap('Logging system initialized');
    }
    
    /**
     * Initializes error handling components
     * 
     * @return void
     */
    private function initializeErrorHandling(): void
    {
        $logger = $this->getLogger();
        
        // Register global error handlers
        UnifiedErrorHandler::registerGlobalHandlers();
        
        // Get global error handler instance
        $handler = UnifiedErrorHandler::getInstance();
        
        // Register in container
        $this->container->register('errorHandler', $handler);
        
        $logger->bootstrap('Error handling system registered');
    }
    
    /**
     * Prepares logging category configuration
     * 
     * @return array
     */
    private function prepareLogCategories(): array
    {
        $categoryConfig = [];
        
        try {
            // Load logging configuration
            $logConfig = \baseKRIZAN\Config\Config::get('logging', []);
            
            foreach ($logConfig as $category => $settings) {
                if (is_array($settings)) {
                    $categoryConfig[$category] = [
                        'level' => $settings['level'] ?? 'INFO',
                        'enabled' => $settings['enabled'] ?? true
                    ];
                }
            }
        } catch (\Throwable $e) {
            // Return empty array in case of error
        }
        
        return $categoryConfig;
    }
    
    /**
     * Finalizes error handler after all services are registered
     * 
     * @return void
     */
    public function finalize(): void
    {
        // Get error handler
        $handler = $this->container->get('errorHandler');
        
        // Set templateRenderer on error handler if it exists
        if ($this->container->has('templateRenderer')) {
            $handler->setTemplateRenderer($this->container->get('templateRenderer'));
        }
        
        // Log initialization
        $logger = $this->getLogger();
        $logger->bootstrap('Error handling system finalized', [
            'development_mode' => \baseKRIZAN\Config\Config::get('environment') === 'development'
        ]);
    }
}