<?php
// app/baseKRIZAN/Bootstrap/Initializers/MiddlewareInitializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

use baseKRIZAN\Http\Middleware\MiddlewareRegistry;
use baseKRIZAN\Http\Middleware\MiddlewareHandler;

/**
 * Initializer for middleware system
 */
class MiddlewareInitializer extends Initializer
{
    /**
     * Middleware configuration by group
     * 
     * @var array<string, array>
     */
    private array $middlewareConfig = [
        'global' => [
        ],
        'web' => [
            'SessionMiddleware' => [
                'priority' => 10,
                'requiredServices' => ['sessionManager', 'logger'],
                'constructor_args' => [2 => false] // Explicitly set lazyMode to false for web
            ],
            'SecurityHeadersMiddleware' => [
                'priority' => 20,
                'requiredServices' => ['logger']
            ],
            'SanitizationMiddleware' => [
                'priority' => 30,
                'requiredServices' => ['container', 'logger']
            ],
            'AuthMiddleware' => [
                'priority' => 40, 
                'condition' => 'authentication.enabled',
                'requiredServices' => ['authentication', 'routes', 'logger']
            ],
            'CsrfMiddleware' => [
                'priority' => 50,
                'requiredServices' => ['sessionManager', 'logger'],
                'constructor_args' => [2 => ''] // Explicitly set failedRedirectUrl to empty string
            ],
            'RateLimiterMiddleware' => [
                'priority' => 70,
                'requiredServices' => ['templateRenderer', 'logger']
            ]
        ],
        'api' => [
            'SessionMiddleware' => [
                'priority' => 10, 
                'requiredServices' => ['sessionManager', 'logger'],
                'constructor_args' => [2 => true] // Set lazyMode to true for API
            ],
            'SecurityHeadersMiddleware' => [
                'priority' => 20,
                'requiredServices' => ['logger']
            ],
            'SanitizationMiddleware' => [
                'priority' => 30,
                'requiredServices' => ['container', 'logger']
            ],
            'ApiAuthMiddleware' => [
                'priority' => 40,
                'condition' => 'api_auth.enabled',
                'requiredServices' => ['apiAuth', 'logger']
            ],
            'RateLimiterMiddleware' => [
                'priority' => 70,
                'requiredServices' => ['templateRenderer', 'logger']
            ]
        ],
        'assets' => [
            'SecurityHeadersMiddleware' => [
                'priority' => 10,
                'requiredServices' => ['logger']
            ],
            'CacheMiddleware' => [
                'priority' => 20,
                'requiredServices' => ['logger']
            ]
        ]
    ];
    
    /**
     * Initializes middleware components
     * 
     * @return void
     */
    public function initialize(): void
    {
        $logger = $this->getLogger();
        
        // Register MiddlewareRegistry
        $this->container->register('middlewareRegistry', function($container) use ($logger) {
            return new MiddlewareRegistry($container, $logger);
        });
        
        // Register MiddlewareHandler
        $this->container->register('middlewareHandler', function($container) use ($logger) {
            return new MiddlewareHandler($logger, $container);
        });
        
        $logger->bootstrap('Middleware system initialized');
    }
    
    /**
     * Registers middleware components after all services are available
     * 
     * @return void
     */
    public function finalize(): void
    {
        $registry = $this->container->get('middlewareRegistry');
        $logger = $this->getLogger();
        
        // Register middleware definitions in registry
        foreach ($this->middlewareConfig as $group => $middlewares) {
            $logger->bootstrap("Registering middleware definitions for group: {$group}", [
                'count' => count($middlewares)
            ]);
            
            foreach ($middlewares as $middlewareName => $config) {
                $middlewareClass = $middlewareName;
                if (!class_exists($middlewareName) && strpos($middlewareName, '\\') === false) {
                    $middlewareClass = "\\baseKRIZAN\\Http\\Middleware\\{$middlewareName}";
                }
                
                $registry->addMiddleware($middlewareClass, array_merge($config, ['group' => $group]));
            }
        }
        
        // Ensure SessionManager is registered before middleware registration
        $this->ensureService('sessionManager', 'SessionInitializer');
        
        // Load middleware configuration from config if it exists
        $configMiddleware = \baseKRIZAN\Config\Config::get('middleware', []);
        if (!empty($configMiddleware)) {
            $this->middlewareConfig = array_merge_recursive($this->middlewareConfig, $configMiddleware);
        }
        
        // Register global middleware
        $this->registerMiddlewareGroup($registry, 'global');
        
        // Determine request type from request object
        $requestType = $this->determineRequestType();
        
        // Register middleware for specific request type
        $this->registerMiddlewareGroup($registry, $requestType);
        
        $logger->bootstrap("Middleware registered for request type: {$requestType}");
    }
    
    /**
     * Determines request type from request object
     * 
     * @return string Request type
     */
    private function determineRequestType(): string
    {
        $requestType = 'web'; // Default
        
        if ($this->container->has('request')) {
            $request = $this->container->get('request');
            $requestType = $request->getAttribute('request_type', 'web');
            
            $this->log("Using request_type from request attribute: {$requestType}");
        }
        
        return $requestType;
    }
    
    /**
     * Registers a group of middleware components
     * 
     * @param MiddlewareRegistry $registry Registry for registration
     * @param string $group Middleware group
     * @return void
     */
    private function registerMiddlewareGroup(MiddlewareRegistry $registry, string $group): void
    {
        $logger = $this->getLogger();
        
        if (!isset($this->middlewareConfig[$group])) {
            $logger->bootstrap("No middleware configured for group: {$group}");
            return;
        }
        
        foreach ($this->middlewareConfig[$group] as $middlewareName => $options) {
            // Check condition if it exists
            if (isset($options['condition'])) {
                $conditionValue = \baseKRIZAN\Config\Config::get($options['condition'], false);
                if (!$conditionValue) {
                    $logger->bootstrap("Skipped middleware {$middlewareName} due to condition: {$options['condition']}");
                    continue;
                }
            }
            
            // Check service availability
            if (isset($options['requiredServices']) && is_array($options['requiredServices'])) {
                $allServicesAvailable = true;
                foreach ($options['requiredServices'] as $service) {
                    if ($service !== 'logger' && $service !== 'container' && !$this->container->has($service)) {
                        $allServicesAvailable = false;
                        $logger->bootstrap("Skipped middleware {$middlewareName} - missing service: {$service}");
                        break;
                    }
                }
                
                if (!$allServicesAvailable) {
                    continue;
                }
            }
            
            // Determine namespace and class
            $middlewareClass = $middlewareName;
            if (!class_exists($middlewareName) && strpos($middlewareName, '\\') === false) {
                $middlewareClass = "\\baseKRIZAN\\Http\\Middleware\\{$middlewareName}";
            }
            
            // Configure middleware for registry
            $middlewareConfig = [
                'group' => $group,
                'priority' => $options['priority'] ?? 100
            ];
            
            // Add required services
            if (isset($options['requiredServices'])) {
                $middlewareConfig['requiredServices'] = $options['requiredServices'];
            }
            
            // Add additional arguments if they exist
            if (isset($options['constructor_args'])) {
                $middlewareConfig['constructor_args'] = $options['constructor_args'];
            }
            
            // Register middleware
            $registry->addMiddleware($middlewareClass, $middlewareConfig);
            $logger->bootstrap("Registered middleware: {$middlewareName} for group: {$group}");
        }
    }
}