<?php
// app/baseKRIZAN/Bootstrap/Initializers/ModuleInitializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

use baseKRIZAN\Modules\ModuleManager;

/**
 * Initializer for module system
 */
class ModuleInitializer extends Initializer
{
    /**
     * Core modules to initialize separately
     * 
     * @var array<string>
     */
    private array $coreModules = ['LUKA', 'BORNA'];
    
    /**
     * Initializes ModuleManager and its dependencies
     * 
     * @return void
     */
    public function initialize(): void
    {
        $logger = $this->getLogger();
        
        // Ensure EventDispatcher is available
        $this->ensureService('eventDispatcher', 'EventInitializer');
        
        // Ensure DatabaseConnection is available
        $this->ensureService('dbConnection', 'DatabaseInitializer');
        
        // Get event dispatcher
        $eventDispatcher = $this->container->get('eventDispatcher');
        
        // Register ModuleManager
        $this->container->register('moduleManager', function($container) use ($eventDispatcher, $logger) {
            return ModuleManager::getInstance(
                $logger, 
                $container, 
                $eventDispatcher
            );
        });
        
        // Register core modules if enabled
        foreach ($this->coreModules as $module) {
            $this->registerCoreModule($module);
        }
        
        $logger->bootstrap('Module system registered');
    }
    
    /**
     * Registers a core module if available
     * 
     * @param string $moduleName Name of the core module
     * @return void
     */
    private function registerCoreModule(string $moduleName): void
    {
        $logger = $this->getLogger();
        $managerClass = "\\baseKRIZAN\\{$moduleName}\\{$moduleName}Manager";
        $configKey = strtolower($moduleName) . '.enabled';
        
        if (class_exists($managerClass)) {
            try {
                $enabled = \baseKRIZAN\Config\Config::get($configKey, false);
                
                if ($enabled) {
                    $this->container->register(lcfirst($moduleName) . 'Manager', function($container) use ($managerClass, $logger) {
                        return $managerClass::getInstance(
                            $logger, 
                            $container
                        );
                    });
                    
                    $logger->bootstrap("{$moduleName} module registered and will be initialized through {$moduleName}Manager");
                } else {
                    $logger->bootstrap("{$moduleName} module disabled in config, skipping initialization");
                }
            } catch (\Throwable $e) {
                $logger->error("Failed to register {$moduleName} module", [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }
    }
    
    /**
     * Loads modules after all services are registered
     * 
     * @return void
     */
    public function finalize(): void
    {
        $logger = $this->getLogger();
        $moduleManager = $this->container->get('moduleManager');
        
        // Load modules
        $moduleManager->loadModules();
        
        // Initialize core modules
        foreach ($this->coreModules as $module) {
            $this->finalizeCoreModule($module);
        }
        
        $logger->bootstrap('Module system initialized with ' . 
            count($moduleManager->getEnabledModules()) . ' enabled modules');
    }
    
    /**
     * Finalizes a core module initialization
     * 
     * @param string $moduleName Name of the core module
     * @return void
     */
    private function finalizeCoreModule(string $moduleName): void
    {
        $logger = $this->getLogger();
        $managerServiceName = lcfirst($moduleName) . 'Manager';
        
        if ($this->container->has($managerServiceName)) {
            try {
                $manager = $this->container->get($managerServiceName);
                $enabled = $manager->isEnabled();
                
                if ($enabled && $this->container->has('middlewareRegistry')) {
                    // Let Manager class handle middleware registration
                    $registry = $this->container->get('middlewareRegistry');
                    $manager->registerMiddleware($registry);
                }
                
                $logger->bootstrap("{$moduleName} module initialization " . 
                    ($enabled ? "completed" : "disabled"));
            } catch (\Throwable $e) {
                $logger->error("Failed to initialize {$moduleName} module", [
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
}