<?php
// app/baseKRIZAN/Bootstrap/Initializers/SecurityInitializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

/**
 * Centralized initializer for all security components
 */
class SecurityInitializer extends Initializer
{
    /**
     * Initializes all security components
     * 
     * @return void
     */
    public function initialize(): void
    {
        $logger = $this->getLogger();
        
        // Ensure SessionManager is available for security components
        $this->ensureService('sessionManager', 'SessionInitializer');
        
        // Initialize CSRF protection
        $this->initializeCsrfProtection($logger);
        
        // Initialize API Authentication
        $this->initializeApiAuth($logger);
        
        // Initialize Authentication (if available)
        $this->initializeAuthentication($logger);
        
        // Register security event listeners if events are available
        if ($this->container->has('eventDispatcher')) {
            $this->registerSecurityEventListeners($logger);
        }
        
        $logger->bootstrap('Security system initialized');
    }
    
    /**
     * Initializes CSRF protection
     * 
     * @param mixed $logger Logger for logging
     * @return void
     */
    private function initializeCsrfProtection($logger): void
    {
        if (class_exists('\\baseKRIZAN\\Security\\CsrfProtection')) {
            $this->container->register('csrfProtection', function($container) use ($logger) {
                $sessionManager = $container->get('sessionManager');
                return new \baseKRIZAN\Security\CsrfProtection($logger, $sessionManager);
            });
            
            $logger->bootstrap('CSRF protection initialized');
        }
    }
    
    /**
     * Initializes API Authentication
     * 
     * @param mixed $logger Logger for logging
     * @return void
     */
    private function initializeApiAuth($logger): void
    {
        if (!$this->container->has('apiAuth') && class_exists('\\baseKRIZAN\\Security\\ApiAuth')) {
            $this->container->register('apiAuth', function($container) use ($logger) {
                // Ensure DatabaseConnection is available
                if (!$container->has('dbConnection')) {
                    (new DatabaseInitializer($container))->initialize();
                }
                
                $dbConnection = $container->get('dbConnection');
                
                // Get cache driver instead of manager
                $cache = null;
                if ($container->has('cache')) {
                    // Get default driver from CacheManager
                    $cache = $container->get('cache')->driver();
                }
                
                $sessionManager = $container->has('sessionManager') 
                    ? $container->get('sessionManager') 
                    : null;
                
                return new \baseKRIZAN\Security\ApiAuth(
                    $dbConnection, 
                    $logger, 
                    $cache, 
                    $sessionManager
                );
            });
            
            $logger->bootstrap('API Authentication initialized');
        }
    }
    
    /**
     * Initializes Authentication system
     * 
     * @param mixed $logger Logger for logging
     * @return void
     */
    private function initializeAuthentication($logger): void
    {
        // Check if Authentication class exists and not already registered
        if (!$this->container->has('authentication') && class_exists('\\baseKRIZAN\\Security\\Authentication')) {
            $this->container->register('authentication', function($container) use ($logger) {
                $sessionManager = $container->get('sessionManager');
                $dbConnection = $container->get('dbConnection');
                
                // Get event dispatcher if available
                $eventDispatcher = $container->has('eventDispatcher') 
                    ? $container->get('eventDispatcher') 
                    : null;
                
                return new \baseKRIZAN\Security\Authentication(
                    $dbConnection,
                    $sessionManager,
                    $logger,
                    $eventDispatcher
                );
            });
            
            $logger->bootstrap('Authentication system initialized');
        }
    }
    
    /**
     * Registers security-related event listeners
     * 
     * @param mixed $logger Logger for logging
     * @return void
     */
    private function registerSecurityEventListeners($logger): void
    {
        $eventDispatcher = $this->container->get('eventDispatcher');
        
        // Register security-related events (if applicable)
        if ($this->container->has('authentication')) {
            $eventDispatcher->addListener('user.login.success', function($event) {
                // Log successful login
                $this->getLogger()->security('Login success', [
                    'user_id' => $event['user_id'] ?? null,
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
                ]);
            });
            
            $eventDispatcher->addListener('user.login.failure', function($event) {
                // Log failed login
                $this->getLogger()->security('Login failure', [
                    'username' => $event['username'] ?? 'unknown',
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                    'reason' => $event['reason'] ?? 'unknown'
                ]);
            });
        }
        
        $logger->bootstrap('Security event listeners registered');
    }
}