<?php
// app/baseKRIZAN/Bootstrap/Initializers/TemplateInitializer.php

namespace baseKRIZAN\Bootstrap\Initializers;

use baseKRIZAN\Template\TemplateLoader;
use baseKRIZAN\Template\DirectiveProcessor;
use baseKRIZAN\Template\TemplateCache;
use baseKRIZAN\Template\TemplateRenderer;
use baseKRIZAN\Template\DefaultDirectives;

/**
 * Initializer for template system
 */
class TemplateInitializer extends Initializer
{
    /**
     * Initializes template system components
     * 
     * @return void
     */
    public function initialize(): void
    {
        $logger = $this->getLogger();
        
        // Ensure SessionManager is available
        $this->ensureService('sessionManager', 'SessionInitializer');
        $sessionManager = $this->container->get('sessionManager');
        
        // Prepare path configuration for template loader
        $templateLoaderConfig = [
            \baseKRIZAN\Config\Config::get('paths.resourcesviews') . '/',
        ];
        
        // Add module paths if they exist
        if (is_dir(\baseKRIZAN\Config\Config::get('paths.moduli'))) {
            $templateLoaderConfig[] = \baseKRIZAN\Config\Config::get('paths.moduli') . '/';
        }
        
        // Register template components
        $this->registerTemplateComponents($templateLoaderConfig, $logger, $sessionManager);
        
        $logger->bootstrap('Template system registered');
    }
    
    /**
     * Registers template system components in the container
     * 
     * @param array $templateLoaderConfig Path configuration
     * @param mixed $logger Logger instance
     * @param mixed $sessionManager SessionManager instance
     * @return void
     */
    private function registerTemplateComponents(array $templateLoaderConfig, $logger, $sessionManager): void
    {
        // Register template loader
        $this->container->register('templateLoader', function($container) use ($templateLoaderConfig, $logger) {
            return new TemplateLoader(
                $templateLoaderConfig,
                $logger
            );
        });
        
        // Register directive processor
        $this->container->register('directiveProcessor', function($container) use ($logger) {
            return new DirectiveProcessor($logger);
        });
        
        // Register template cache
        $this->container->register('templateCache', function($container) use ($logger) {
            $developmentMode = \baseKRIZAN\Config\Config::get('environment') === 'development';
            return new TemplateCache(
                \baseKRIZAN\Config\Config::get('paths.cacheviews'),
                !$developmentMode,
                $logger
            );
        });
        
        // Register template renderer
        $this->container->register('templateRenderer', function($container) use ($logger, $sessionManager) {
            return new TemplateRenderer(
                $logger,
                $sessionManager,
                $container->get('templateLoader'),
                $container->get('directiveProcessor'),
                $container->get('templateCache'),
                new \baseKRIZAN\Config\Config()
            );
        });
    }
    
    /**
     * Initializes template system after all components are available
     * 
     * @return void
     */
    public function finalize(): void
    {
        $logger = $this->getLogger();
        
        // Register default directives
        $processor = $this->container->get('directiveProcessor');
        $config = new \baseKRIZAN\Config\Config();
        $defaultDirectives = new DefaultDirectives($config, $logger);
        $defaultDirectives->registerAll($processor);
        
        // Set TemplateRenderer in ErrorHandler if it exists
        if ($this->container->has('errorHandler')) {
            $errorHandler = $this->container->get('errorHandler');
            $templateRenderer = $this->container->get('templateRenderer');
            
            if (method_exists($errorHandler, 'setTemplateRenderer')) {
                $errorHandler->setTemplateRenderer($templateRenderer);
                $logger->bootstrap('TemplateRenderer set in ErrorHandler');
            }
        }
        
        $logger->bootstrap('Template system initialized');
    }
}