<?php

namespace baseKRIZAN\Cache;

/**
 * Universal Cache Interface
 * 
 * A standardized interface that combines PSR-16 (SimpleCache) compatibility
 * with additional methods for specialized caching needs.
 */
interface CacheInterface
{
    /**
     * Get an item from the cache
     *
     * @param string $key Cache key
     * @param mixed $default Default value if key not found
     * @return mixed The cached value or default
     */
    public function get(string $key, $default = null);
    
    /**
     * Store an item in the cache
     *
     * @param string $key Cache key
     * @param mixed $value Value to store
     * @param int|null|\DateInterval $ttl Time to live in seconds, null for default TTL
     * @return bool Success status
     */
    public function set(string $key, $value, $ttl = null): bool;
    
    /**
     * Remove an item from the cache
     *
     * @param string $key Cache key
     * @return bool Success status
     */
    public function delete(string $key): bool;
    
    /**
     * Check if an item exists in the cache
     *
     * @param string $key Cache key
     * @return bool True if exists
     */
    public function has(string $key): bool;
    
    /**
     * Clear all items from the cache
     *
     * @return bool Success status
     */
    public function clear(): bool;
    
    /**
     * Get multiple cache items
     * 
     * @param iterable $keys Cache keys
     * @param mixed $default Default value
     * @return iterable Items
     */
    public function getMultiple(iterable $keys, $default = null): iterable;
    
    /**
     * Set multiple cache items
     * 
     * @param iterable $values Key-value pairs
     * @param null|int|\DateInterval $ttl TTL
     * @return bool Success
     */
    public function setMultiple(iterable $values, $ttl = null): bool;
    
    /**
     * Delete multiple cache items
     * 
     * @param iterable $keys Keys to delete
     * @return bool Success
     */
    public function deleteMultiple(iterable $keys): bool;
    
    /**
     * Run garbage collection to remove expired items
     *
     * @return int Number of items removed
     */
    public function gc(): int;
    
    /**
     * Get cache statistics
     *
     * @return array Array of statistics
     */
    public function getStats(): array;
}