<?php

namespace baseKRIZAN\Cache;

use baseKRIZAN\Error\Logger;

/**
 * Enhanced Cache Manager for handling multiple cache implementations
 */
class CacheManager
{
    /**
     * Singleton instance
     */
    private static ?self $instance = null;
    
    /**
     * Available cache drivers
     */
    private array $drivers = [];
    
    /**
     * Default driver name
     */
    private string $defaultDriver = 'file';
    
    /**
     * Logger instance
     */
    private ?Logger $logger;
    
    /**
     * Constructor
     */
    private function __construct(?Logger $logger = null)
    {
        $this->logger = $logger;
        
        // Set default driver from config
        $defaultDriver = \baseKRIZAN\Config\Config::get('cache.default', 'file');
        if ($defaultDriver) {
            $this->defaultDriver = $defaultDriver;
        }
        
        // Register file cache driver by default
        $this->registerFileDriver();
        
        if ($logger) {
            $logger->services('Cache manager initialized', [
                'default_driver' => $this->defaultDriver
            ]);
        }
    }
    
    /**
     * Get singleton instance
     */
    public static function getInstance(?Logger $logger = null): self
    {
        if (self::$instance === null) {
            self::$instance = new self($logger);
        } elseif ($logger !== null && self::$instance->logger === null) {
            self::$instance->logger = $logger;
        }
        
        return self::$instance;
    }
    
    /**
     * Register file cache driver
     */
    private function registerFileDriver(): void
    {
        $cachePath = \baseKRIZAN\Config\Config::get('paths.cache', APP_ROOT . '/storage/cache');
        $defaultTtl = (int)\baseKRIZAN\Config\Config::get('cache.file.default_ttl', 3600);
        $memoryCacheLimit = (int)\baseKRIZAN\Config\Config::get('cache.file.memory_limit', 100);
        
        $this->drivers['file'] = new FileCache($cachePath, $this->logger, $defaultTtl, $memoryCacheLimit);
        
        if ($this->logger) {
            $this->logger->services('File cache driver registered', [
                'path' => $cachePath,
                'default_ttl' => $defaultTtl,
                'memory_limit' => $memoryCacheLimit
            ]);
        }
    }
    
    /**
     * Register a cache driver
     */
    public function registerDriver(string $name, CacheInterface $driver): self
    {
        $this->drivers[$name] = $driver;
        
        if ($this->logger) {
            $this->logger->services('Cache driver registered: ' . $name);
        }
        
        return $this;
    }
    
    /**
     * Get a cache driver instance
     */
    public function driver(string $name = null): CacheInterface
    {
        $driverName = $name ?: $this->defaultDriver;
        
        if (!isset($this->drivers[$driverName])) {
            throw new \RuntimeException("Cache driver not found: $driverName");
        }
        
        return $this->drivers[$driverName];
    }
    
    /**
     * Set the default cache driver
     */
    public function setDefaultDriver(string $name): self
    {
        if (!isset($this->drivers[$name])) {
            throw new \RuntimeException("Cannot set unknown cache driver as default: $name");
        }
        
        $this->defaultDriver = $name;
        
        if ($this->logger) {
            $this->logger->services('Default cache driver changed to: ' . $name);
        }
        
        return $this;
    }
    
    /**
     * Get an item from the default cache
     */
    public function get(string $key, $default = null)
    {
        return $this->driver()->get($key, $default);
    }
    
    /**
     * Store an item in the default cache
     */
    public function set(string $key, $value, $ttl = null): bool
    {
        return $this->driver()->set($key, $value, $ttl);
    }
    
    /**
     * Remove an item from the default cache
     */
    public function delete(string $key): bool
    {
        return $this->driver()->delete($key);
    }
    
    /**
     * Check if an item exists in the default cache
     */
    public function has(string $key): bool
    {
        return $this->driver()->has($key);
    }
    
    /**
     * Clear all items from the default cache
     */
    public function clear(): bool
    {
        return $this->driver()->clear();
    }
    
    /**
     * Get multiple items from the default cache
     */
    public function getMultiple(iterable $keys, $default = null): iterable
    {
        return $this->driver()->getMultiple($keys, $default);
    }
    
    /**
     * Store multiple items in the default cache
     */
    public function setMultiple(iterable $values, $ttl = null): bool
    {
        return $this->driver()->setMultiple($values, $ttl);
    }
    
    /**
     * Delete multiple items from the default cache
     */
    public function deleteMultiple(iterable $keys): bool
    {
        return $this->driver()->deleteMultiple($keys);
    }
    
    /**
     * Run garbage collection
     */
    public function gc(): int
    {
        $total = 0;
        
        foreach ($this->drivers as $name => $driver) {
            $removed = $driver->gc();
            $total += $removed;
            
            if ($this->logger && $removed > 0) {
                $this->logger->services("Cache GC: Removed $removed expired items from $name cache");
            }
        }
        
        return $total;
    }
    
    /**
     * Get cache statistics for all drivers
     */
    public function getAllStats(): array
    {
        $stats = [];
        
        foreach ($this->drivers as $name => $driver) {
            $stats[$name] = $driver->getStats();
        }
        
        return $stats;
    }
}