<?php

namespace baseKRIZAN\Cache;

use baseKRIZAN\Error\Logger;

/**
 * Enhanced file-based cache implementation
 */
class FileCache implements CacheInterface
{
    /**
     * Cache directory path
     */
    protected string $directory;
    
    /**
     * Logger instance
     */
    protected ?Logger $logger;
    
    /**
     * Default TTL in seconds (1 hour)
     */
    protected int $defaultTtl = 3600;
    
    /**
     * Memory cache for frequently accessed items
     */
    protected array $memoryCache = [];
    
    /**
     * Memory cache limit
     */
    protected int $memoryCacheLimit = 100;
    
    /**
     * Cache statistics
     */
    protected array $stats = [
        'hits' => 0,
        'misses' => 0,
        'writes' => 0
    ];
    
    /**
     * Constructor
     *
     * @param string $directory Cache directory path
     * @param Logger|null $logger Logger instance
     * @param int $defaultTtl Default TTL in seconds
     * @param int $memoryCacheLimit Memory cache size limit
     */
    public function __construct(
        string $directory, 
        ?Logger $logger = null, 
        int $defaultTtl = 3600,
        int $memoryCacheLimit = 100
    ) {
        $this->directory = rtrim($directory, '/');
        $this->logger = $logger;
        $this->defaultTtl = $defaultTtl;
        $this->memoryCacheLimit = $memoryCacheLimit;
        
        // Ensure cache directory exists
        if (!is_dir($this->directory)) {
            mkdir($this->directory, 0755, true);
        }
        
        if ($logger) {
            $logger->services('FileCache initialized', ['directory' => $this->directory]);
        }
    }
    
    /**
     * Set logger instance
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }
    
    /**
     * @inheritDoc
     */
    public function get(string $key, $default = null)
    {
        // Check memory cache first
        if (isset($this->memoryCache[$key])) {
            $data = $this->memoryCache[$key];
            
            // Check if expired in memory
            if (isset($data['expiry']) && $data['expiry'] < time()) {
                unset($this->memoryCache[$key]);
            } else {
                $this->stats['hits']++;
                return $data['value'];
            }
        }
        
        $path = $this->getFilePath($key);
        
        if (!file_exists($path)) {
            $this->stats['misses']++;
            return $default;
        }
        
        $content = file_get_contents($path);
        if ($content === false) {
            $this->stats['misses']++;
            return $default;
        }
        
        $data = unserialize($content);
        
        // Check if data is expired
        if (isset($data['expiry']) && $data['expiry'] < time()) {
            $this->delete($key);
            $this->stats['misses']++;
            return $default;
        }
        
        // Store in memory cache for future access
        $this->addToMemoryCache($key, $data);
        
        $this->stats['hits']++;
        return $data['value'] ?? $default;
    }
    
    /**
     * @inheritDoc
     */
    public function set(string $key, $value, $ttl = null): bool
    {
        $path = $this->getFilePath($key);
        $directory = dirname($path);
        
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }
        
        $ttlSeconds = $this->parseTtl($ttl);
        
        $data = [
            'value' => $value,
            'created_at' => time()
        ];
        
        // Set expiry if TTL provided
        if ($ttlSeconds > 0) {
            $data['expiry'] = time() + $ttlSeconds;
        }
        
        // Add to memory cache
        $this->addToMemoryCache($key, $data);
        
        $result = file_put_contents($path, serialize($data));
        
        if ($result !== false) {
            $this->stats['writes']++;
        }
        
        return $result !== false;
    }
    
    /**
     * @inheritDoc
     */
    public function delete(string $key): bool
    {
        // Remove from memory cache
        unset($this->memoryCache[$key]);
        
        $path = $this->getFilePath($key);
        
        if (file_exists($path)) {
            return unlink($path);
        }
        
        return true;
    }
    
    /**
     * @inheritDoc
     */
    public function has(string $key): bool
    {
        // Check memory cache first
        if (isset($this->memoryCache[$key])) {
            $data = $this->memoryCache[$key];
            
            // Check if expired in memory
            if (isset($data['expiry']) && $data['expiry'] < time()) {
                unset($this->memoryCache[$key]);
                return false;
            }
            
            return true;
        }
        
        $path = $this->getFilePath($key);
        
        if (!file_exists($path)) {
            return false;
        }
        
        $content = file_get_contents($path);
        if ($content === false) {
            return false;
        }
        
        $data = unserialize($content);
        
        // Check if data is expired
        if (isset($data['expiry']) && $data['expiry'] < time()) {
            $this->delete($key);
            return false;
        }
        
        return true;
    }
    
    /**
     * @inheritDoc
     */
    public function clear(): bool
    {
        // Clear memory cache
        $this->memoryCache = [];
        
        // Create a directory structure to avoid too many files in one directory
        $files = $this->scanCacheDir($this->directory);
        
        $success = true;
        
        foreach ($files as $file) {
            if (!unlink($file)) {
                $success = false;
            }
        }
        
        // Reset stats
        $this->resetStats();
        
        return $success;
    }
    
    /**
     * @inheritDoc
     */
    public function getMultiple(iterable $keys, $default = null): iterable
    {
        $result = [];
        
        foreach ($keys as $key) {
            $result[$key] = $this->get($key, $default);
        }
        
        return $result;
    }
    
    /**
     * @inheritDoc
     */
    public function setMultiple(iterable $values, $ttl = null): bool
    {
        $success = true;
        
        foreach ($values as $key => $value) {
            $success = $this->set($key, $value, $ttl) && $success;
        }
        
        return $success;
    }
    
    /**
     * @inheritDoc
     */
    public function deleteMultiple(iterable $keys): bool
    {
        $success = true;
        
        foreach ($keys as $key) {
            $success = $this->delete($key) && $success;
        }
        
        return $success;
    }
    
    /**
     * @inheritDoc
     */
    public function gc(): int
    {
        $files = $this->scanCacheDir($this->directory);
        $removed = 0;
        
        foreach ($files as $file) {
            $content = file_get_contents($file);
            
            if ($content !== false) {
                $data = unserialize($content);
                
                if (isset($data['expiry']) && $data['expiry'] < time()) {
                    if (unlink($file)) {
                        $removed++;
                    }
                }
            }
        }
        
        if ($this->logger && $removed > 0) {
            $this->logger->services('Cache garbage collection removed ' . $removed . ' files');
        }
        
        return $removed;
    }
    
    /**
     * @inheritDoc
     */
    public function getStats(): array
    {
        $stats = $this->stats;
        
        // Calculate hit ratio
        $total = $stats['hits'] + $stats['misses'];
        $stats['hit_ratio'] = $total > 0 ? round(($stats['hits'] / $total) * 100, 2) : 0;
        
        // Get memory cache size
        $stats['memory_items'] = count($this->memoryCache);
        
        // Count disk items
        $files = $this->scanCacheDir($this->directory);
        $stats['disk_items'] = count($files);
        
        return $stats;
    }
    
    /**
     * Reset cache statistics
     */
    public function resetStats(): void
    {
        $this->stats = [
            'hits' => 0,
            'misses' => 0,
            'writes' => 0
        ];
    }
    
    /**
     * Get cache file path for a key
     */
    protected function getFilePath(string $key): string
    {
        $hashedKey = md5($key);
        
        // Create a directory structure to avoid too many files in one directory
        $dir = substr($hashedKey, 0, 2);
        $subDir = substr($hashedKey, 2, 2);
        
        $path = $this->directory . '/' . $dir . '/' . $subDir;
        
        if (!is_dir($path)) {
            mkdir($path, 0755, true);
        }
        
        return $path . '/' . $hashedKey . '.cache';
    }
    
    /**
     * Add item to memory cache
     */
    protected function addToMemoryCache(string $key, array $data): void
    {
        // Maintain memory cache size limit
        if (count($this->memoryCache) >= $this->memoryCacheLimit) {
            // Remove the oldest item
            reset($this->memoryCache);
            unset($this->memoryCache[key($this->memoryCache)]);
        }
        
        $this->memoryCache[$key] = $data;
    }
    
    /**
     * Convert TTL to seconds
     */
    protected function parseTtl($ttl): int
    {
        if ($ttl === null) {
            return $this->defaultTtl;
        }
        
        if (is_int($ttl)) {
            return max(0, $ttl);
        }
        
        if ($ttl instanceof \DateInterval) {
            $reference = new \DateTimeImmutable();
            $endTime = $reference->add($ttl);
            return max(0, $endTime->getTimestamp() - $reference->getTimestamp());
        }
        
        return $this->defaultTtl;
    }
    
    /**
     * Scan cache directory recursively to find all cache files
     */
    protected function scanCacheDir(string $dir): array
    {
        if (!is_dir($dir)) {
            return [];
        }
        
        $files = [];
        $items = scandir($dir);
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }
            
            $path = $dir . '/' . $item;
            
            if (is_dir($path)) {
                $files = array_merge($files, $this->scanCacheDir($path));
            } elseif (substr($item, -6) === '.cache') {
                $files[] = $path;
            }
        }
        
        return $files;
    }
}