<?php

namespace baseKRIZAN\Config;

/**
 * Pojednostavljenja Config klasa za upravljanje konfiguracijom
 */
class Config {
    private static array $config = [];
    private static bool $isLoaded = false;
    private static array $sensitiveKeys = [
        'dbpassword', 
        'MAIL_USERNAME',
        'MAIL_PASSWORD', 
        'FIREBASE_API_KEY',
        'FIREBASE_SERVER_KEY', 
        'GOOGLE_API_KEY', 
        'APP_SECRET'
    ];

    /**
     * Učitava konfiguraciju iz konfiguracijskih datoteka
     * 
     * @param array $paths Putanje do konfiguracijskih datoteka
     * @return void
     */
    public static function load(array $paths = []): void {
        if (self::$isLoaded) {
            return;
        }

        // Učitaj iz konfiguracijskih datoteka
        foreach ($paths as $path) {
            if (file_exists($path)) {
                $fileConfig = require $path;
                if (is_array($fileConfig)) {
                    self::$config = array_merge(self::$config, $fileConfig);
                }
            }
        }

        self::$isLoaded = true;
        self::loadEnvironmentVariables();
    }

    /**
     * Load all environment variables into the config
     */
    public static function loadEnvironmentVariables(): void {
        foreach ($_ENV as $key => $value) {
            if (!isset(self::$config[$key])) {
                self::$config[$key] = self::castValue($value);
            }
        }
        
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                continue; // Skip HTTP headers
            }
            if (!isset(self::$config[$key]) && !isset($_ENV[$key])) {
                self::$config[$key] = self::castValue($value);
            }
        }
    }

    /**
     * Dohvaća vrijednost iz konfiguracije
     * 
     * @param string $key Ključ konfiguracije
     * @param mixed $default Zadana vrijednost
     * @return mixed Vrijednost konfiguracije
     */
    public static function get(string $key, $default = null) {
        // Za složene ključeve (s točkama)
        if (strpos($key, '.') !== false) {
            return self::getNestedValue($key, $default);
        }
        
        // Prvo provjeri je li u konfiguracijskim datotekama
        if (isset(self::$config[$key])) {
            return self::$config[$key];
        }
        
        // Ako nije, provjeri u env varijablama
        return env($key, $default);
    }

    /**
     * Provjerava postoji li ključ u konfiguraciji
     * 
     * @param string $key Ključ konfiguracije
     * @return bool Postoji li ključ
     */
    public static function has(string $key): bool {
        if (strpos($key, '.') !== false) {
            $segments = explode('.', $key);
            $config = self::$config;

            foreach ($segments as $segment) {
                if (!isset($config[$segment])) {
                    return false;
                }
                $config = $config[$segment];
            }
            return true;
        }
        
        return isset(self::$config[$key]) || isset($_ENV[$key]);
    }

    /**
     * Vraća cijelu konfiguraciju
     * 
     * @return array Cijela konfiguracija
     */
    public static function all(): array {
        return self::$config;
    }

    /**
     * Postavlja vrijednost konfiguracije
     * 
     * @param string $key Ključ konfiguracije
     * @param mixed $value Vrijednost konfiguracije
     * @return void
     */
    public static function set(string $key, $value): void {
        self::$config[$key] = $value;
    }

    /**
     * Dohvaća vrijednost ugniježđenog ključa
     * 
     * @param string $key Ugniježđeni ključ (npr. "database.host")
     * @param mixed $default Zadana vrijednost
     * @return mixed Vrijednost konfiguracije
     */
    private static function getNestedValue(string $key, $default = null) {
        $segments = explode('.', $key);
        $config = self::$config;

        foreach ($segments as $segment) {
            if (!isset($config[$segment])) {
                return $default;
            }
            $config = $config[$segment];
        }

        return $config;
    }

    /**
     * Konvertira vrijednost u odgovarajući tip
     * 
     * @param mixed $value Vrijednost koju treba konvertirati
     * @return mixed Konvertirana vrijednost
     */
    public static function castValue($value) {
        if ($value === 'true' || $value === 'TRUE') {
            return true;
        }
        
        if ($value === 'false' || $value === 'FALSE') {
            return false;
        }
        
        if (is_numeric($value)) {
            return $value + 0; // Convert to int or float
        }
        
        return $value;
    }

    /**
     * Vraća konfiguraciju koja se može koristiti u JavaScriptu
     * 
     * @return array Konfiguracija za JavaScript
     */
    public static function getJavascriptConfig(): array {
        // Dohvati potrebne dijelove konfiguracije za JavaScript
        $jsConfig = [
            'appname' => self::get('appname'),
            'company' => self::get('company'),
            'paths' => [
                'app_url' => self::get('paths.app_url'),
                'root' => self::get('paths.root')
            ],
            'google' => [
                'firebaseprojid' => self::get('google.firebaseprojid')
            ]
        ];
        
        // Filtriraj null vrijednosti
        return array_filter($jsConfig, function($value) {
            // Zadrži prazne nizove ali filtriraj null vrijednosti
            if (is_array($value)) {
                return !empty(array_filter($value, function($v) {
                    return $v !== null;
                }));
            }
            return $value !== null;
        });
    }
    
    /**
     * Dodaje novi osjetljivi ključ
     * 
     * @param string $key Ključ koji treba označiti kao osjetljiv
     * @return void
     */
    public static function addSensitiveKey(string $key): void {
        if (!in_array($key, self::$sensitiveKeys)) {
            self::$sensitiveKeys[] = $key;
        }
    }
    
    /**
     * Provjerava je li ključ označen kao osjetljiv
     * 
     * @param string $key Ključ za provjeru
     * @return bool Je li ključ osjetljiv
     */
    public static function isSensitive(string $key): bool {
        return in_array($key, self::$sensitiveKeys);
    }
    
    /**
     * Vraća konfiguraciju za debugiranje, maskirajući osjetljive vrijednosti
     * 
     * @return array Konfiguracija za debug
     */
    public static function debug(): array {
        $debugConfig = [];
        foreach (self::$config as $key => $value) {
            if (self::isSensitive($key)) {
                $debugConfig[$key] = '******';
            } else {
                $debugConfig[$key] = $value;
            }
        }
        return $debugConfig;
    }
}