<?php

namespace baseKRIZAN\Database;

use PDO;

class DatabaseConnection
{
    private static $instance = null;
    private $pdo;
    private $eventDispatcher = null;
    private $logger = null;

    private function __construct()
    {
        try {
            // Inicijalizacija PDO veze
            $this->pdo = new PDO(
                'mysql:host=' . \baseKRIZAN\Config\Config::get('database.host') . 
                ';dbname=' . \baseKRIZAN\Config\Config::get('database.name') . 
                ';charset=utf8mb4',
                \baseKRIZAN\Config\Config::get('database.username'),
                \baseKRIZAN\Config\Config::get('database.password')
            );
            // Postavljanje atributa za error mode
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (\PDOException $e) {
            throw new \RuntimeException("Database connection failed: " . $e->getMessage());
        }
    }

    /**
     * Set event dispatcher and logger services
     * 
     * @param \baseKRIZAN\Events\EventDispatcher $eventDispatcher Event dispatcher service
     * @param \baseKRIZAN\Error\Logger $logger Logger service
     * @return $this For method chaining
     */
    public function setServices(\baseKRIZAN\Events\EventDispatcher $eventDispatcher = null, \baseKRIZAN\Error\Logger $logger = null)
    {
        $this->eventDispatcher = $eventDispatcher;
        $this->logger = $logger;
        return $this;
    }

    /**
     * Get singleton instance of database connection
     * 
     * @return DatabaseConnection The singleton instance
     */
    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Get PDO instance
     * 
     * @return PDO The PDO connection object
     */
    public function getPDO()
    {
        return $this->pdo;
    }

    /**
     * Create a DatabaseConnection instance from an existing PDO connection
     * 
     * @param PDO $pdo Existing PDO connection
     * @return DatabaseConnection Instance wrapping the provided PDO connection
     */
    public static function createFromPDO(PDO $pdo): self
    {
        $instance = new self();
        $instance->pdo = $pdo;
        return $instance;
    }

    /**
     * Execute a query and return the statement object
     * 
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return \PDOStatement The executed statement
     */
    public function executeQuery(string $query, array $params = []): \PDOStatement
    {
        $startTime = microtime(true);
        
        // Prepare the statement
        $stmt = $this->pdo->prepare($query);
        
        // Execute with parameters
        $stmt->execute($params);
        
        // Calculate execution time
        $executionTime = microtime(true) - $startTime;
        
        // Dispatch event if EventDispatcher is available and query isn't for excluded tables
        if ($this->eventDispatcher && !$this->isExcludedFromLogging($query)) {
            $eventData = [
                'query' => $query,
                'params' => $params,
                'execution_time' => $executionTime
            ];
            
            if ($this->logger) {
                $this->logger->event('Dispatching database.query event', [
                    'execution_time' => $executionTime,
                    'query_preview' => substr($query, 0, 50) . (strlen($query) > 50 ? '...' : '')
                ]);
            }
            
            $this->eventDispatcher->dispatch('database.query', $eventData);
        }
        
        return $stmt;
    }

    /**
     * Check if a query should be excluded from logging
     * 
     * @param string $query SQL query
     * @return bool Whether the query should be excluded
     */
    private function isExcludedFromLogging(string $query): bool
    {
        // Dobavi isključene tablice iz konfiguracije
        $excludedTables = \baseKRIZAN\Config\Config::get('luka.excluded_tables', []);
        
        // Ako je prazan niz, ništa ne isključujemo
        if (empty($excludedTables)) {
            return false;
        }
        
        // Provjeri je li upit vezan za bilo koju isključenu tablicu
        foreach ($excludedTables as $table) {
            if (!empty($table) && stripos($query, $table) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Execute a query without returning results (INSERT, UPDATE, DELETE)
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return int Number of affected rows
     */
    public function execute(string $query, array $params = []): int
    {
        $stmt = $this->executeQuery($query, $params);
        return $stmt->rowCount();
    }

    // NOVE METODE ZA STANDARDIZACIJU

    /**
     * Begin a transaction
     *
     * @return bool Success of operation
     */
    public function beginTransaction()
    {
        return $this->pdo->beginTransaction();
    }

    /**
     * Check if a transaction is active
     *
     * @return bool Transaction status
     */
    public function inTransaction()
    {
        return $this->pdo->inTransaction();
    }

    /**
     * Commit a transaction
     *
     * @return bool Success of operation
     */
    public function commit()
    {
        return $this->pdo->commit();
    }

   /**
     * Rollback a transaction
     *
     * @return bool Success of operation
     */
    public function rollBack()
    {
        return $this->pdo->rollBack();
    }

    /**
     * Return the last inserted ID
     *
     * @param string|null $name Optional sequence name
     * @return string Last inserted ID
     */
    public function lastInsertId($name = null)
    {
        return $this->pdo->lastInsertId($name);
    }

    /**
     * Convert query result to numeric array
     *
     * @param \PDOStatement $stmt PDO statement
     * @return array Result as numeric array
     */
    public function fetchNum(\PDOStatement $stmt)
    {
        return $stmt->fetch(PDO::FETCH_NUM);
    }

   /**
     * Execute query that returns only one value
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return mixed Result
     */
    public function querySingleValue(string $query, array $params = [])
    {
        $stmt = $this->executeQuery($query, $params);
        $result = $stmt->fetch(PDO::FETCH_NUM);
        return $result ? $result[0] : null;
    }

    /**
     * Execute query and return value from first column of first row
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @param int $columnNumber Optional: Column index (0 is default, first column)
     * @return mixed|false Value from first column or false if no results
     */
    public function queryAndFetchColumn(string $query, array $params = [], int $columnNumber = 0)
    {
        $stmt = $this->executeQuery($query, $params);
        return $this->fetchColumn($stmt, $columnNumber);
    }

    /**
     * Convert first column of first row to scalar value
     *
     * @param \PDOStatement $stmt PDO statement
     * @param int $columnNumber Optional: Column index (0 is default, first column)
     * @return mixed|false Value from first column or false if no results
     */
    public function fetchColumn(\PDOStatement $stmt, int $columnNumber = 0)
    {
        return $stmt->fetchColumn($columnNumber);
    }

    /**
     * Execute query and return all results as objects
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @return array Results as array of objects
     */
    public function queryAndFetchAllObjects(string $query, array $params = [], $className = '\stdClass', array $constructorArgs = [])
    {
        $stmt = $this->executeQuery($query, $params);
        return $this->fetchAllObjects($stmt, $className, $constructorArgs);
    }

   /**
     * Convert all query results to array of objects
     *
     * @param \PDOStatement $stmt PDO statement
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @return array All results as array of objects
     */
    public function fetchAllObjects(\PDOStatement $stmt, $className = '\stdClass', array $constructorArgs = [])
    {
        return $stmt->fetchAll(PDO::FETCH_CLASS, $className, $constructorArgs);
    }

   /**
     * Execute query and return first result as object
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @return object|false Result as object or false
     */
    public function queryAndFetchObject(string $query, array $params = [], $className = '\stdClass', array $constructorArgs = [])
    {
        $stmt = $this->executeQuery($query, $params);
        return $this->fetchObject($stmt, $className, $constructorArgs);
    }

    /**
     * Convert query result to object of specified class
     *
     * @param \PDOStatement $stmt PDO statement
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @return object Result as object
     */
    public function fetchObject(\PDOStatement $stmt, $className = '\stdClass', array $constructorArgs = [])
    {
        return $stmt->fetchObject($className, $constructorArgs);
    }

   /**
     * Execute query and return all results as associative array
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return array Results as associative array
     */
    public function queryAndFetchAllAssoc(string $query, array $params = [])
    {
        $stmt = $this->executeQuery($query, $params);
        return $this->fetchAllAssoc($stmt);
    }

    /**
     * Convert all query results to associative array
     *
     * @param \PDOStatement $stmt PDO statement
     * @return array All results as associative array
     */
    public function fetchAllAssoc(\PDOStatement $stmt)
    {
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

   /**
     * Execute query and return first result as associative array
     *
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return array|false Result as associative array or false
     */
    public function queryAndFetchAssoc(string $query, array $params = [])
    {
        $stmt = $this->executeQuery($query, $params);
        return $this->fetchAssoc($stmt);
    }

    /**
     * Convert query result to associative array
     *
     * @param \PDOStatement $stmt PDO statement
     * @return array Result as associative array
     */
    public function fetchAssoc(\PDOStatement $stmt)
    {
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Zaštita singleton patterna
    private function __clone() {}
    
    public function __wakeup()
    {
        throw new \Exception("Deserialization of the database connection is not allowed.");
    }

    public function __sleep() 
    {
        throw new \Exception("Serialization of the database connection is not allowed.");
    }
}