<?php
// app/baseKRIZAN/Error/BaseException.php

namespace baseKRIZAN\Error;

/**
 * Bazna klasa za sve custom iznimke u aplikaciji
 */
class BaseException extends \Exception
{
    /**
     * Tip iznimke (kategorija)
     * @var string
     */
    protected string $exceptionType;
    
    /**
     * HTTP status kod za odgovor
     * @var int
     */
    protected int $httpStatusCode;
    
    /**
     * Dodatni podaci koji će se koristiti za logiranje i prikaz greške
     * @var array
     */
    protected array $context;
    
    /**
     * Označava treba li prikazati detalje korisnicima u produkciji
     * @var bool
     */
    protected bool $isPublic;
    
    /**
     * Konstruktor
     * 
     * @param string $message Poruka o grešci
     * @param int $code Kod greške
     * @param \Throwable|null $previous Prethodna iznimka
     * @param string $exceptionType Tip iznimke
     * @param int $httpStatusCode HTTP status kod
     * @param array $context Dodatni podaci za logiranje
     * @param bool $isPublic Da li je greška javna (prikazati detalje u produkciji)
     */
    public function __construct(
        string $message = "",
        int $code = 0,
        ?\Throwable $previous = null,
        string $exceptionType = ExceptionType::UNKNOWN,
        int $httpStatusCode = 500,
        array $context = [],
        bool $isPublic = false
    ) {
        parent::__construct($message, $code, $previous);
        
        $this->exceptionType = $exceptionType;
        $this->httpStatusCode = $httpStatusCode;
        $this->context = $context;
        $this->isPublic = $isPublic;
    }
    
    /**
     * Dohvaća tip iznimke
     * 
     * @return string Tip iznimke
     */
    public function getExceptionType(): string
    {
        return $this->exceptionType;
    }
    
    /**
     * Dohvaća HTTP status kod
     * 
     * @return int HTTP status kod
     */
    public function getHttpStatusCode(): int
    {
        return $this->httpStatusCode;
    }
    
    /**
     * Dohvaća dodatne podatke za logiranje
     * 
     * @return array Dodatni podaci
     */
    public function getContext(): array
    {
        return $this->context;
    }
    
    /**
     * Vraća da li je iznimka javna
     * 
     * @return bool Da li prikazati detalje u produkciji
     */
    public function isPublic(): bool
    {
        return $this->isPublic;
    }
    
    /**
     * Dodaje podatke u kontekst
     * 
     * @param array $context Dodatni podaci za dodati
     * @return self
     */
    public function addContext(array $context): self
    {
        $this->context = array_merge($this->context, $context);
        return $this;
    }
}