<?php
// app/baseKRIZAN/Error/DatabaseException.php

namespace baseKRIZAN\Error;

/**
 * Iznimka za greške vezane uz bazu podataka
 */
class DatabaseException extends BaseException
{
    /**
     * Konstruktor
     * 
     * @param string $message Poruka o grešci
     * @param int $code Kod greške
     * @param \Throwable|null $previous Prethodna iznimka
     * @param array $context Dodatni podaci za logiranje
     */
    public function __construct(
        string $message = "Database error occurred",
        int $code = 0,
        ?\Throwable $previous = null,
        array $context = []
    ) {
        parent::__construct(
            $message,
            $code,
            $previous,
            ExceptionType::DATABASE,
            500,
            $context,
            false
        );
    }
    
    /**
     * Kreira iznimku iz PDOException
     * 
     * @param \PDOException $exception Originalna PDO iznimka
     * @param string|null $customMessage Prilagođena poruka
     * @return self
     */
    public static function fromPDOException(\PDOException $exception, ?string $customMessage = null): self
    {
        $message = $customMessage ?? 'Database operation failed: ' . $exception->getMessage();
        
        // Stvori Database iznimku od PDO iznimke
        $dbException = new self(
            $message,
            (int)$exception->getCode(),
            $exception,
            [
                'sql_state' => $exception->errorInfo[0] ?? null,
                'driver_code' => $exception->errorInfo[1] ?? null,
                'driver_message' => $exception->errorInfo[2] ?? null,
            ]
        );
        
        return $dbException;
    }
    
    /**
     * Kreira iznimku za grešku povezivanja na bazu podataka
     * 
     * @param string $message Poruka o grešci
     * @param \Throwable|null $previous Prethodna iznimka
     * @return self
     */
    public static function connectionError(string $message = "Database connection failed", ?\Throwable $previous = null): self
    {
        return new self(
            $message,
            0,
            $previous,
            ['connection_error' => true]
        );
    }
    
    /**
     * Kreira iznimku za grešku kod upita
     * 
     * @param string $query SQL upit
     * @param string $message Poruka o grešci
     * @param \Throwable|null $previous Prethodna iznimka
     * @return self
     */
    public static function queryError(string $query, string $message = "Query execution failed", ?\Throwable $previous = null): self
    {
        return new self(
            $message,
            0,
            $previous,
            ['query' => $query]
        );
    }
}