<?php
// app/baseKRIZAN/Error/ErrorLogger.php

namespace baseKRIZAN\Error;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Services\Container;

/**
 * Pojednostavljena klasa za logiranje iznimki
 * 
 * Pruža funkcionalnost za logiranje iznimki s detaljnim kontekstom
 * i određivanje razine logiranja na temelju tipa iznimke.
 */
class ErrorLogger
{
    /**
     * Logger instanca
     * @var Logger
     */
    private Logger $logger;
    
    /**
     * Container instanca (opcijski)
     * @var Container|null
     */
    private ?Container $container;
    
    /**
     * Kontrolira količinu detalja u logu
     * @var bool
     */
    private bool $verbose;
    
    /**
     * Lista exception klasa koje se ignoriraju ili logiraju na nižoj razini
     * @var array
     */
    private array $ignoredExceptions = [
        'NotFoundException' => 'notice', // 404 se logira kao notice
        'ValidationException' => 'notice', // Validation greške se logiraju kao notice
    ];
    
    /**
     * Konstruktor
     * 
     * @param Logger $logger Logger instanca
     * @param Container|null $container Container instanca
     * @param bool $verbose Da li logirati detalje
     */
    public function __construct(
        Logger $logger, 
        ?Container $container = null,
        bool $verbose = true
    ) {
        $this->logger = $logger;
        $this->container = $container;
        $this->verbose = $verbose;
    }
    
    /**
     * Logira iznimku s potpunim kontekstom
     * 
     * @param \Throwable $exception Iznimka koju treba logirati
     * @param Request|null $request Request objekt (opcijski)
     * @param array $additionalContext Dodatni kontekst za logiranje
     * @return void
     */
    public function logException(\Throwable $exception, ?Request $request = null, array $additionalContext = []): void
    {
        // Osnovi kontekst zajednički za sve iznimke
        $context = [
            'exception' => get_class($exception),
            'message' => $exception->getMessage(),
            'code' => $exception->getCode(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine()
        ];
        
        // Dodaj stack trace ako je verbose način rada
        if ($this->verbose) {
            $context['trace'] = $this->formatTraceForLog($exception);
        }
        
        // Dodaj detalje iz BaseException iznimki
        if ($exception instanceof BaseException) {
            $context['exception_type'] = $exception->getExceptionType();
            $context['http_status'] = $exception->getHttpStatusCode();
            $context['is_public'] = $exception->isPublic() ? 'yes' : 'no';
            
            // Spoji BaseException kontekst s osnovnim kontekstom
            $context = array_merge($context, $exception->getContext());
        } else {
            // Za standardne PHP iznimke, odredimo tip
            $context['exception_type'] = ExceptionType::mapExceptionToType($exception);
        }
        
        // Dodaj request informacije ako je dostupno
        if ($request) {
            $context['request'] = $this->getRequestInfo($request);
            
            // Dodaj informacije o korisnikovoj sesiji
            $this->addUserInfo($context);
        }
        
        // Dodaj dodatni kontekst ako je specificiran
        if (!empty($additionalContext)) {
            $context = array_merge($context, $additionalContext);
        }
        
        // Logiraj iznimku s odgovarajućom razinom
        $this->logWithAppropriateLevel($exception, $context);
    }
    
    /**
     * Određuje razinu logiranja na temelju tipa iznimke
     * 
     * @param \Throwable $exception Iznimka koju treba logirati
     * @param array $context Kontekst za logiranje
     * @return void
     */
    private function logWithAppropriateLevel(\Throwable $exception, array $context): void
    {
        $className = basename(str_replace('\\', '/', get_class($exception)));
        
        // Provjeri je li iznimka na listi za ignoriranje ili posebnu razinu
        if (isset($this->ignoredExceptions[$className])) {
            $level = $this->ignoredExceptions[$className];
            
            // Ako je specificirano 'ignore', ne logiramo
            if ($level === 'ignore') {
                return;
            }
            
            // Logiraj s definiranom razinom
            if (method_exists($this->logger, $level)) {
                $this->logger->$level('Exception: ' . $exception->getMessage(), $context);
            } else {
                $this->logger->bootstrap('Exception (using default level): ' . $exception->getMessage(), $context);
            }
            return;
        }
        
        // Odaberi razinu na temelju tipa iznimke
        if ($exception instanceof BaseException) {
            $exceptionType = $exception->getExceptionType();
        } else {
            $exceptionType = ExceptionType::mapExceptionToType($exception);
        }
        
        $level = $this->getLogLevelForExceptionType($exceptionType);
        
        // Logiraj iznimku sa specifičnom razinom za taj tip
        if (method_exists($this->logger, $level)) {
            $this->logger->$level('Exception: ' . $exception->getMessage(), $context);
        } else {
            $this->logger->error('Exception (using fallback level): ' . $exception->getMessage(), $context);
        }
    }
    
    /**
     * Određuje razinu logiranja na temelju tipa iznimke
     * 
     * @param string $exceptionType Tip iznimke
     * @return string Razina logiranja (error, warning, notice, itd.)
     */
    private function getLogLevelForExceptionType(string $exceptionType): string
    {
        return match($exceptionType) {
            ExceptionType::CRITICAL => 'critical',
            ExceptionType::SECURITY => 'error',
            ExceptionType::DATABASE => 'error',
            ExceptionType::VALIDATION => 'notice',
            ExceptionType::ROUTING => 'notice',
            ExceptionType::INFRASTRUCTURE => 'warning',
            ExceptionType::EXTERNAL => 'warning',
            ExceptionType::OPERATIONAL => 'warning',
            default => 'error'
        };
    }
    
    /**
     * Dodaje informacije o korisniku u kontekst
     * 
     * @param array &$context Referenca na kontekst za logiranje
     * @return void
     */
    private function addUserInfo(array &$context): void
    {
        // Dodaj user information if authenticated and container is available
        if ($this->container && $this->container->has('authentication')) {
            try {
                $auth = $this->container->get('authentication');
                if ($auth->isLoggedIn()) {
                    $user = $auth->getUser();
                    $context['user'] = [
                        'id' => $user->id ?? null,
                        'email' => $user->user_email ?? null,
                        'role' => $user->role ?? null
                    ];
                } else {
                    $context['user'] = 'not logged in';
                }
            } catch (\Throwable $e) {
                // Ako ne možemo dohvatiti korisnika, samo nastavimo
                $context['user_error'] = $e->getMessage();
            }
        }
    }
    
    /**
     * Dohvaća informacije o zahtjevu
     * 
     * @param Request $request Request objekt
     * @return array Formatirane informacije o zahtjevu
     */
    private function getRequestInfo(Request $request): array
    {
        return [
            'method' => $request->getMethod(),
            'path' => $request->getPath(),
            'query' => $request->getQuery(),
            'ip' => $request->getIp(),
            'user_agent' => $request->getHeader('User-Agent')
        ];
    }
    
    /**
     * Formatira stack trace za log
     * 
     * @param \Throwable $exception Iznimka čiji trace formatiramo
     * @return string Formatirani trace za log
     */
    private function formatTraceForLog(\Throwable $exception): string
    {
        $trace = $exception->getTraceAsString();
        // Skrati predugačke trace-ove
        if (strlen($trace) > 1500) {
            $trace = substr($trace, 0, 1500) . '... [truncated]';
        }
        return $trace;
    }
    
    /**
     * Postavlja verbose način logiranja
     * 
     * @param bool $verbose Da li logirati detalje
     * @return self
     */
    public function setVerbose(bool $verbose): self
    {
        $this->verbose = $verbose;
        return $this;
    }
    
    /**
     * Dodaje iznimku na listu za ignoriranje ili posebnu razinu logiranja
     * 
     * @param string $exceptionClass Naziv klase iznimke
     * @param string|null $logLevel Razina logiranja ili null za potpuno ignoriranje
     * @return self
     */
    public function ignoreException(string $exceptionClass, ?string $logLevel = 'ignore'): self
    {
        // Izdvoji samo ime klase bez namespace-a
        $className = basename(str_replace('\\', '/', $exceptionClass));
        $this->ignoredExceptions[$className] = $logLevel ?? 'ignore';
        return $this;
    }
}