<?php
// app/baseKRIZAN/Error/ExceptionType.php

namespace baseKRIZAN\Error;

/**
 * Definira tipove iznimki za bolje kategoriziranje grešaka u aplikaciji
 */
class ExceptionType
{
    /**
     * Kritična infrastrukturna greška koja zahtijeva hitnu intervenciju
     */
    public const CRITICAL = 'critical';
    
    /**
     * Sigurnosna greška poput neuspjelog login-a, CSRF tokena, itd.
     */
    public const SECURITY = 'security';
    
    /**
     * Greška vezana uz pristup i manipulaciju podataka
     */
    public const DATABASE = 'database';
    
    /**
     * Greška vezana uz validaciju korisničkog unosa
     */
    public const VALIDATION = 'validation';
    
    /**
     * Greška vezana uz rute (404, 405)
     */
    public const ROUTING = 'routing';
    
    /**
     * Greška vezana uz infrastrukturu (cache, session, itd.)
     */
    public const INFRASTRUCTURE = 'infrastructure';
    
    /**
     * Greška vezana uz eksternu integraciju (API, servisi)
     */
    public const EXTERNAL = 'external';
    
    /**
     * Greška vezana uz operaciju ili biznis logiku
     */
    public const OPERATIONAL = 'operational';
    
    /**
     * Nepoznata greška
     */
    public const UNKNOWN = 'unknown';
    
    /**
     * Vraća sve dostupne tipove grešaka
     * 
     * @return array Lista svih tipova grešaka
     */
    public static function getAllTypes(): array
    {
        return [
            self::CRITICAL,
            self::SECURITY,
            self::DATABASE,
            self::VALIDATION,
            self::ROUTING,
            self::INFRASTRUCTURE,
            self::EXTERNAL,
            self::OPERATIONAL,
            self::UNKNOWN
        ];
    }
    
    /**
     * Mapira PHP iznimke na naše tipove iznimki
     * 
     * @param \Throwable $exception Iznimka koju treba mapirati
     * @return string Tip iznimke
     */
    public static function mapExceptionToType(\Throwable $exception): string
    {
        $class = get_class($exception);
        
        return match(true) {
            $exception instanceof \PDOException => self::DATABASE,
            $exception instanceof \InvalidArgumentException => self::VALIDATION,
            $exception instanceof \ErrorException => self::mapErrorExceptionToType($exception),
            str_contains($class, 'Database') => self::DATABASE,
            str_contains($class, 'Validation') => self::VALIDATION,
            str_contains($class, 'Security') => self::SECURITY,
            str_contains($class, 'Route') || str_contains($class, 'Http') => self::ROUTING,
            str_contains($class, 'External') || str_contains($class, 'Api') => self::EXTERNAL,
            default => self::UNKNOWN
        };
    }
    
    /**
     * Mapira error exception na tip iznimke na temelju severity
     * 
     * @param \ErrorException $exception Error exception
     * @return string Tip iznimke
     */
    private static function mapErrorExceptionToType(\ErrorException $exception): string
    {
        $severity = $exception->getSeverity();
        
        return match($severity) {
            E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR => self::CRITICAL,
            E_WARNING, E_CORE_WARNING, E_COMPILE_WARNING, E_USER_WARNING => self::OPERATIONAL,
            E_NOTICE, E_USER_NOTICE => self::OPERATIONAL,
            E_STRICT, E_DEPRECATED, E_USER_DEPRECATED => self::OPERATIONAL,
            default => self::UNKNOWN
        };
    }
}