<?php

namespace baseKRIZAN\Error;

use Psr\Log\LoggerInterface as PsrLoggerInterface;
use Psr\Log\LogLevel;
use Stringable;

/**
 * Pojednostavljena Logger klasa koja implementira PSR-3 LoggerInterface
 * 
 * Pruža unaprijeđeno logiranje s kategorijama i različitim nivoima
 * s integriranom funkcionalnošću LogManager-a
 */
class Logger implements PsrLoggerInterface, LoggerInterface
{
    /** @var string Putanja do log direktorija */
    private string $logPath;
    
    /** @var bool Oznaka za development mode */
    private bool $developmentMode;
    
    /** @var array Sprema hash već logiranih grešaka da izbjegne dupliciranje */
    private array $loggedErrors = [];
    
    /** @var array Konfiguracijske postavke za kategorije logiranja */
    private array $logCategories = [
        'bootstrap' => ['DEBUG', 'Bootstrap', true],
        'database' => ['INFO', 'Database', false],
        'http' => ['DEBUG', 'HTTP', true],
        'middleware' => ['DEBUG', 'Middleware', true],
        'routing' => ['DEBUG', 'Routing', false],
        'security' => ['WARNING', 'Security', true],
        'session' => ['INFO', 'Session', false],
        'template' => ['INFO', 'Template', false],
        'assets' => ['DEBUG', 'Assets', false],
        'validation' => ['INFO', 'Validation', false],
        'services' => ['DEBUG', 'Services', false],
        'notification' => ['INFO', 'Notification', false],
        'event' => ['INFO', 'Event', false],
        'modules' => ['DEBUG', 'Modules', false],
        'luka' => ['DEBUG', 'luka', false],
        'borna' => ['DEBUG', 'borna', false]
    ];
    
    /** @var int Broj dana za čuvanje logova */
    private int $rotationDays = 30;

    /**
     * Konstruktor
     * 
     * @param string $logPath Putanja do log direktorija
     * @param bool $developmentMode Development mode flag
     * @param array $logCategories Dodatne kategorije za logiranje
     * @param int $rotationDays Broj dana za čuvanje logova
     */
    public function __construct(
        string $logPath, 
        bool $developmentMode, 
        array $logCategories = [],
        int $rotationDays = 30
    ) {
        $this->logPath = $logPath;
        $this->developmentMode = $developmentMode;
        $this->rotationDays = $rotationDays;
        
        // Spoji s default kategorijama ako su proslijeđene
        if (!empty($logCategories)) {
            foreach ($logCategories as $category => $config) {
                if (isset($this->logCategories[$category])) {
                    // Za postojeće kategorije, ažuriraj konfiguraciju
                    $this->logCategories[$category][0] = $config['level'] ?? $this->logCategories[$category][0];
                    $this->logCategories[$category][2] = $config['enabled'] ?? $this->logCategories[$category][2];
                } else {
                    // Za nove kategorije, dodaj s default vrijednostima
                    $this->logCategories[$category] = [
                        $config['level'] ?? 'INFO',
                        $config['prefix'] ?? ucfirst($category),
                        $config['enabled'] ?? true
                    ];
                }
            }
        }
        
        // Automatski pokreni čišćenje starih logova
        $this->rotateAndCleanLogs();
    }

    /**
     * Pokreće rotaciju logova i briše stare datoteke
     * 
     * @return void
     */
    public function rotateAndCleanLogs(): void
    {
        try {
            if (!is_dir($this->logPath)) {
                mkdir($this->logPath, 0777, true);
                return;
            }

            $currentTime = time();
            $cutoffTime = $currentTime - ($this->rotationDays * 86400); // 86400 sekundi = 1 dan

            $logFiles = glob($this->logPath . '/*.log');
            
            if (!$logFiles) {
                return;
            }

            $deletedFiles = 0;
            foreach ($logFiles as $file) {
                $fileName = basename($file);
                
                // Provjeri da li ime datoteke odgovara našem formatu (YYYY-MM-DD.log)
                if (preg_match('/^(\d{4}-\d{2}-\d{2})\.log$/', $fileName, $matches)) {
                    $fileDate = $matches[1];
                    $fileTime = strtotime($fileDate);
                    
                    if ($fileTime && $fileTime < $cutoffTime) {
                        if (unlink($file)) {
                            $deletedFiles++;
                        }
                    }
                }
            }

            if ($deletedFiles > 0) {
                $this->info("Log rotation completed", [
                    'deleted_files' => $deletedFiles,
                    'retention_days' => $this->rotationDays
                ]);
            }
        } catch (\Throwable $e) {
            // Tiho ignoriraj greške tijekom inicijalizacije (jer još ne logiramo)
            // ili logiraj ako smo u fazi gdje možemo
            error_log("Log rotation failed: " . $e->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function emergency(string|Stringable $message, array $context = []): void
    {
        $this->log(LogLevel::EMERGENCY, $message, $context);
    }

    /**
     * @inheritDoc
     */
    public function alert(string|Stringable $message, array $context = []): void
    {
        $this->log(LogLevel::ALERT, $message, $context);
    }

    /**
     * @inheritDoc
     */
    public function critical(string|Stringable $message, array $context = []): void
    {
        $this->log(LogLevel::CRITICAL, $message, $context);
    }

    /**
     * @inheritDoc
     */
    public function error(string|Stringable $message, array $context = []): void
    {
        // Hashiranje da izbjegnemo dupliciranje istih grešaka
        $errorKey = md5((string)$message . (isset($context['file']) ? $context['file'] : '') . 
                        (isset($context['line']) ? $context['line'] : ''));
        
        if (!isset($this->loggedErrors[$errorKey])) {
            $this->loggedErrors[$errorKey] = true;
            $this->log(LogLevel::ERROR, $message, $context);
        }
    }

    /**
     * @inheritDoc
     */
    public function warning(string|Stringable $message, array $context = []): void
    {
        $this->log(LogLevel::WARNING, $message, $context);
    }

    /**
     * @inheritDoc
     */
    public function notice(string|Stringable $message, array $context = []): void
    {
        $this->log(LogLevel::NOTICE, $message, $context);
    }

    /**
     * @inheritDoc
     */
    public function info(string|Stringable $message, array $context = []): void
    {
        $this->log(LogLevel::INFO, $message, $context);
    }

    /**
     * @inheritDoc
     */
    public function debug(string|Stringable $message, array $context = []): void
    {
        if ($this->developmentMode) {
            $this->log(LogLevel::DEBUG, $message, $context);
        }
    }

    /**
     * Glavna metoda za logiranje u kategoriju
     * 
     * @param string $category Kategorija za logiranje
     * @param string|Stringable $message Poruka za logiranje
     * @param array $context Dodatni kontekst
     * @param string|null $level Opcionalni level za override
     * @return void
     */
    public function logToCategory(string $category, string|Stringable $message, array $context = [], ?string $level = null): void
    {
        if (!$this->shouldLogCategory($category)) {
            return;
        }
        
        $logLevel = $level ? strtolower($level) : strtolower($this->getCategoryLevel($category));
        $prefix = $this->getCategoryPrefix($category);
        
        $this->log($logLevel, "[$prefix] " . $message, $context);
    }

    /**
     * Metode za specifične kategorije logiranja
     */
    
    public function bootstrap(string $message, array $context = []): void
    {
        $this->logToCategory('bootstrap', $message, $context);
    }

    public function database(string $message, array $context = []): void
    {
        $this->logToCategory('database', $message, $context);
    }
    
    public function http(string $message, array $context = []): void
    {
        $this->logToCategory('http', $message, $context);
    }
    
    public function middleware(string $message, array $context = []): void
    {
        $this->logToCategory('middleware', $message, $context);
    }

    public function routing(string $message, array $context = []): void
    {
        $this->logToCategory('routing', $message, $context);
    }

    public function security(string $message, array $context = []): void
    {
        $this->logToCategory('security', $message, $context);
    }

    public function session(string $message, array $context = []): void
    {
        $this->logToCategory('session', $message, $context);
    }
    
    public function template(string $message, array $context = [], string $level = null): void
    {
        $this->logToCategory('template', $message, $context, $level);
    }

    public function assets(string $message, array $context = []): void
    {
        $this->logToCategory('assets', $message, $context);
    }
    
    public function validation(string $message, array $context = []): void
    {
        $this->logToCategory('validation', $message, $context);
    }

    public function services(string $message, array $context = []): void
    {
        $this->logToCategory('services', $message, $context);
    }

    public function notification(string $message, array $context = []): void
    {
        $this->logToCategory('notification', $message, $context);
    }

    public function event(string $message, array $context = []): void
    {
        $this->logToCategory('event', $message, $context);
    }

    public function modules(string $message, array $context = []): void
    {
        $this->logToCategory('modules', $message, $context);
    }

    public function luka(string $message, array $context = []): void
    {
        $this->logToCategory('luka', $message, $context);
    }

    public function borna(string $message, array $context = []): void
    {
        $this->logToCategory('borna', $message, $context);
    }
    
    /**
     * Dohvaća konfigurirani log level za kategoriju
     * 
     * @param string $category Kategorija za koju se traži level
     * @return string
     */
    private function getCategoryLevel(string $category): string
    {
        return $this->logCategories[$category][0] ?? 'INFO';
    }
    
    /**
     * Dohvaća prefix za kategoriju
     * 
     * @param string $category Kategorija
     * @return string
     */
    private function getCategoryPrefix(string $category): string
    {
        return $this->logCategories[$category][1] ?? ucfirst($category);
    }
    
    /**
     * Provjerava je li debug logiranje omogućeno
     * 
     * @return bool True ako je debug logiranje omogućeno
     */
    public function isDebugEnabled(): bool
    {
        return $this->developmentMode;
    }

    /**
     * Provjerava treba li logirati specifičnu kategoriju
     * 
     * @param string $category Kategorija za provjeru
     * @return bool
     */
    private function shouldLogCategory(string $category): bool
    {
        if (!isset($this->logCategories[$category])) {
            return $this->developmentMode;
        }
        
        return $this->logCategories[$category][2] ?? true;
    }

    /**
     * Glavna metoda za logiranje
     * 
     * @param string $level Level za logiranje (PSR-3 levels)
     * @param string|Stringable $message Poruka za logiranje
     * @param array $context Dodatni kontekst
     * @return void
     */
    public function log($level, $message, array $context = []): void
    {
        // Dohvati vremensku zonu iz postavki ili koristi default system timezone
        $defaultTimezone = date_default_timezone_get();
        
        // Provjeri da li je iz Config-a potrebno podesiti timezone
        $configTimezone = null;
        if (class_exists('\\baseKRIZAN\\Services\\Config')) {
            $configTimezone = \baseKRIZAN\Config\Config::get('defaulttimezone');
            if ($configTimezone) {
                date_default_timezone_set($configTimezone);
            }
        }
        
        // Normaliziraj razinu loga prema PSR-3 standardu
        $level = $this->normalizeLogLevel($level);
        
        // Odredi putanju za log datoteku
        $logFile = $this->logPath . '/' . date('Y-m-d') . '.log';
        
        // Formatiraj kontekst za logiranje
        $contextStr = !empty($context) ? $this->formatContext($context) : '';
        
        // Convert Stringable to string if needed
        $messageStr = $message instanceof Stringable ? $message->__toString() : (string)$message;
        
        // Kreiraj log poruku
        $logMessage = sprintf(
            "[%s] %s: %s %s\n",
            date('Y-m-d H:i:s'),
            strtoupper($level),
            $messageStr,
            $contextStr
        );

        // Osiguraj da direktorij za logove postoji
        if (!is_dir(dirname($logFile))) {
            mkdir(dirname($logFile), 0777, true);
        }

        // Zapiši log
        error_log($logMessage, 3, $logFile);
        
        // Vrati originalni timezone
        if ($configTimezone) {
            date_default_timezone_set($defaultTimezone);
        }

        // Dispatch event if EventDispatcher is available
        if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
            $app = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
            $container = $app->getContainer();
            
            // Check for recursive logging - add this line:
            if ($level !== 'event' && strpos($message, 'EventDispatcher:') === false) {
                if ($container->has('eventDispatcher')) {
                    $eventDispatcher = $container->get('eventDispatcher');
                    
                    // Dispatch logger.log event
                    $eventData = [
                        'level' => $level,
                        'message' => $message instanceof Stringable ? $message->__toString() : (string)$message,
                        'context' => $context
                    ];
                    
                    $eventDispatcher->dispatch('logger.log', $eventData);
                }
            }
        }
    }

    /**
     * Normalizira log level prema PSR-3 standardu
     * 
     * @param string $level Uneseni level
     * @return string Normalizirani level
     */
    private function normalizeLogLevel(string $level): string
    {
        $level = strtolower($level);
        
        // Osiguraj da je razina jedna od validnih PSR-3 razina
        $validLevels = [
            LogLevel::EMERGENCY, 
            LogLevel::ALERT, 
            LogLevel::CRITICAL, 
            LogLevel::ERROR, 
            LogLevel::WARNING, 
            LogLevel::NOTICE, 
            LogLevel::INFO, 
            LogLevel::DEBUG
        ];
        
        if (!in_array($level, $validLevels)) {
            // Ako nije validna razina, defaultaj na 'info'
            return LogLevel::INFO;
        }
        
        return $level;
    }

    /**
     * Formatira kontekst za log zapis
     * 
     * @param array $context Kontekst za formatiranje
     * @return string Formatirani kontekst
     */
    private function formatContext(array $context): string
    {
        $formatted = [];
        foreach ($context as $key => $value) {
            if (is_array($value) || is_object($value)) {
                $value = json_encode($value, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
            } elseif (!is_string($value) && !is_numeric($value)) {
                $value = var_export($value, true);
            }
            $formatted[] = "$key: $value";
        }
        return implode(', ', $formatted);
    }

    /**
     * Logira s proširenim kontekstom uključujući file/line podatke
     * 
     * @param string $level Log level
     * @param string $message Poruka
     * @param array $context Kontekst
     * @param string|null $file Filename
     * @param int|null $line Line number
     * @return void
     */
    public function logWithContext(
        string $level, 
        string $message, 
        array $context = [], 
        ?string $file = null, 
        ?int $line = null
    ): void {
        $extendedContext = array_merge($context, [
            'file' => $file ?? debug_backtrace()[1]['file'] ?? null,
            'line' => $line ?? debug_backtrace()[1]['line'] ?? null,
            'request_id' => uniqid()
        ]);

        $this->log($level, $message, $extendedContext);
    }
    
    /**
     * Dohvaća putanju log direktorija
     * 
     * @return string
     */
    public function getLogPath(): string
    {
        return $this->logPath;
    }
    
    /**
     * Ažurira konfiguraciju za kategoriju
     * 
     * @param string $category Kategorija koju treba ažurirati
     * @param string|null $level Novi log level
     * @param bool|null $enabled Status omogućenosti
     * @return void
     */
    public function updateCategoryConfig(string $category, ?string $level = null, ?bool $enabled = null): void
    {
        if (!isset($this->logCategories[$category])) {
            $this->logCategories[$category] = ['INFO', ucfirst($category), true];
        }
        
        if ($level !== null) {
            $this->logCategories[$category][0] = $level;
        }
        
        if ($enabled !== null) {
            $this->logCategories[$category][2] = $enabled;
        }
    }
    
    /**
     * Učitava konfiguraciju kategorija iz array-a
     * 
     * @param array $config Konfiguracija kategorija
     * @return void
     */
    public function loadCategoryConfig(array $config): void
    {
        foreach ($config as $category => $settings) {
            $level = $settings['level'] ?? null;
            $enabled = $settings['enabled'] ?? null;
            
            $this->updateCategoryConfig($category, $level, $enabled);
        }
    }
    
    /**
     * Postavlja broj dana za rotaciju logova
     * 
     * @param int $days Broj dana za čuvanje
     * @return void
     */
    public function setRotationDays(int $days): void
    {
        if ($days >= 1) {
            $this->rotationDays = $days;
        }
    }
    
    /**
     * Dohvaća konfiguraciju svih kategorija
     * 
     * @return array
     */
    public function getCategories(): array
    {
        $result = [];
        foreach ($this->logCategories as $category => $config) {
            $result[$category] = [
                'level' => $config[0],
                'prefix' => $config[1],
                'enabled' => $config[2]
            ];
        }
        return $result;
    }

    /**
     * Specify which properties should not be serialized
     * 
     * @return array
     */
    public function __sleep()
    {
        return ['logPath', 'developmentMode', 'logCategories', 'rotationDays'];
    }

    /**
     * Recreate any resources needed after unserialization
     * 
     * @return void
     */
    public function __wakeup()
    {
        $this->loggedErrors = [];
    }
}