<?php
// app/baseKRIZAN/Error/SecurityException.php

namespace baseKRIZAN\Error;

/**
 * Iznimka za sigurnosne greške (neautorizirani pristup, CSRF, itd.)
 */
class SecurityException extends BaseException
{
    /**
     * Konstruktor
     * 
     * @param string $message Poruka o grešci
     * @param int $code Kod greške
     * @param \Throwable|null $previous Prethodna iznimka
     * @param array $context Dodatni podaci za logiranje
     * @param int $httpStatusCode HTTP status kod (default 403)
     */
    public function __construct(
        string $message = "Security violation",
        int $code = 0,
        ?\Throwable $previous = null,
        array $context = [],
        int $httpStatusCode = 403
    ) {
        parent::__construct(
            $message,
            $code,
            $previous,
            ExceptionType::SECURITY,
            $httpStatusCode,
            $context,
            false // Sigurnosne greške nisu javne
        );
    }
    
    /**
     * Kreira iznimku za CSRF grešku
     * 
     * @param string $message Poruka o grešci
     * @param array $context Dodatni podaci za logiranje
     * @return self
     */
    public static function csrfValidationFailed(string $message = "Invalid CSRF token", array $context = []): self
    {
        return new self(
            $message,
            0,
            null,
            array_merge(['type' => 'csrf_validation'], $context),
            403
        );
    }
    
    /**
     * Kreira iznimku za neautorizirani pristup
     * 
     * @param string $message Poruka o grešci
     * @param string $resource Resurs kojem se pokušalo pristupiti
     * @return self
     */
    public static function unauthorized(string $message = "Unauthorized access", string $resource = ''): self
    {
        $context = ['type' => 'unauthorized'];
        
        if (!empty($resource)) {
            $context['resource'] = $resource;
        }
        
        return new self(
            $message,
            0,
            null,
            $context,
            401
        );
    }
    
    /**
     * Kreira iznimku za zabranjeni pristup (forbidden)
     * 
     * @param string $message Poruka o grešci
     * @param string $permission Dozvola koja je nedostajala
     * @return self
     */
    public static function forbidden(string $message = "Access forbidden", string $permission = ''): self
    {
        $context = ['type' => 'forbidden'];
        
        if (!empty($permission)) {
            $context['required_permission'] = $permission;
        }
        
        return new self(
            $message,
            0,
            null,
            $context,
            403
        );
    }
}