<?php
// app/baseKRIZAN/Error/UnifiedErrorHandler.php

namespace baseKRIZAN\Error;

use baseKRIZAN\Http\Response;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\RequestClassifier;
use baseKRIZAN\Template\TemplateRenderer;
use baseKRIZAN\Services\Container;

/**
 * Pojednostavljeni centralizirani handler za sve greške u aplikaciji
 */
class UnifiedErrorHandler 
{
    /**
     * Singleton instance
     * @var self|null
     */
    private static ?self $instance = null;
    
    /**
     * Da li je aplikacija u development modu
     * @var bool
     */
    private bool $developmentMode;
    
    /**
     * Logger instanca
     * @var Logger
     */
    private Logger $logger;
    
    /**
     * Template renderer (opcijski)
     * @var TemplateRenderer|null
     */
    private ?TemplateRenderer $templateRenderer;
    
    /**
     * Error logger instanca
     * @var ErrorLogger
     */
    private ErrorLogger $errorLogger;
    
    /**
     * Container instanca (opcijski)
     * @var Container|null
     */
    private ?Container $container;
    
    /**
     * Konfiguracija za error handling
     * @var array
     */
    private array $config;
    
    /**
     * Gets or creates singleton instance for global error handling
     * 
     * @return self Error handler instance
     */
    public static function getInstance(): self
    {
        if (self::$instance === null) {
            // Try to get from container if available
            if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
                try {
                    $app = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
                    $container = $app->getContainer();
                    
                    if ($container && $container->has('errorHandler')) {
                        self::$instance = $container->get('errorHandler');
                        return self::$instance;
                    }
                } catch (\Throwable $e) {
                    // Silent fail, continue to create new instance
                }
            }
            
            // Create new instance with sensible defaults
            $developmentMode = defined('DEVELOPMENT_MODE') ? DEVELOPMENT_MODE : false;
            
            // Ensure log directory exists
            $logPath = APP_ROOT . '/storage/logs';
            if (!is_dir($logPath)) {
                mkdir($logPath, 0777, true);
            }
            
            $logger = new Logger($logPath, $developmentMode);
            self::$instance = new self($developmentMode, null, $logger, null);
        }
        
        return self::$instance;
    }

    /**
     * Registers global error, exception and shutdown handlers
     * Should be called early in the bootstrap process
     * 
     * @return void
     */
    public static function registerGlobalHandlers(): void
    {
        $instance = self::getInstance();
        
        // Register handlers
        set_error_handler([$instance, 'handleError']);
        set_exception_handler([$instance, 'handleException']);
        register_shutdown_function([$instance, 'handleShutdown']);
    }

    /**
     * Determines the most appropriate response format based on the request
     * 
     * @param Request|null $request Request object
     * @return string Response format ('html', 'json', 'xml', 'plain')
     */
    public function determineResponseFormat(?Request $request = null): string
    {
        // Ako je request dostupan, koristi njegove metode
        if ($request) {
            if ($request->wantsJson() || $request->isAjax()) {
                return 'json';
            }
            
            // Koristi Request::accepts metodu
            if ($request->accepts('application/xml')) {
                return 'xml';
            }
            
            if ($request->accepts('text/plain')) {
                return 'plain';
            }
            
            return 'html';
        }
        
        // Fallback na analizu headera ako Request objekt nije dostupan
        $format = 'html';
        
        // Check Accept header for JSON
        if (isset($_SERVER['HTTP_ACCEPT']) && 
            strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false) {
            $format = 'json';
        }
        
        // Check for AJAX request
        if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
            $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
            $format = 'json';
        }
        
        // Check Accept header for XML
        if (isset($_SERVER['HTTP_ACCEPT']) && 
            strpos($_SERVER['HTTP_ACCEPT'], 'application/xml') !== false) {
            $format = 'xml';
        }
        
        // Check Accept header for plaintext
        if (isset($_SERVER['HTTP_ACCEPT']) && 
            strpos($_SERVER['HTTP_ACCEPT'], 'text/plain') !== false) {
            $format = 'plain';
        }
        
        return $format;
    }
    
    /**
     * Analyzes request type to determine best error response format
     * 
     * @param Request $request Request object
     * @param RequestClassifier|null $classifier Optional classifier
     * @return string Appropriate response format
     */
    public function determineResponseFormatAdvanced(Request $request, ?RequestClassifier $classifier = null): string
    {
        // Ako imamo classifier, koristi ga
        if ($classifier) {
            $category = $classifier->getRequestCategory($request);
            
            // Različiti formati odgovora za različite kategorije
            return match($category) {
                'api' => 'json',
                'asset' => 'plain',
                'redirect' => 'html',
                default => 'html'
            };
        }
        
        // Fallback na standardnu metodu
        return $this->determineResponseFormat($request);
    }
    
    /**
     * Konstruktor
     * 
     * @param bool $developmentMode Da li je aplikacija u development modu
     * @param TemplateRenderer|null $templateRenderer Template renderer instanca
     * @param Logger $logger Logger instanca
     * @param Container|null $container Container instanca
     */
    public function __construct(
        bool $developmentMode, 
        ?TemplateRenderer $templateRenderer, 
        Logger $logger,
        ?Container $container = null
        ) {
            $this->developmentMode = $developmentMode;
            $this->templateRenderer = $templateRenderer;
            $this->logger = $logger;
            $this->container = $container;
            $this->errorLogger = new ErrorLogger($logger, $container);
            
            // Učitaj konfiguraciju za error handling
            $this->loadErrorConfig();
        }
        
    /**
     * Učitava konfiguraciju za error handling
     * 
     * @return void
     */
    private function loadErrorConfig(): void
    {
        // Default konfiguracija
        $defaultConfig = [
            'display_errors' => $this->developmentMode,
            'log_errors' => true,
            'error_reporting' => E_ALL,
            'templates' => [
                'error' => [
                    'dev' => 'Error/errordev.php',
                    'prod' => 'Error/errorprod.php'
                ],
                'http_error' => 'Error/staticerrortemplate.html.php',
                'validation_error' => 'Error/validationerror.php'
            ]
        ];
        
        // Pokušaj dohvatiti konfiguraciju iz Config servisa
        $configFromService = [];
        
        try {
            if (class_exists('\\baseKRIZAN\\Services\\Config')) {
                $configFromService = \baseKRIZAN\Config\Config::get('error', []);
            }
        } catch (\Throwable $e) {
            // Ignoriraj greške i koristi default konfiguraciju
            $this->logger->bootstrap('Could not load error configuration from Config service', [
                'error' => $e->getMessage()
            ]);
        }
        
        // Spoji default i učitanu konfiguraciju
        $this->config = array_merge($defaultConfig, $configFromService);
    }
     
    /**
     * Registrira error i exception handlere
     * 
     * @return void
     */
    public function register(): void
    {
        // Postavi razinu error reporting-a
        error_reporting($this->config['error_reporting']);
        
        // Registriraj handlere
        set_error_handler([$this, 'handleError']);
        set_exception_handler([$this, 'handleException']);
        
        // Registriraj shutdown funkciju za hvatanje fatal errora
        register_shutdown_function([$this, 'handleShutdown']);
        
        $this->logger->bootstrap('Error handlers registered successfully');
    }
     
    /**
     * Rukuje PHP errorima i pretvara ih u ErrorException
     * 
     * @param int $severity Severity level
     * @param string $message Error poruka
     * @param string $file Fajl u kojem se dogodio error
     * @param int $line Linija koda
     * @throws \ErrorException
     * @return bool
     */
    public function handleError(int $severity, string $message, string $file, int $line): bool
    {
        // Ignoriraj errore koji nisu uključeni u error_reporting
        if (!(error_reporting() & $severity)) {
            return false;
        }
        
        // Pretvori error u iznimku
        throw new \ErrorException($message, 0, $severity, $file, $line);
    }
     
    /**
     * Rukuje PHP shutdown-om da uhvati fatalne errore
     * 
     * @return void
     */
    public function handleShutdown(): void
    {
        $error = error_get_last();
        
        // Ako je fatalna greška koja je zaustavila izvršavanje
        if ($error !== null && in_array($error['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_PARSE])) {
            $this->handleException(new \ErrorException(
                $error['message'], 
                0, 
                $error['type'], 
                $error['file'], 
                $error['line']
            ));
        }
    }
     
    /**
     * Rukuje iznimkama i vraća odgovarajući odgovor
     * 
     * @param \Throwable $e Iznimka koja se dogodila
     * @param string|null $format Format odgovora (html, json, itd.)
     * @param Request|null $request HTTP zahtjev ako je dostupan
     * @return Response|string Odgovor za korisnika
     */
    public function handleException(\Throwable $e, ?string $format = 'html', ?Request $request = null)
    {
        // Logiraj iznimku
        if ($this->config['log_errors']) {
            $this->errorLogger->logException($e, $request);
        }
        
        // Pripremi podatke o grešci
        $errorData = $this->prepareErrorData($e);
        
        // Odredi HTTP status kod
        $statusCode = $this->getStatusCodeForException($e);
        
        // Vrati odgovor u traženom formatu
        $response = match($format) {
            'json' => $this->createJsonResponse($errorData, $statusCode),
            'xml' => $this->createXmlResponse($errorData, $statusCode),
            'plain' => $this->createPlainTextResponse($errorData, $statusCode),
            default => $this->createHtmlResponse($errorData, $statusCode, $e, $request)
        };
        
        // Ako je Response objekt, postavi logger
        if ($response instanceof Response && method_exists($response, 'setLogger')) {
            $response->setLogger($this->logger);
        }
        
        return $response;
    }
    
    /**
     * Pretvara iznimku direktno u Response objekt
     * 
     * @param \Throwable $e Iznimka
     * @param Request|null $request HTTP zahtjev ako je dostupan
     * @param RequestClassifier|null $classifier Optional classifier za određivanje tipa zahtjeva
     * @return Response Response objekt
     */
    public function exceptionToResponse(\Throwable $e, ?Request $request = null, ?RequestClassifier $classifier = null): Response
    {
        // Odredi format odgovora
        $format = $request && $classifier ? 
            $this->determineResponseFormatAdvanced($request, $classifier) : 
            $this->determineResponseFormat($request);
        
        // Logiraj iznimku
        if ($this->config['log_errors']) {
            $this->errorLogger->logException($e, $request);
        }
        
        // Pripremi podatke o grešci
        $errorData = $this->prepareErrorData($e);
        
        // Odredi HTTP status kod
        $statusCode = $this->getStatusCodeForException($e);
        
        // Vrati odgovor u traženom formatu
        $response = match($format) {
            'json' => $this->createJsonResponse($errorData, $statusCode),
            'xml' => $this->createXmlResponse($errorData, $statusCode),
            'plain' => $this->createPlainTextResponse($errorData, $statusCode),
            default => $this->createHtmlResponse($errorData, $statusCode, $e, $request)
        };
        
        // Dodaj logger na response
        if ($this->logger && method_exists($response, 'setLogger')) {
            $response->setLogger($this->logger);
        }
        
        return $response;
    }
    
    /**
     * Priprema podatke o grešci za prezentaciju
     * 
     * @param \Throwable $e Iznimka koja se dogodila
     * @return array Podaci o grešci
     */
    private function prepareErrorData(\Throwable $e): array
    {
        // Osnovna poruka za prikaz korisniku
        $userMessage = 'An unexpected error occurred';
        
        // Za iznimke koje su označene kao javne ili u dev modu, prikaži stvarnu poruku
        if ($this->developmentMode || ($e instanceof BaseException && $e->isPublic())) {
            $userMessage = $e->getMessage();
        }
        
        // Osnovni podaci o greški
        $errorData = [
            'message' => $userMessage,
            'error_type' => $this->getReadableExceptionType($e),
            'status_code' => $this->getStatusCodeForException($e),
            'show_details' => $this->developmentMode
        ];
        
        // Dodaj detalje ako smo u development modu
        if ($this->developmentMode) {
            $errorData = array_merge($errorData, [
                'exception_class' => get_class($e),
                'error_message' => $e->getMessage(),
                'error_code' => $e->getCode(),
                'error_file' => $e->getFile(),
                'error_line' => $e->getLine(),
                'error_trace' => $e->getTraceAsString()
            ]);
            
            // Dodaj kontekst iz BaseException
            if ($e instanceof BaseException) {
                $errorData['context'] = $e->getContext();
            }
            
            // Dodaj validation errors za ValidationException
            if ($e instanceof ValidationException) {
                $errorData['validation_errors'] = $e->getValidationErrors();
            }
        } 
        // Za ValidationException uvijek prikaži greške validacije
        elseif ($e instanceof ValidationException) {
            $errorData['validation_errors'] = $e->getValidationErrors();
        }
        
        return $errorData;
    }
        
    /**
     * Vraća čitljivi tip iznimke
     * 
     * @param \Throwable $e Iznimka
     * @return string Čitljivi tip iznimke
     */
    private function getReadableExceptionType(\Throwable $e): string
    {
        if ($e instanceof BaseException) {
            $type = $e->getExceptionType();
            
            return match($type) {
                ExceptionType::VALIDATION => 'Validation Error',
                ExceptionType::DATABASE => 'Database Error',
                ExceptionType::SECURITY => 'Security Error',
                ExceptionType::ROUTING => 'Routing Error',
                ExceptionType::INFRASTRUCTURE => 'System Error',
                ExceptionType::EXTERNAL => 'External Service Error',
                ExceptionType::OPERATIONAL => 'Operational Error',
                ExceptionType::CRITICAL => 'Critical Error',
                default => 'Bootstrap Error'
            };
        }
        
        // Mapiraj standardne PHP iznimke
        return match(true) {
            $e instanceof \ErrorException => $this->getReadableErrorType($e->getSeverity()),
            $e instanceof \PDOException => 'Database Error',
            $e instanceof \InvalidArgumentException => 'Invalid Input',
            $e instanceof \RuntimeException => 'Runtime Error',
            $e instanceof \LogicException => 'Logic Error',
            default => 'Bootstrap Error'
        };
    }
        
    /**
     * Vraća čitljivi tip PHP errora
     * 
     * @param int $severity PHP error severity
     * @return string Čitljivi tip errora
     */
    private function getReadableErrorType(int $severity): string
    {
        return match($severity) {
            E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR => 'Fatal Error',
            E_WARNING, E_CORE_WARNING, E_COMPILE_WARNING, E_USER_WARNING => 'Warning',
            E_NOTICE, E_USER_NOTICE => 'Notice',
            E_STRICT => 'Strict Standards',
            E_DEPRECATED, E_USER_DEPRECATED => 'Deprecated',
            default => 'Unknown Error'
        };
    }
        
    /**
     * Određuje HTTP status kod za iznimku
     * 
     * @param \Throwable $e Iznimka
     * @return int HTTP status kod
     */
    private function getStatusCodeForException(\Throwable $e): int
    {
        // Za BaseException, koristi definirani HTTP status
        if ($e instanceof BaseException) {
            return $e->getHttpStatusCode();
        }
        
        // Za ostale iznimke, mapiraj na odgovarajući HTTP status
        return match(true) {
            $e instanceof \InvalidArgumentException => 400,
            $e instanceof \PDOException => 500,
            default => 500
        };
    }
     
    /**
     * Kreira HTML odgovor za grešku
     * 
     * @param array $errorData Podaci o grešci
     * @param int $statusCode HTTP status kod
     * @param \Throwable $exception Originalna iznimka
     * @param Request|null $request HTTP zahtjev ako je dostupan
     * @return Response
     */
    private function createHtmlResponse(array $errorData, int $statusCode, \Throwable $exception, ?Request $request = null): Response 
    {
        // Koristi Response::renderViewError ako je ValidationException
        if ($exception instanceof ValidationException) {
            return (new Response())->renderError(
                $errorData['message'],
                $errorData['error_type'],
                $statusCode
            );
        }
        
        // Osiguraj da imamo template renderer
        if ($this->templateRenderer === null) {
            // Fallback na Response::html
            $html = $this->generateBasicHtmlErrorPage($errorData);
            return Response::html($html, $statusCode);
        }
        
        // Odaberi template na temelju tipa iznimke i moda
        $template = $this->developmentMode ? 
            $this->config['templates']['error']['dev'] : 
            $this->config['templates']['error']['prod'];
        
        try {
            // Prvo renderiraj error template
            $content = $this->templateRenderer->render($template, $errorData);
            
            // Dohvati authentication informacije za layout
            $loggedIn = false;
            $user = null;
            
            // Pokušaj dohvatiti authentication ako je dostupan
            if ($this->container !== null && $this->container->has('authentication')) {
                $authentication = $this->container->get('authentication');
                $loggedIn = $authentication->isLoggedIn();
                $user = $authentication->getUser();
            }
            
            // Renderaj layout s error sadržajem
            $fullPage = $this->templateRenderer->render('layout.html.php', [
                'output' => $content,
                'title' => 'Error ' . $statusCode,
                'loggedIn' => $loggedIn,
                'user' => $user
            ]);
            
            return Response::html($fullPage, $statusCode);
        } catch (\Throwable $e) {
            // Ako template rendering ne uspije, vrati basic HTML odgovor
            $this->logger->error('Failed to render error template', [
                'error' => $e->getMessage(),
                'template' => $template
            ]);
            
            $html = $this->generateBasicHtmlErrorPage($errorData);
            return Response::html($html, $statusCode);
        }
    }
        
    /**
     * Generira osnovni HTML za error page
     * 
     * @param array $errorData Podaci o grešci
     * @return string HTML sadržaj
     */
    private function generateBasicHtmlErrorPage(array $errorData): string
    {
        $html = '<!DOCTYPE html><html><head><title>Error</title>';
        $html .= '<style>body{font-family:Arial,sans-serif;margin:0;padding:30px;line-height:1.6;}';
        $html .= '.error-container{max-width:600px;margin:0 auto;background:#f8f8f8;padding:20px;border-radius:5px;box-shadow:0 2px 4px rgba(0,0,0,.1);}';
        $html .= 'h1{color:#e74c3c;margin-top:0;}pre{background:#f1f1f1;padding:10px;overflow:auto;}</style></head>';
        $html .= '<body><div class="error-container"><h1>' . htmlspecialchars($errorData['error_type']) . '</h1>';
        $html .= '<p>' . htmlspecialchars($errorData['message']) . '</p>';
        
        // Dodaj detalje u development modu
        if ($errorData['show_details']) {
            $html .= '<h2>Error Details</h2>';
            $html .= '<p><strong>File:</strong> ' . htmlspecialchars($errorData['error_file'] ?? 'Unknown') . '</p>';
            $html .= '<p><strong>Line:</strong> ' . htmlspecialchars($errorData['error_line'] ?? 'Unknown') . '</p>';
            $html .= '<h3>Stack Trace</h3>';
            $html .= '<pre>' . htmlspecialchars($errorData['error_trace'] ?? 'No stack trace available') . '</pre>';
        }
        
        $html .= '</div></body></html>';
        
        return $html;
    }
     
    /**
     * Kreira JSON odgovor za grešku
     * 
     * @param array $errorData Podaci o grešci
     * @param int $statusCode HTTP status kod
     * @return Response
     */
    private function createJsonResponse(array $errorData, int $statusCode): Response 
    {
        // U produkciji prikaži samo osnovne informacije
        $responseData = $this->developmentMode ? 
            ['error' => true, 'data' => $errorData] : 
            ['error' => true, 'message' => $errorData['message'], 'type' => $errorData['error_type']];
        
        // Validation errors uvijek uključi i u produkciji
        if (isset($errorData['validation_errors'])) {
            $responseData['validation_errors'] = $errorData['validation_errors'];
        }
        
        return Response::json($responseData, $statusCode);
    }
    
    /**
     * Kreira XML odgovor za grešku
     * 
     * @param array $errorData Podaci o grešci
     * @param int $statusCode HTTP status kod
     * @return Response
     */
    private function createXmlResponse(array $errorData, int $statusCode): Response
    {
        // Kreiraj XML response
        $xml = new \SimpleXMLElement('<e></e>');
        
        // U produkciji prikaži samo osnovne informacije
        if (!$this->developmentMode) {
            $xml->addChild('message', htmlspecialchars($errorData['message']));
            $xml->addChild('type', htmlspecialchars($errorData['error_type']));
            
            // Validation errors uvijek uključi i u produkciji
            if (isset($errorData['validation_errors'])) {
                $validationNode = $xml->addChild('validation_errors');
                $this->arrayToXml($errorData['validation_errors'], $validationNode);
            }
        } else {
            $this->arrayToXml($errorData, $xml);
        }
        
        return new Response($xml->asXML(), $statusCode, [
            'Content-Type' => 'application/xml; charset=UTF-8'
        ]);
    }
    
    /**
     * Konvertira array u XML
     * 
     * @param array $array Podaci za konverziju
     * @param \SimpleXMLElement $xml XML element
     * @return void
     */
    private function arrayToXml(array $array, \SimpleXMLElement $xml): void
    {
        foreach ($array as $key => $value) {
            if (is_array($value)) {
                if (!is_numeric($key)) {
                    $subnode = $xml->addChild($key);
                    $this->arrayToXml($value, $subnode);
                } else {
                    $subnode = $xml->addChild('item');
                    $this->arrayToXml($value, $subnode);
                }
            } else {
                if (!is_numeric($key)) {
                    $xml->addChild($key, htmlspecialchars((string)$value));
                } else {
                    $xml->addChild('item', htmlspecialchars((string)$value));
                }
            }
        }
    }
    
    /**
     * Kreira plaintext odgovor za grešku
     * 
     * @param array $errorData Podaci o grešci
     * @param int $statusCode HTTP status kod
     * @return Response
     */
    private function createPlainTextResponse(array $errorData, int $statusCode): Response
    {
        $text = $errorData['error_type'] . ': ' . $errorData['message'];
        
        // U development modu dodaj više detalja
        if ($errorData['show_details']) {
            $text .= "\n\nFile: " . ($errorData['error_file'] ?? 'Unknown');
            $text .= "\nLine: " . ($errorData['error_line'] ?? 'Unknown');
            $text .= "\n\nStack Trace:\n" . ($errorData['error_trace'] ?? 'No stack trace available');
        }
        
        return Response::text($text, $statusCode);
    }
     
    /**
     * Rukuje HTTP greškama (404, 403, itd.)
     * 
     * @param int $statusCode HTTP status kod
     * @param string $message Poruka o grešci
     * @param Request|null $request HTTP zahtjev ako je dostupan
     * @return Response
     */
    public function handleHttpError(int $statusCode, string $message = '', ?Request $request = null): Response
    {
        // Koristi standarnu poruku ako nije specificirana
        $message = $message ?: $this->getDefaultMessageForStatusCode($statusCode);
        
        // Osnovi podaci za error template
        $errorData = [
            'error_type' => $this->getStatusCodeTitle($statusCode),
            'message' => $message,
            'status_code' => $statusCode,
            'show_details' => false
        ];
        
        // Logiraj greške s lošim HTTP statusima (4xx i 5xx)
        if ($statusCode >= 400) {
            $logLevel = $statusCode >= 500 ? 'error' : 'notice';
            
            $this->logger->$logLevel('HTTP error ' . $statusCode, [
                'message' => $message,
                'path' => $request ? $request->getPath() : 'unknown',
                'ip' => $request ? $request->getIp() : 'unknown'
            ]);
        }
        
        // Vrati odgovor u odgovarajućem formatu
        if ($request && ($request->wantsJson() || $request->isAjax())) {
            return Response::json(['error' => true, 'message' => $message, 'type' => $errorData['error_type']], $statusCode);
        }
        
        // Za HTML response, koristi Response::renderViewError
        return (new Response())->renderError(
            $message,
            $errorData['error_type'],
            $statusCode
        );
    }
    
    /**
     * Vraća standardnu poruku za HTTP status kod
     * 
     * @param int $statusCode HTTP status kod
     * @return string Standardna poruka
     */
    private function getDefaultMessageForStatusCode(int $statusCode): string
    {
        return match($statusCode) {
            400 => 'Bad Request',
            401 => 'Unauthorized',
            403 => 'Forbidden',
            404 => 'Not Found',
            405 => 'Method Not Allowed',
            409 => 'Conflict',
            422 => 'Unprocessable Entity',
            429 => 'Too Many Requests',
            500 => 'Internal Server Error',
            501 => 'Not Implemented',
            502 => 'Bad Gateway',
            503 => 'Service Unavailable',
            default => 'An error occurred'
        };
    }
    
    /**
     * Vraća čitljivi naslov za HTTP status kod
     * 
     * @param int $statusCode HTTP status kod
     * @return string Naslov za status kod
     */
    private function getStatusCodeTitle(int $statusCode): string
    {
        return match($statusCode) {
            400 => 'Bad Request',
            401 => 'Unauthorized',
            403 => 'Forbidden',
            404 => 'Page Not Found',
            405 => 'Method Not Allowed',
            409 => 'Conflict',
            422 => 'Validation Error',
            429 => 'Rate Limit Exceeded',
            500 => 'Server Error',
            501 => 'Not Implemented',
            502 => 'Bad Gateway',
            503 => 'Service Unavailable',
            default => 'Error ' . $statusCode
        };
    }
    
    /**
     * Postavlja ErrorLogger instancu
     * 
     * @param ErrorLogger $errorLogger Nova ErrorLogger instanca
     * @return self
     */
    public function setErrorLogger(ErrorLogger $errorLogger): self
    {
        $this->errorLogger = $errorLogger;
        return $this;
    }
    
    /**
     * Uključuje ili isključuje prikaz error detalja
     * 
     * @param bool $display Da li prikazati detalje
     * @return self
     */
    public function displayErrorDetails(bool $display): self
    {
        $this->config['display_errors'] = $display;
        return $this;
    }
    
    /**
     * Gets the template renderer
     * 
     * @return TemplateRenderer|null
     */
    public function getTemplateRenderer(): ?TemplateRenderer
    {
        return $this->templateRenderer;
    }

    /**
     * Sets the template renderer
     * 
     * @param TemplateRenderer $templateRenderer
     * @return self
     */
    public function setTemplateRenderer(TemplateRenderer $templateRenderer): self
    {
        $this->templateRenderer = $templateRenderer;
        return $this;
    }
    
    /**
     * Get the logger instance
     *
     * @return Logger
     */
    public function getLogger(): Logger
    {
        return $this->logger;
    }
    
    /**
     * Set the logger instance
     *
     * @param Logger $logger
     * @return self
     */
    public function setLogger(Logger $logger): self
    {
        $this->logger = $logger;
        $this->errorLogger = new ErrorLogger($logger, $this->container);
        return $this;
    }
    
    /**
     * Get the container instance
     *
     * @return Container|null
     */
    public function getContainer(): ?Container
    {
        return $this->container;
    }
    
    /**
     * Set the container instance
     *
     * @param Container $container
     * @return self
     */
    public function setContainer(Container $container): self
    {
        $this->container = $container;
        $this->errorLogger = new ErrorLogger($this->logger, $container);
        return $this;
    }
    
    /**
     * Creates a Response with an exception instance from the logger
     * 
     * @param string $message Error message
     * @param int $code HTTP status code
     * @param string $exceptionType Type of exception (see ExceptionType class)
     * @param Request|null $request The current request if available
     * @return Response
     */
    public function createErrorResponse(string $message, int $code = 500, string $exceptionType = ExceptionType::UNKNOWN, ?Request $request = null): Response
    {
        // Create an appropriate exception based on the type
        $exception = match($exceptionType) {
            ExceptionType::VALIDATION => new ValidationException([], $message, $code),
            ExceptionType::DATABASE => new DatabaseException($message, $code),
            ExceptionType::SECURITY => new SecurityException($message, $code),
            ExceptionType::ROUTING => new NotFoundException($message, 'route'),
            default => new BaseException($message, $code, null, $exceptionType, $code, [], true)
        };
        
        // Use our standard exception handling to create a response
        return $this->exceptionToResponse($exception, $request);
    }
}