<?php
// app/baseKRIZAN/Events/EventDispatcher.php

namespace baseKRIZAN\Events;

use baseKRIZAN\Error\Logger;

/**
 * Centralizirani sustav za dispečiranje događaja
 * 
 * Omogućuje slušatelje (listeners) da se pretplate na događaje,
 * a pošiljatelje (dispatchers) da šalju događaje s podacima.
 * 
 * @package baseKRIZAN\Events
 */
class EventDispatcher
{
    /**
     * @var array Registrirani slušatelji
     */
    private array $listeners = [];
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Konstruktor
     * 
     * @param Logger|null $logger
     */
    public function __construct(?Logger $logger = null)
    {
        $this->logger = $logger;
    }
    
    /**
     * Registrira slušatelja za događaj
     * 
     * @param string $event Ime događaja
     * @param callable $listener Callback funkcija koja će se pozvati kada se dogodi događaj
     * @param int $priority Prioritet slušatelja (veći broj = viši prioritet)
     * @return self
     */
    public function addListener(string $event, callable $listener, int $priority = 0): self
    {
        $this->listeners[$event][$priority][] = $listener;
        
        if ($this->logger) {
            $this->logger->event('EventDispatcher: Added listener', [
                'event' => $event,
                'priority' => $priority
            ]);
        }
        
        return $this;
    }
    
    /**
     * Registrira slušatelje za više događaja odjednom
     * 
     * @param array $events Asocijativni niz [ime_događaja => [callable, ...]]
     * @param int $priority Prioritet slušatelja (veći broj = viši prioritet)
     * @return self
     */
    public function addListeners(array $events, int $priority = 0): self
    {
        foreach ($events as $event => $listeners) {
            foreach ($listeners as $listener) {
                $this->addListener($event, $listener, $priority);
            }
        }
        
        return $this;
    }
    
    /**
     * Uklanja slušatelja za događaj
     * 
     * @param string $event Ime događaja
     * @param callable $listener Callback funkcija koja je prethodno registrirana
     * @return self
     */
    public function removeListener(string $event, callable $listener): self
    {
        if (!isset($this->listeners[$event])) {
            return $this;
        }
        
        foreach ($this->listeners[$event] as $priority => $listeners) {
            foreach ($listeners as $index => $registeredListener) {
                if ($registeredListener === $listener) {
                    unset($this->listeners[$event][$priority][$index]);
                    
                    if ($this->logger) {
                        $this->logger->event('EventDispatcher: Removed listener', [
                            'event' => $event,
                            'priority' => $priority
                        ]);
                    }
                }
            }
            
            // Ukloni prazne prioritete
            if (empty($this->listeners[$event][$priority])) {
                unset($this->listeners[$event][$priority]);
            }
        }
        
        // Ukloni prazne događaje
        if (empty($this->listeners[$event])) {
            unset($this->listeners[$event]);
        }
        
        return $this;
    }
    
    /**
     * Provjerava ima li slušatelja za događaj
     * 
     * @param string $event Ime događaja
     * @return bool
     */
    public function hasListeners(string $event): bool
    {
        return !empty($this->listeners[$event]);
    }
    
    /**
     * Dispečira događaj s podacima
     * 
     * @param string $event Ime događaja
     * @param array $data Podaci događaja
     * @return EventInterface|null
     */
    public function dispatch(string $event, array $data = []): ?EventInterface
    {
        if (!$this->hasListeners($event)) {
            return null;
        }
        
        // Kreiraj Event objekt
        $eventObject = new Event($event, $data);
        
        if ($this->logger) {
            $this->logger->event('EventDispatcher: Dispatching event', [
                'event' => $event,
                'data_keys' => array_keys($data)
            ]);
        }
        
        // Sortiraj prioritete (od najvećeg prema najmanjem)
        $priorities = array_keys($this->listeners[$event]);
        rsort($priorities);
        
        // Pozovi slušatelje po prioritetu
        foreach ($priorities as $priority) {
            foreach ($this->listeners[$event][$priority] as $listener) {
                try {
                    // Pozovi slušatelja s Event objektom
                    $listener($eventObject);
                    
                    // Ako je propagacija zaustavljena, prekini
                    if ($eventObject->isPropagationStopped()) {
                        if ($this->logger) {
                            $this->logger->event('EventDispatcher: Event propagation stopped', [
                                'event' => $event
                            ]);
                        }
                        break 2;
                    }
                } catch (\Throwable $e) {
                    if ($this->logger) {
                        $this->logger->event('EventDispatcher: Error in event listener', [
                            'event' => $event,
                            'error' => $e->getMessage(),
                            'file' => $e->getFile(),
                            'line' => $e->getLine()
                        ]);
                    }
                }
            }
        }
        
        return $eventObject;
    }
    
    /**
     * Uklanja sve slušatelje za događaj
     * 
     * @param string|null $event Ime događaja ili null za sve
     * @return self
     */
    public function clearListeners(?string $event = null): self
    {
        if ($event === null) {
            $this->listeners = [];
            
            if ($this->logger) {
                $this->logger->event('EventDispatcher: Cleared all listeners');
            }
        } else {
            unset($this->listeners[$event]);
            
            if ($this->logger) {
                $this->logger->event('EventDispatcher: Cleared listeners for event', [
                    'event' => $event
                ]);
            }
        }
        
        return $this;
    }
    
    /**
     * Dohvaća sve registrirane slušatelje
     * 
     * @param string|null $event Ime događaja ili null za sve
     * @return array
     */
    public function getListeners(?string $event = null): array
    {
        if ($event === null) {
            return $this->listeners;
        }
        
        return $this->listeners[$event] ?? [];
    }
}