<?php
// app/baseKRIZAN/Http/EntryPoint.php

namespace baseKRIZAN\Http;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Http\Middleware\MiddlewareHandler;
use baseKRIZAN\Routing\Routes;
use baseKRIZAN\Template\TemplateRenderer;
use baseKRIZAN\Error\UnifiedErrorHandler;
use baseKRIZAN\Routing\RouteDispatcher;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Session\SessionManager;

/**
 * Glavna ulazna točka za HTTP zahtjeve.
 * 
 * Ova klasa obrađuje inicijalne HTTP zahtjeve, pokreće middleware chain
 * i prosljeđuje zahtjeve odgovarajućem kontroleru.
 */
class EntryPoint
{
    /**
     * Middleware handler za obradu middleware komponenti
     * 
     * @var MiddlewareHandler
     */
    private MiddlewareHandler $middlewareHandler;
    
    /**
     * Rute aplikacije
     * 
     * @var Routes
     */
    private Routes $routes;
    
    /**
     * Template renderer
     * 
     * @var TemplateRenderer
     */
    private TemplateRenderer $templateRenderer;
    
    /**
     * Error handler
     * 
     * @var UnifiedErrorHandler
     */
    private UnifiedErrorHandler $errorHandler;
    
    /**
     * Route dispatcher
     * 
     * @var RouteDispatcher
     */
    private RouteDispatcher $routeDispatcher;
    
    /**
     * Putanja zahtjeva
     * 
     * @var string
     */
    private string $path;
    
    /**
     * HTTP metoda
     * 
     * @var string
     */
    private string $method;
    
    /**
     * Logger
     * 
     * @var Logger
     */
    private Logger $logger;
    
    /**
     * Session manager
     * 
     * @var SessionManager
     */
    private SessionManager $sessionManager;
    
    /**
     * DI Container
     * 
     * @var Container
     */
    private Container $container;
    
    /**
     * Request classifier
     * 
     * @var RequestClassifier
     */
    private RequestClassifier $requestClassifier;

    /**
     * Request instance
     * 
     * @var Request|null
     */
    private ?Request $request = null;

    /**
     * Inicijalizira EntryPoint.
     * 
     * @param string $path Putanja zahtjeva
     * @param string $method HTTP metoda
     * @param Routes $routes Router instanca
     * @param Container $container DI container
     */
    public function __construct(string $path, string $method, Routes $routes, Container $container) 
    {
        $this->path = $path;
        $this->method = $method;
        $this->routes = $routes;
        $this->container = $container;
        
        // Osiguraj učitavanje dependencija
        $this->ensureDependenciesLoaded();
        
        // Dohvati servise iz containera
        $this->logger = $container->get('logger');
        $this->templateRenderer = $container->get('templateRenderer');
        $this->errorHandler = $container->get('errorHandler');
        $this->sessionManager = $container->get('sessionManager');
        
        // Dohvati ili kreiraj RequestClassifier
        $this->requestClassifier = $this->resolveRequestClassifier();
        
        // Logiraj inicijalizaciju EntryPointa
        $this->logger->http('EntryPoint initialized', [
            'path' => $path,
            'method' => $method
        ]);
        
        $this->routeDispatcher = new RouteDispatcher($this->templateRenderer, $this->logger);
        
        // Stvori MiddlewareHandler s automatskom konfiguracijom
        $this->middlewareHandler = new MiddlewareHandler($this->logger, $this->container);
        
        // Inicijalizacija početnog zahtjeva
        $this->createInitialRequest();
        
        // Postavi middleware-ove
        $this->setupMiddlewares();
    }

    /**
     * Dohvaća ili kreira RequestClassifier
     * 
     * @return RequestClassifier
     */
    private function resolveRequestClassifier(): RequestClassifier
    {
        // Prvo provjeri je li već registriran u containeru
        if ($this->container->has('requestClassifier')) {
            return $this->container->get('requestClassifier');
        }
        
        // Ako nije, kreiraj novu instancu
        $classifier = RequestClassifier::getInstance($this->logger);
        
        // Registriraj u container za buduću upotrebu
        $this->container->register('requestClassifier', $classifier);
        
        return $classifier;
    }

    /**
     * Stvara inicijalni objekt zahtjeva
     * 
     * @return void
     */
    private function createInitialRequest(): void
    {
        // Stvori novi Request objekt
        $this->request = new Request();
        $this->request->setPath($this->path);
        
        // Postavi informaciju da je ovo primarni Request objekt kako bi se izbjeglo dupliranje logiranja
        $this->request->setAttribute('primary_request', true);
        
        // Inicijalni tip zahtjeva je 'web' osim ako se ne dokaže drugačije
        $requestType = 'web';
        
        // Pretprocesiranje: provjeri rutu ali bez izvršavanja kontrolera
        if (method_exists($this->routes, 'preProcessRoute')) {
            $routeInfo = $this->routes->preProcessRoute($this->path, $this->method);
            
            if ($routeInfo && isset($routeInfo['middleware'])) {
                $middleware = $routeInfo['middleware'];
                
                // Ako je middleware array, uzmi prvi element (glavni tip)
                if (is_array($middleware) && !empty($middleware)) {
                    // Za slučaj ako imamo više middleware komponenti, gledamo je li jedna od njih 'api'
                    foreach ($middleware as $mw) {
                        if ($mw === 'api') {
                            $requestType = 'api';
                            break;
                        }
                    }
                } else if ($middleware === 'api') {
                    $requestType = 'api';
                }
                
                $this->logger->http('Route pre-processed', [
                    'path' => $this->path,
                    'determined_type' => $requestType,
                    'middleware' => is_array($middleware) ? implode(', ', $middleware) : $middleware
                ]);
            } else {
                // Ako nije pronađena ruta, koristi requestClassifier za detekciju
                $requestType = $this->requestClassifier->getRequestCategory($this->request);
                
                $this->logger->http('Route not found in pre-processing, falling back to classifier', [
                    'path' => $this->path,
                    'determined_type' => $requestType
                ]);
            }
        } else {
            // Ako preProcessRoute nije dostupan, koristi requestClassifier
            $requestType = $this->requestClassifier->getRequestCategory($this->request);
        }
        
        // Postavi tip zahtjeva
        $this->request->setAttribute('request_type', $requestType);
        
        // Postavimo Logger na Request - ovo će generirati log poruku
        if (method_exists($this->request, 'setLogger')) {
            $this->request->setLogger($this->logger);
        }
        
        // Postavi container u request za potrebe validacije
        if (method_exists($this->request, 'setContainer')) {
            $this->request->setContainer($this->container);
        }
        
        // Registracija u container
        $this->container->register('request', $this->request, 'request');
    }
    
    /**
     * Osigurava da su sve potrebne ovisnosti učitane.
     * 
     * @return void
     */
    private function ensureDependenciesLoaded(): void
    {
        // Provjeri je li applikacija već inicijalizirana (preko Bootstrap::getInstance)
        $app = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
        
        // Osiguraj da su bazne dependencije učitane
        if (!$this->container->has('templateRenderer')) {
            // Ako je aplikacija već bootstrapana, samo dohvati container
            if ($app && method_exists($app, 'getContainer') && $app->getContainer() !== $this->container) {
                // Koristi container iz aplikacije
                $appContainer = $app->getContainer();
                $this->container->merge($appContainer);
                
                // Logiraj samo ako je logger dostupan
                if ($this->container->has('logger')) {
                    $this->container->get('logger')->http('Using container from Bootstrap instance');
                }
            }
            
            // Ako i dalje nema potrebnih servisa, učitaj osnovne providere direktno
            if (!$this->container->has('templateRenderer')) {
                $this->loadCoreServices();
            }
        }
    }
    
    /**
     * Učitava osnovne servise ako nisu dostupni kroz Bootstrap
     * 
     * @return void
     */
    private function loadCoreServices(): void
    {
        // Logiraj samo ako je logger dostupan
        if ($this->container->has('logger')) {
            $this->container->get('logger')->http('Manually loading core services');
        }
        
        // Učitaj osnovne servise ako postoje odgovarajući provideri
        if (!$this->container->has('templateRenderer') && class_exists('\\baseKRIZAN\\Bootstrap\\Initializers\\TemplateInitializer')) {
            (new \baseKRIZAN\Bootstrap\Initializers\TemplateInitializer($this->container))->initialize();
        }
        
        if (!$this->container->has('sessionManager') && class_exists('\\baseKRIZAN\\Bootstrap\\Initializers\\SessionInitializer')) {
            (new \baseKRIZAN\Bootstrap\Initializers\SessionInitializer($this->container))->initialize();
        }
        
        if (!$this->container->has('routes') && class_exists('\\baseKRIZAN\\Bootstrap\\Initializers\\RouteInitializer')) {
            (new \baseKRIZAN\Bootstrap\Initializers\RouteInitializer($this->container))->initialize();
        }
    }

   /**
     * Postavlja middleware komponente na temelju vrste zahtjeva.
     * 
     * @return void
     */
    private function setupMiddlewares(): void
    {
        try {
            // Dohvati kategoriju zahtjeva
            $requestType = $this->request->getAttribute('request_type', 'web');
            
            $this->logger->http('Setting up middlewares for request type', [
                'request_type' => $requestType
            ]);
            
            // Važno: Provjeri stanje MiddlewareRegistry
            if ($this->container->has('middlewareRegistry')) {
                $registry = $this->container->get('middlewareRegistry');
                $this->logger->http('MiddlewareRegistry status', [
                    'defined_groups' => array_keys($registry->listMiddlewares()),
                    'api_middlewares_count' => count($registry->listMiddlewares()['api'] ?? [])
                ]);
            } else {
                $this->logger->error('MiddlewareRegistry not found in container');
            }
            
            // Dodaj middleware-ove za detektirani tip zahtjeva
            $this->middlewareHandler->addGroup($requestType);
            
            // Logiranje broja učitanih middleware-a
            $this->logger->http('Middlewares setup completed', [
                'request_type' => $requestType,
                'middleware_count' => count($this->middlewareHandler->getMiddlewares())
            ]);
            
        } catch (\Exception $e) {
            $this->logger->error('Failed to setup middlewares', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'path' => $this->path
            ]);
        }
    }

    /**
     * Pokreće obradu zahtjeva.
     * 
     * @return void
     */
    public function run(): void 
    {
        try {
            // Koristimo već kreirani request objekt
            $request = $this->request;
            
            // Eksplicitno postaviti request_type na 'api' za API zahtjeve
            if (strpos($request->getPath(), '/api/') !== false) {
                $request->setAttribute('request_type', 'api');
                $this->logger->http('Request classified as API based on path', [
                    'path' => $request->getPath()
                ]);
            }
            
            // VAŽNO: Postavi routerDefinedType u requestClassifier
            if ($request->getAttribute('request_type') === 'api' && $this->requestClassifier) {
                $this->requestClassifier->setRouterDefinedType('api');
            }
            
            // Logiraj obradu zahtjeva
            $this->logger->http('Processing request', [
                'path' => $request->getPath(),
                'method' => $request->getMethod(),
                'request_type' => $request->getAttribute('request_type', 'web')
            ]);
            
            // Funkcija za rukovanje zahtjevom nakon middleware procesiranja
            $handler = function (Request $request): Response {
                try {
                    $routeData = $this->routes->resolveRoute($request->getPath(), $request->getMethod());
                    
                    // IMPORTANT: Check if route was found before trying to use it
                    if (!$routeData) {
                        if ($this->logger) {
                            $this->logger->http('Route not found', [
                                'path' => $request->getPath(),
                                'method' => $request->getMethod()
                            ]);
                        }
                        
                        // Handle 404 Not Found
                        return $this->handleNotFound();
                    }
                    
                    if (isset($routeData['params'])) {
                        $request->setParams($routeData['params']);
                    }
                    
                    // Prosljeđivanje zahtjeva na route dispatcher
                    return $this->routeDispatcher->dispatch(
                        $request, 
                        $routeData,
                        $this->container
                    );
                } catch (\Throwable $e) {
                    $this->logger->error('Error during request handling', [
                        'message' => $e->getMessage(),
                        'path' => $request->getPath()
                    ]);
                    
                    // Use the global error handler
                    $errorHandler = \baseKRIZAN\Error\UnifiedErrorHandler::getInstance();
                    $format = $request->isAjax() || $request->wantsJson() ? 'json' : 'html';
                    
                    $errorResponse = $errorHandler->handleException($e, $format, $request);
                        
                    if ($errorResponse instanceof Response) {
                        return $errorResponse;
                    } else {
                        return new Response($errorResponse, 500);
                    }
                }
            };

            // Get response from middleware chain
            $response = $this->middlewareHandler->handle($request, $handler);
        
            // Postavi logger na odgovor
            if (method_exists($response, 'setLogger')) {
                $response->setLogger($this->logger);
            }
            
            // Send the response
            $response->send();
            
        } catch (\Throwable $e) {
            $this->logger->error('Fatal error in EntryPoint', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            // Use the global error handler
            $errorHandler = \baseKRIZAN\Error\UnifiedErrorHandler::getInstance();
            $errorResponse = $errorHandler->handleException($e, 'html');
            
            if ($errorResponse instanceof Response) {
                $errorResponse->send();
            } else {
                (new Response($errorResponse, 500))->send();
            }
        } finally {
            // Čišćenje request scoped servisa
            if (method_exists($this->container, 'clearScope')) {
                $this->container->clearScope('request');
            }
        }
    }

    /**
     * Rukovanje 404 greškom za nepostojeće stranice.
     * 
     * @return Response 404 odgovor
     */
    private function handleNotFound(): Response 
    {
        $this->logger->http('Page not found', [
            'path' => $this->path,
            'method' => $this->method
        ]);
        
        return (new Response())->renderError('Page not found', 'Error', 404);
    }
}