<?php
// app/baseKRIZAN/Http/Middleware/ApiAuthMiddleware.php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Security\ApiAuth;

/**
 * Middleware za API autentifikaciju
 */
class ApiAuthMiddleware extends Middleware
{
    /**
     * API Auth service
     * 
     * @var ApiAuth|null
     */
    private ?ApiAuth $apiAuth;
    
    /**
     * Konstruktor
     * 
     * @param ApiAuth|null $apiAuth API Auth service
     * @param Logger|null $logger Logger
     */
    public function __construct(?ApiAuth $apiAuth = null, ?Logger $logger = null)
    {
        parent::__construct($logger);
        $this->apiAuth = $apiAuth;
        $this->priority = 40;
    }

    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response
    {
        // Check if global authentication is disabled
        $authEnabled = \baseKRIZAN\Config\Config::get('authentication.enabled', true);
        
        // Check if API authentication is specifically disabled
        $apiAuthEnabled = \baseKRIZAN\Config\Config::get('api_auth.enabled', true);
        
        // If either global auth or API auth is disabled, skip authentication
        if (!$authEnabled || !$apiAuthEnabled) {
            if ($this->logger) {
                $this->logger->middleware("ApiAuthMiddleware: Authentication disabled globally, skipping checks", [
                    'global_auth_enabled' => $authEnabled ? 'true' : 'false',
                    'api_auth_enabled' => $apiAuthEnabled ? 'true' : 'false'
                ]);
            }
            
            // Set api_authenticated to true to prevent issues in controllers
            $request->setAttribute('api_authenticated', true);
            
            return $next($request);
        }
        
        // Ako apiAuth servis nije dostupan, samo propusti zahtjev
        if ($this->apiAuth === null) {
            if ($this->logger) {
                $this->logger->middleware("ApiAuthMiddleware: apiAuth service not available, skipping");
            }
            return $next($request);
        }
        
        // Dohvati token iz headera ili zahtjeva
        $token = $this->extractToken($request);
        
        // Ako token nije dostupan, nastavi bez autentifikacije
        if (empty($token)) {
            if ($this->logger) {
                $this->logger->middleware("ApiAuthMiddleware: No token provided");
            }
            
            // Odluči trebamo li blokirati zahtjev ili pustiti bez autentifikacije
            if ($request->getAttribute('requires_auth', false)) {
                return Response::json([
                    'error' => 'Unauthorized',
                    'message' => 'API token is required'
                ], 401);
            }
            
            return $next($request);
        }
        
        // Validacija tokena s ApiAuth servisom
        $userData = $this->apiAuth->validateToken($token);
        
        if ($userData) {
            // Token je validan, postavi podatke o korisniku u atribute zahtjeva
            $request->setAttribute('api_authenticated', true);
            $request->setAttribute('api_user', $userData);
            $request->setAttribute('user_id', $userData['user_id']);
            
            if ($this->logger) {
                $this->logger->middleware("API authentication successful", [
                    'user_id' => $userData['user_id'],
                    'token_id' => $userData['token_id'],
                    'path' => $request->getPath()
                ]);
            }
        } else {
            // Token nije validan
            $request->setAttribute('api_authenticated', false);
            
            if ($this->logger) {
                $this->logger->middleware("API authentication failed: Invalid token", [
                    'path' => $request->getPath()
                ]);
            }
            
            // Ako ruta zahtijeva autentifikaciju, vrati grešku
            if ($request->getAttribute('requires_auth', false)) {
                return Response::json([
                    'error' => 'Unauthorized',
                    'message' => 'Invalid API token'
                ], 401);
            }
        }
        
        return $next($request);
    }
    
    /**
     * Izdvaja token iz različitih izvora zahtjeva
     * 
     * @param Request $request Zahtjev
     * @return string|null Token ili null ako nije pronađen
     */
    private function extractToken(Request $request): ?string
    {
        // Pokušaj dohvatiti iz Authorization headera
        $authHeader = $request->getHeader('Authorization');
        if (!empty($authHeader)) {
            if (is_array($authHeader)) {
                $authHeader = $authHeader[0];
            }
            
            // Bearer token
            if (strpos($authHeader, 'Bearer ') === 0) {
                return substr($authHeader, 7);
            }
        }
        
        // Pokušaj dohvatiti iz X-API-Token headera
        $apiTokenHeader = $request->getHeader('X-API-Token');
        if (!empty($apiTokenHeader)) {
            return is_array($apiTokenHeader) ? $apiTokenHeader[0] : $apiTokenHeader;
        }
        
        // Pokušaj dohvatiti iz query stringa
        $tokenParam = $request->getQuery('token');
        if (!empty($tokenParam)) {
            return $tokenParam;
        }
        
        return null;
    }
}