<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Routing\Routes;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

/**
 * Middleware za autentifikaciju
 * 
 * @package baseKRIZAN\Http\Middleware
 */
class AuthMiddleware extends Middleware
{
    /**
     * @var Authentication
     */
    private Authentication $authentication;
    
    /**
     * @var Routes
     */
    private Routes $routes;

    /**
     * Konstruktor
     * 
     * @param Authentication $authentication Authentication service
     * @param Routes $routes Routes service
     * @param Logger|null $logger Logger
     */
    public function __construct(
        Authentication $authentication, 
        Routes $routes,
        ?Logger $logger = null
    ) {
        parent::__construct($logger);
        $this->authentication = $authentication;
        $this->routes = $routes;
        $this->priority = 45; // Niži prioritet, nakon sesije i sanitizacije
    }

    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response
    {
        // Check if authentication is globally enabled
        $authEnabled = \baseKRIZAN\Config\Config::get('authentication.enabled', true);
        
        // If authentication is disabled globally, skip auth checks but still provide 
        // essential functionality like attaching the user if they happen to be logged in
        if (!$authEnabled) {
            if ($this->logger) {
                $this->logger->middleware('Authentication globally disabled, skipping auth checks', [
                    'path' => $request->getPath()
                ]);
            }
            
            // Still refresh the session if user is already logged in (convenience)
            if ($this->authentication->isLoggedIn()) {
                $this->authentication->refreshSession();
                $request->setAttribute('user', $this->authentication->getUser());
            } else {
                // Set empty user in request attributes to prevent errors in controllers
                $request->setAttribute('user', null);
            }
            
            // Skip further authentication checks and continue with request
            return $next($request);
        }
        
        // Rest of your original authentication code...
        // Refresh session if user is logged in
        if ($this->authentication->isLoggedIn()) {
            // Ovo će koristiti refreshSession metodu koja uključuje
            // i regeneraciju session ID-a kad je potrebno
            $this->authentication->refreshSession();
            
            // Postavi korisnika u atribute zahtjeva
            $request->setAttribute('user', $this->authentication->getUser());
        }

        // Check if it's an AJAX request
        if ($request->isAjax() && !$this->authentication->isLoggedIn()) {
            return new Response(
                json_encode(['error' => 'Unauthorized']),
                401,
                ['Content-Type' => 'application/json']
            );
        }

        // Pass request to next middleware
        return $next($request);
    }
}