<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\BORNA\BornaManager;
use baseKRIZAN\Security\CsrfProtection;

/**
 * Middleware for enforcing advanced security rules and protections
 * Works as an enhancement to the core SecurityHeadersMiddleware and CsrfMiddleware
 */
class BornaMiddleware extends Middleware
{
    private ?BornaManager $securityManager = null;
    private float $startTime;
    private bool $enabled = false;
    private ?CsrfProtection $csrfProtection = null;
    
    /**
     * Constructor
     */
    public function __construct(?Logger $logger = null)
    {
        parent::__construct($logger);
        $this->priority = 50; // Medium priority
        
        // Initialize CSRF Protection if we have a SessionManager
        $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
        if ($container->has('sessionManager')) {
            $sessionManager = $container->get('sessionManager');
            $this->csrfProtection = new CsrfProtection($logger, $sessionManager);
        }
    }
    
    /**
     * Process request through middleware
     */
    public function process(Request $request, callable $next): Response
    {
        // Initialize timing
        $this->startTime = microtime(true);
        
        // Check if security module is enabled
        $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
        if ($container->has('bornaManager')) {
            $this->securityManager = $container->get('bornaManager');
            $this->enabled = $this->securityManager->isEnabled();
        }
        
        // If security is disabled, pass through
        if (!$this->enabled || !$this->securityManager) {
            return $next($request);
        }
        
        // Skip additional checks for CSRF excluded routes 
        // Use CSRF Protection object to check exclusions
        $route = $request->getPath();
        $extension = pathinfo($route, PATHINFO_EXTENSION);
        
        // Check if route is excluded
        if ($this->csrfProtection) {
            $excludedRoutes = $this->csrfProtection->getExcludedRoutes();
            $excludedExtensions = $this->csrfProtection->getExcludedExtensions();
            
            if (in_array($route, $excludedRoutes) || in_array($extension, $excludedExtensions)) {
                if ($this->logger) {
                    $this->logger->middleware('BORNA: Skipping additional security checks for excluded route', [
                        'path' => $route
                    ]);
                }
                
                return $next($request);
            }
        }
        
        // Check if CSRF has already been validated by CsrfMiddleware
        $csrfValidated = $request->getAttribute('csrf_validated');
        
        // If the request has passed CSRF validation or is a GET request, 
        // skip CSRF checks but perform additional security analysis
        if ($csrfValidated === true || in_array($request->getMethod(), ['GET', 'HEAD', 'OPTIONS'])) {
            if ($this->logger) {
                $this->logger->middleware('BORNA: Request already CSRF validated or safe method');
            }
            return $this->processAdditionalSecurity($request, $next);
        }
        
        // If we reach here, the request wasn't validated by CsrfMiddleware
        // and isn't using a safe method - just pass through to next middleware
        // which would likely be CsrfMiddleware
        return $next($request);
    }
    
    /**
     * Process additional security checks beyond CSRF validation
     * 
     * @param Request $request The request object
     * @param callable $next The next middleware
     * @return Response
     */
    private function processAdditionalSecurity(Request $request, callable $next): Response
    {        
        // Prepare request data for analysis
        $requestData = $this->prepareRequestData($request);
        
        // Check if IP is already blocked
        if ($this->securityManager->isIPBlocked($requestData['ip'])) {
            if ($this->logger) {
                $this->logger->middleware('Blocked request from blocked IP', [
                    'ip' => $requestData['ip'],
                    'path' => $requestData['path']
                ]);
            }
            
            return $this->createBlockedResponse();
        }
        
        // Evaluate request against firewall rules
        $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
        $firewallRules = $container->get('bornaFirewallRules');
        $triggeredRules = $firewallRules->evaluateRequest($requestData);
        
        // Check if any rules require blocking
        foreach ($triggeredRules as $triggered) {
            if ($triggered['action'] === 'block') {
                if ($this->logger) {
                    $this->logger->middleware('Blocked request due to firewall rule', [
                        'ip' => $requestData['ip'],
                        'path' => $requestData['path'],
                        'rule_id' => $triggered['rule_id'],
                        'rule_name' => $triggered['rule']['name']
                    ]);
                }
                
                return $this->createBlockedResponse();
            }
            
            // Handle redirect action
            if ($triggered['action'] === 'redirect' && isset($triggered['rule']['redirect_to'])) {
                return new Response('', 302, [
                    'Location' => $triggered['rule']['redirect_to']
                ]);
            }
        }
        
        // Analyze request for security threats
        $analysisResult = $this->securityManager->analyzeRequest($requestData);
        
        // Get threshold from config or fallback to default
        $blockThreshold = (int)\baseKRIZAN\Config\Config::get('borna.block_threshold');
        
        // Check if threat score exceeds block threshold
        if ($analysisResult['threat_score'] >= $blockThreshold) {
            if ($this->logger) {
                $this->logger->middleware('Blocked request due to high threat score', [
                    'ip' => $requestData['ip'],
                    'path' => $requestData['path'],
                    'threat_score' => $analysisResult['threat_score'],
                    'threats' => $analysisResult['threats']
                ]);
            }
            
            // Auto-block will be handled by the BornaManager
            return $this->createBlockedResponse();
        }
        
        // Continue request processing
        $response = $next($request);
        
        // Log execution time for security checks
        if ($this->logger) {
            $executionTime = microtime(true) - $this->startTime;
            $this->logger->middleware('BORNA security middleware executed', [
                'execution_time' => round($executionTime, 4) . 's',
                'path' => $request->getPath(),
                'threat_score' => $analysisResult['threat_score'] ?? 0
            ]);
        }
        
        return $response;
    }
    
    /**
     * Prepare request data for security analysis
     */
    private function prepareRequestData(Request $request): array
    {
        // Include CSRF token if available
        $csrfToken = $request->getPost('csrf_token') 
            ?? $request->getHeader('X-CSRF-Token')
            ?? $request->getHeader('X-XSRF-Token')
            ?? null;
            
        return [
            'ip' => $request->getIp(),
            'path' => $request->getPath(),
            'method' => $request->getMethod(),
            'query_params' => $request->getQuery(),
            'post_params' => $request->getPost(),
            'cookies' => $this->getRequestCookies(),
            'headers' => $this->getRequestHeaders($request),
            'session_id' => session_id() ?: null,
            'user_id' => $_SESSION['user_id'] ?? null,
            'timestamp' => date('Y-m-d H:i:s'),
            'csrf_token' => $csrfToken,
            'is_ajax' => $request->isAjax()
        ];
    }
    
    /**
     * Helper method to safely get request cookies
     */
    private function getRequestCookies(): array
    {
        return $_COOKIE;
    }
    
    /**
     * Helper method to safely get request headers
     */
    private function getRequestHeaders(Request $request): array
    {
        // Implement a method to safely get headers without directly accessing the private property
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $headerKey = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));
                $headers[$headerKey] = $value;
            }
        }
        return $headers;
    }
    
    /**
     * Create response for blocked requests
     */
    private function createBlockedResponse(): Response
    {
        // Get the container
        $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
        
        // Create a request object for the current request
        $request = new Request();
        
        // First, try to use the UnifiedErrorHandler if available
        if ($container->has('errorHandler')) {
            $errorHandler = $container->get('errorHandler');
            
            // Use handleHttpError which will now wrap in layout.html.php
            return $errorHandler->handleHttpError(
                403, 
                'Your request has been blocked due to security concerns.', 
                $request
            );
        }
        
        // Second option: Use Response::renderViewError if UnifiedErrorHandler isn't available
        // This already wraps in the layout
        return (new Response())->renderError(
            'Your request has been blocked due to security concerns.',
            'Access Denied',
            403
        );
    }
}