<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

/**
 * Middleware za upravljanje HTTP cache headerima
 * 
 * @package baseKRIZAN\Http\Middleware
 */
class CacheMiddleware extends Middleware
{
    /**
     * @var int
     */
    private int $defaultMaxAge = 3600; // 1 sat
    
    /**
     * @var array
     */
    private array $extensionTTL = [
        // Imge tipovi imaju duži TTL
        'jpg' => 604800, // 7 dana
        'jpeg' => 604800,
        'png' => 604800,
        'gif' => 604800,
        'ico' => 2592000, // 30 dana
        'svg' => 604800,
        
        // CSS/JS imaju srednji TTL
        'css' => 86400, // 1 dan
        'js' => 86400,
        
        // Font tipovi imaju duži TTL
        'woff' => 2592000,
        'woff2' => 2592000,
        'ttf' => 2592000,
        'eot' => 2592000,
        'otf' => 2592000
    ];
    
    /**
     * Konstruktor
     * 
     * @param Logger|null $logger Logger
     * @param int $defaultMaxAge Zadani max-age
     */
    public function __construct(?Logger $logger = null, int $defaultMaxAge = 3600)
    {
        parent::__construct($logger);
        $this->defaultMaxAge = $defaultMaxAge;
        $this->priority = 90; // Nizak prioritet (izvršava se kasnije)
    }
    
    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response
    {
        // Prvo propusti zahtjev kroz pipeline
        $response = $next($request);
        
        // Zatim dodaj cache headere ako je primjenjivo
        if ($this->shouldCache($request)) {
            $extension = pathinfo($request->getPath(), PATHINFO_EXTENSION);
            $ttl = $this->getTTLForExtension($extension);
            
            $response->cache($ttl);
        } elseif ($this->shouldPreventCache($request)) {
            // Dodaj no-cache headere za dinamičke zahtjeve
            $response->noCache();
        }
        
        return $response;
    }
    
    /**
     * Određuje treba li cache-irati odgovor
     * 
     * @param Request $request Zahtjev
     * @return bool
     */
    private function shouldCache(Request $request): bool
    {
        // Samo za GET zahtjeve
        if ($request->getMethod() !== 'GET') {
            return false;
        }
        
        // Cacheabilni tipovi datoteka
        $extension = strtolower(pathinfo($request->getPath(), PATHINFO_EXTENSION));
        
        return isset($this->extensionTTL[$extension]);
    }
    
    /**
     * Određuje treba li eksplicitno spriječiti cache
     * 
     * @param Request $request Zahtjev
     * @return bool
     */
    private function shouldPreventCache(Request $request): bool
    {
        // Korisnik je prijavljen - spriječi cache za dinamičke stranice
        if ($request->getAttribute('user') !== null) {
            return true;
        }
        
        // Dinamički zahtjevi
        $dynamicPaths = ['/admin', '/user', '/profile', '/dashboard', '/messages'];
        foreach ($dynamicPaths as $path) {
            if (strpos($request->getPath(), $path) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Određuje TTL za specifičnu ekstenziju
     * 
     * @param string $extension Ekstenzija datoteke
     * @return int TTL u sekundama
     */
    private function getTTLForExtension(string $extension): int
    {
        $extension = strtolower($extension);
        
        // Koristi predefiniranu vrijednost ako postoji
        if (isset($this->extensionTTL[$extension])) {
            return $this->extensionTTL[$extension];
        }
        
        // Inače koristi default
        return $this->defaultMaxAge;
    }
}