<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Security\CsrfProtection;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Error\Logger;

/**
 * Middleware za CSRF zaštitu
 */
class CsrfMiddleware extends Middleware
{
    /**
     * Session manager
     */
    private SessionManager $sessionManager;
    
    /**
     * CSRF Protection instance
     */
    private CsrfProtection $csrfProtection;
    
    /**
     * URL for redirect on CSRF failure
     */
    private string $failedRedirectUrl = '';
    
    /**
     * Constructor
     */
    public function __construct(
        SessionManager $sessionManager,
        ?Logger $logger = null,
        string $failedRedirectUrl = ''
    ) {
        parent::__construct($logger);
        $this->sessionManager = $sessionManager;
        $this->failedRedirectUrl = $failedRedirectUrl;
        $this->priority = 40; // Relativno visok prioritet
        
        // Initialize CSRF protection
        $this->csrfProtection = new CsrfProtection($this->logger, $this->sessionManager);
    }

    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response
    {
        // Check if validation should be skipped
        if ($this->shouldSkipCsrfCheck($request)) {
            return $next($request);
        }
        
        // Generate token and add to request attributes
        $token = $this->csrfProtection->generateToken();
        $request->setAttribute('csrfToken', $token);
        
        // Skip validation for safe methods
        if (in_array($request->getMethod(), ['GET', 'HEAD', 'OPTIONS'])) {
            return $next($request);
        }
        
        // Extract token from request
        $receivedToken = $this->extractTokenFromRequest($request);
        
        // Validate token
        $isValid = $this->csrfProtection->validateToken($receivedToken, $request);
        
        // Set validation status in request attributes
        $request->setAttribute('csrf_validated', $isValid);
        
        if (!$isValid) {
            return $this->handleInvalidToken($request);
        }
        
        // Token is valid, continue with request processing
        return $next($request);
    }
    
    /**
     * Check if CSRF check should be skipped for the current request
     */
    private function shouldSkipCsrfCheck(Request $request): bool
    {
        $path = $request->getPath();
        
        // API requests are excluded
        if ($request->getAttribute('request_type') === 'api') {
            return true;
        }
        
        // Asset requests are excluded
        if ($request->getAttribute('request_type') === 'asset') {
            return true;
        }
        
        // Check directly excluded routes
        $excludedRoutes = $this->csrfProtection->getExcludedRoutes();
        if (in_array($path, $excludedRoutes)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Extract CSRF token from request
     */
    private function extractTokenFromRequest(Request $request): ?string
    {
        // Try to extract from POST parameters
        $token = $request->getPost('csrf_token');
        if ($token) {
            return $token;
        }
        
        // Try to extract from headers
        foreach (['X-CSRF-Token', 'X-XSRF-Token'] as $headerName) {
            $headerValue = $request->getHeader($headerName);
            if (!empty($headerValue)) {
                return is_array($headerValue) ? $headerValue[0] : $headerValue;
            }
        }
        
        return null;
    }
    
    /**
     * Handle invalid CSRF token
     */
    private function handleInvalidToken(Request $request): Response
    {
        if ($this->logger) {
            $this->logger->middleware('CSRF: Token validation failed', [
                'path' => $request->getPath()
            ]);
        }
        
        // Return JSON response for AJAX requests
        if ($request->isAjax() || $request->wantsJson()) {
            return Response::json([
                'success' => false,
                'error' => 'CSRF token verification failed',
                'message' => 'Invalid security token provided. Please refresh the page and try again.'
            ], 403);
        }
        
        // Redirect if redirect URL is set
        if (!empty($this->failedRedirectUrl)) {
            return Response::redirect($this->failedRedirectUrl, 303); // 303 See Other
        }
        
        // Default response: error view
        return (new Response())->renderError(
            'Invalid security token. Please go back and try again.',
            'Security Error',
            403
        );
    }
}