<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\LUKA\MetricsCollector;
use baseKRIZAN\LUKA\LukaManager;

/**
 * Middleware for collecting performance metrics during request processing
 */
class LukaMiddleware extends Middleware
{
    private ?MetricsCollector $collector = null;
    private float $startTime;
    private bool $enabled = false;
    
    /**
     * Routes or patterns to exclude from metrics collection
     * 
     * @var array
     */
    private array $excludedRoutes = [];
    
    /**
     * File extensions to exclude from metrics collection
     * 
     * @var array
     */
    private array $excludedExtensions = [];

    /**
     * Constructor
     * 
     * @param Logger|null $logger Logger
     */
    public function __construct(?Logger $logger = null)
    {
        parent::__construct($logger);
        $this->priority = 15; // Higher priority, collect metrics early
        
        // Load excluded routes and extensions from configuration
        $this->loadExclusionConfiguration();
    }

    /**
     * Process request through middleware
     */
    public function process(Request $request, callable $next): Response
    {
        // Initialize timing
        $this->startTime = microtime(true);
    
        // Debug: Provjeri je li kolektor dostupan
        if ($this->logger) {
            $this->logger->middleware('Checking collector availability', [
                'collector_exists' => isset($this->collector) ? 'yes' : 'no',
                'middleware_enabled' => $this->enabled ? 'yes' : 'no'
            ]);
        }
        
        // Check if metrics collection is enabled using global container
        $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
        if ($container->has('lukaMetricsCollector')) {
            $this->collector = $container->get('lukaMetricsCollector');
            $this->enabled = true;
        }
        
        // Check if the current route should be excluded from metrics
        $path = $request->getPath();
        if ($this->shouldExcludeRoute($path)) {
            if ($this->logger) {
                $this->logger->middleware('Skipping metrics for excluded route', [
                    'path' => $path
                ]);
            }
            
            // Skip metrics collection but continue with request processing
            return $next($request);
        }
        
        // Start tracking the request if metrics collection is enabled
        if ($this->enabled && $this->collector) {
            // Debug: Provjeri pozivanje startRequest
            $this->logger->middleware('Starting request tracking', [
                'path' => $request->getPath(),
                'method' => $request->getMethod()
            ]);
            // Start tracking the request (get headers safely)
            $this->collector->startRequest(
                $request->getPath(),
                $request->getMethod(),
                $this->getRequestHeaders($request)
            );
        }
        
        // Continue request processing
        $response = $next($request);
        
        // Record metrics at the end of request
        if ($this->enabled && $this->collector) {
            $executionTime = microtime(true) - $this->startTime;
        
            // Debug: Provjeri prije dovršetka zahtjeva
            $this->logger->middleware('Completing request tracking', [
                'path' => $request->getPath(),
                'status_code' => $response->getStatusCode(),
                'execution_time' => round($executionTime, 4) . 's'
            ]);
            
            // Record total request execution time
            $this->collector->recordComponentExecution('request_total', $executionTime, [
                'path' => $request->getPath(),
                'method' => $request->getMethod()
            ]);
            
            // Complete request tracking
            $this->collector->completeRequest($response->getStatusCode());
            
            // Log middleware execution
            if ($this->logger) {
                $this->logger->middleware('metrics middleware executed', [
                    'execution_time' => round($executionTime, 4) . 's',
                    'path' => $request->getPath()
                ]);
            }
        }
        
        return $response;
    }
    
    /**
     * Load route exclusion configuration from Config service
     */
    private function loadExclusionConfiguration(): void
    {
        // Load excluded routes from configuration
        $excludedRoutesConfig = \baseKRIZAN\Config\Config::get('luka.excluded_routes');
        if (!empty($excludedRoutesConfig)) {
            $this->excludedRoutes = array_map('trim', explode(',', $excludedRoutesConfig));
        } else {
            // Default routes to exclude if not configured
            $this->excludedRoutes = [
                'health', 'ping', 'status',
                'metrics', 'prometheus',
                'favicon.ico'
            ];
        }
        
        // Load excluded file extensions from configuration
        $excludedExtensionsConfig = \baseKRIZAN\Config\Config::get('luka.excluded_extensions');
        if (!empty($excludedExtensionsConfig)) {
            $this->excludedExtensions = array_map('trim', explode(',', $excludedExtensionsConfig));
        } else {
            // Default file extensions to exclude if not configured
            $this->excludedExtensions = [
                'css', 'js', 'jpg', 'jpeg', 'png', 'gif', 'svg', 
                'ico', 'woff', 'woff2', 'ttf', 'eot', 'map'
            ];
        }
        
        if ($this->logger) {
            $this->logger->middleware('Loaded exclusion configuration', [
                'excluded_routes' => count($this->excludedRoutes),
                'excluded_extensions' => count($this->excludedExtensions)
            ]);
        }
    }
    
    /**
     * Determine if the current route should be excluded from metrics
     */
    private function shouldExcludeRoute(string $path): bool
    {
        // Check against excluded routes/paths
        foreach ($this->excludedRoutes as $excludedRoute) {
            if ($excludedRoute[0] === '/' && preg_match($excludedRoute, $path)) {
                // Regex pattern match
                return true;
            } elseif (strpos($path, $excludedRoute) !== false) {
                // Simple substring match
                return true;
            }
        }
        
        // Check file extension
        $extension = pathinfo($path, PATHINFO_EXTENSION);
        if (!empty($extension) && in_array(strtolower($extension), $this->excludedExtensions)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Helper method to safely get request headers
     */
    private function getRequestHeaders(Request $request): array
    {
        // Implement a method to safely get headers without directly accessing the private property
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $headerKey = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));
                $headers[$headerKey] = $value;
            }
        }
        return $headers;
    }
    
    /**
     * Helper method to safely get response size using getBody instead of reflection
     */
    private function getResponseSize(Response $response): int
    {
        try {
            // Use the getBody method if available
            if (method_exists($response, 'getBody')) {
                $content = $response->getBody();
                return is_string($content) ? strlen($content) : 0;
            }
            
            // Fallback to headers if Content-Length is set
            $headers = $response->getHeaders();
            if (isset($headers['Content-Length'])) {
                return (int)$headers['Content-Length'];
            }
            
            return 0;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->middleware('Failed to determine response size', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0; // Unable to determine size
        }
    }
    
    /**
     * @inheritDoc
     */
    public function terminate(Request $request, Response $response): void
    {
        // Check if the current route should be excluded from metrics
        if ($this->shouldExcludeRoute($request->getPath())) {
            return;
        }
        
        // Record additional metrics if enabled
        if ($this->enabled && $this->collector) {
            // Calculate full request time including terminate phase
            $fullExecutionTime = microtime(true) - $this->startTime;
            
            // Record event for full request lifecycle
            $this->collector->recordEvent('request_completed', [
                'path' => $request->getPath(),
                'method' => $request->getMethod(),
                'status' => $response->getStatusCode(),
                'execution_time' => $fullExecutionTime,
                'response_size' => $this->getResponseSize($response)
            ]);
            
            // Dodano: Osiguraj da se zahtjev završi ako već nije
            $this->collector->completeRequest($response->getStatusCode());
        }
    }
    
    /**
     * Add a route or pattern to exclude from metrics
     */
    public function addExcludedRoute(string $route): void
    {
        if (!in_array($route, $this->excludedRoutes)) {
            $this->excludedRoutes[] = $route;
            
            if ($this->logger) {
                $this->logger->middleware('Added excluded route', [
                    'route' => $route
                ]);
            }
        }
    }
    
    /**
     * Add a file extension to exclude from metrics
     */
    public function addExcludedExtension(string $extension): void
    {
        $extension = strtolower(trim($extension, '.'));
        if (!in_array($extension, $this->excludedExtensions)) {
            $this->excludedExtensions[] = $extension;
            
            if ($this->logger) {
                $this->logger->middleware('Added excluded extension', [
                    'extension' => $extension
                ]);
            }
        }
    }
}