<?php
// app/baseKRIZAN/Http/Middleware/MiddlewareHandler.php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Http\Middleware\MiddlewareRegistry;

/**
 * Manages middleware execution.
 */
class MiddlewareHandler
{
    /**
     * Registered middleware instances for current request
     * 
     * @var array<MiddlewareInterface>
     */
    private array $middlewares = [];
    
    /**
     * Logger instance
     */
    private Logger $logger;
    
    /**
     * DI container
     */
    private Container $container;
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, Container $container)
    {
        $this->logger = $logger;
        $this->container = $container;
    }

    /**
     * Add middleware to pipeline
     */
    public function add(MiddlewareInterface $middleware): self
    {
        $this->middlewares[] = $middleware;
        return $this;
    }
    
    /**
     * Add middleware by name from container
     * 
     * @throws \RuntimeException If middleware is not found or invalid
     */
    public function addByName(string $middlewareName): self
    {
        // Koristite novi pristup registracije
        $middleware = $this->container->get($middlewareName);
        
        if (!$middleware instanceof MiddlewareInterface) {
            throw new \RuntimeException("Service '{$middlewareName}' is not a valid middleware");
        }
        
        $this->middlewares[] = $middleware;
        return $this;
    }
    
    /**
     * Add group of middlewares by type
     */
    public function addGroup(string $groupType): self 
    {
        // Osiguraj postojanje MiddlewareRegistry
        if (!$this->container->has('middlewareRegistry')) {
            $this->logger->middleware('MiddlewareRegistry not found in container, creating new one');
            $this->container->register('middlewareRegistry', 
                new MiddlewareRegistry($this->container, $this->logger)
            );
        }

        $registry = $this->container->get('middlewareRegistry');
        
        $registeredMiddlewares = $registry->registerMiddlewares(strtolower($groupType));

        $this->logger->middleware("Registering middlewares for group: {$groupType}", [
            'count' => count($registeredMiddlewares),
            'middlewares' => array_map(function($m) { 
                return get_class($m); 
            }, $registeredMiddlewares)
        ]);

        foreach ($registeredMiddlewares as $middleware) {
            $this->middlewares[] = $middleware;
        }
        
        return $this;
    }
    
    /**
     * Execute middleware pipeline and handle request
     * 
     * @param Request $request The request
     * @param callable $controller Controller callback
     * @return Response The response
     * @throws \Throwable If middleware processing fails
     */
    public function handle(Request $request, callable $controller): Response
    {
        try {
            // Set logger on request if possible, but don't create duplicate log entries
            // Check if request is already primary to avoid redundant logging
            if (method_exists($request, 'setLogger') && !$request->getAttribute('primary_request', false)) {
                // Mark this request as non-primary to prevent duplicate logging
                $request->setAttribute('primary_request', false);
                $request->setLogger($this->logger);
            }
            
            if ($this->logger) {
                $this->logger->middleware('Starting middleware chain', [
                    'middleware_count' => count($this->middlewares),
                    'path' => $request->getPath()
                ]);
            }

            // NOVI DIO - provjera middlewarea
            if (empty($this->middlewares)) {
                // Log warning if no middlewares are loaded
                if ($this->logger) {
                    $this->logger->middleware('No middlewares loaded for current request', [
                        'path' => $request->getPath()
                    ]);
                }
            }
            
            // Sort middlewares by priority (lower number = higher priority)
            usort($this->middlewares, function($a, $b) {
                return $a->getPriority() <=> $b->getPriority();
            });
            
            // Create pipeline endpoint that executes controller
            $stack = function (Request $req) use ($controller): Response {
                if ($this->logger) {
                    $this->logger->middleware('All middleware processed, executing controller');
                }
                
                $response = $controller($req);
                
                // Set logger on response if possible
                if (method_exists($response, 'setLogger')) {
                    $response->setLogger($this->logger);
                }
                
                return $response;
            };

            // Build middleware pipeline
            $pipeline = $this->buildPipeline($this->middlewares, $stack);

            // Execute pipeline
            $response = $pipeline($request);
            
            // Log completion of middleware chain
            if ($this->logger) {
                $this->logger->middleware('Middleware chain completed');
            }
            
            // Call terminate methods on all middlewares
            $this->terminateMiddlewares($this->middlewares, $request, $response);

            return $response;
        } catch (\Throwable $e) {
            $this->logger->error("Middleware error", [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'route' => $request->getPath()
            ]);
            throw $e;
        }
    }
    
    /**
     * Build middleware pipeline in a debuggable way
     * 
     * @param array $middlewares Array of middlewares
     * @param callable $destination Final destination (controller)
     * @return callable Pipeline function
     */
    private function buildPipeline(array $middlewares, callable $destination): callable
    {
        // Start with the destination callback
        $pipeline = $destination;
        
        // Wrap middleware in reverse order
        for ($i = count($middlewares) - 1; $i >= 0; $i--) {
            $middleware = $middlewares[$i];
            $middlewareName = get_class($middleware);
            $logger = $this->logger;
            
            // Create closure for current middleware
            $nextPipeline = $pipeline;
            $pipeline = function (Request $request) use ($middleware, $nextPipeline, $middlewareName, $logger) {
                if ($logger) {
                    $logger->middleware('Executing middleware', [
                        'middleware' => $middlewareName
                    ]);
                }
                
                return $middleware->process($request, $nextPipeline);
            };
        }
        
        return $pipeline;
    }

    /**
     * Dohvaća registrirane middleware instance
     * 
     * @return array<MiddlewareInterface> Registrirani middleware-i
     */
    public function getMiddlewares(): array
    {
        return $this->middlewares;
    }
    
    /**
     * Call terminate methods on all middlewares
     * 
     * @param array $middlewares Middleware instances
     * @param Request $request The request
     * @param Response $response The response
     */
    private function terminateMiddlewares(array $middlewares, Request $request, Response $response): void
    {
        foreach ($middlewares as $middleware) {
            try {
                $middleware->terminate($request, $response);
            } catch (\Throwable $e) {
                $this->logger->middleware("Middleware terminate error", [
                    'middleware' => get_class($middleware),
                    'message' => $e->getMessage()
                ]);
            }
        }
    }
    
    /**
     * Clear all middlewares
     * 
     * @return self For method chaining
     */
    public function clear(): self
    {
        $this->middlewares = [];
        return $this;
    }
}