<?php
namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Error\Logger;

class MiddlewareRegistry
{
    private Container $container;
    private Logger $logger;
    private array $middlewareDefinitions = [
        'global' => [],
        'web' => [],
        'api' => [],
        'assets' => []
    ];

    public function __construct(Container $container, Logger $logger)
    {
        $this->container = $container;
        $this->logger = $logger;
    }

    public function registerMiddlewares(string $group): array
    {
        $registeredMiddlewares = [];

        // Dohvati definicije middleware-a za globalnu i specifičnu grupu
        $groupMiddlewares = array_merge(
            $this->middlewareDefinitions['global'] ?? [],
            $this->middlewareDefinitions[$group] ?? []
        );
        
        // Ako je grupa 'api' a nemamo definiranih middleware-a, koristimo default API middleware konfiguraciju
        if ($group === 'api' && empty($groupMiddlewares)) {
            $this->logger->middleware("No middleware definitions found for API group, using defaults");
            $groupMiddlewares = [
                '\\baseKRIZAN\\Http\\Middleware\\SecurityHeadersMiddleware' => [
                    'priority' => 20,
                    'requiredServices' => ['logger']
                ],
                '\\baseKRIZAN\\Http\\Middleware\\SanitizationMiddleware' => [
                    'priority' => 30,
                    'requiredServices' => ['container', 'logger']
                ]
            ];
        }

        $this->logger->middleware("Registering middlewares for group: {$group}", [
            'defined_count' => count($groupMiddlewares),
            'defined_middlewares' => array_keys($groupMiddlewares)
        ]);

        foreach ($groupMiddlewares as $middlewareClass => $config) {
            try {
                if (!class_exists($middlewareClass)) {
                    $this->logger->middleware("Middleware class not found: $middlewareClass");
                    continue;
                }

                // Identifikacija potrebnih servisa
                $requiredServices = $config['requiredServices'] ?? [];
                
                // Dodajemo podrazumevane servise na osnovu klase middleware-a
                if (strpos($middlewareClass, 'SessionMiddleware') !== false && !in_array('sessionManager', $requiredServices)) {
                    $requiredServices[] = 'sessionManager';
                }
                
                if (strpos($middlewareClass, 'RateLimiterMiddleware') !== false) {
                    if (!in_array('templateRenderer', $requiredServices)) {
                        $requiredServices[] = 'templateRenderer';
                    }
                }
                
                if (strpos($middlewareClass, 'SanitizationMiddleware') !== false) {
                    if (!in_array('container', $requiredServices)) {
                        array_unshift($requiredServices, 'container'); // Dodajemo na početak jer je prvi parametar
                    }
                }
                
                if (strpos($middlewareClass, 'CsrfMiddleware') !== false) {
                    if (!in_array('sessionManager', $requiredServices)) {
                        array_unshift($requiredServices, 'sessionManager');
                    } else {
                        // Ako već postoji, presloži da bude prvi
                        $key = array_search('sessionManager', $requiredServices);
                        if ($key !== 0) {
                            unset($requiredServices[$key]);
                            array_unshift($requiredServices, 'sessionManager');
                        }
                    }
                    
                    // Izbaci csrfProtection iz liste ako postoji
                    $csrfKey = array_search('csrfProtection', $requiredServices);
                    if ($csrfKey !== false) {
                        unset($requiredServices[$csrfKey]);
                    }
                }

                // Provjera dostupnosti svih potrebnih servisa
                $allServicesAvailable = true;
                $missingServices = [];
                
                foreach ($requiredServices as $service) {
                    if ($service !== 'logger' && $service !== 'container' && !$this->container->has($service)) {
                        $allServicesAvailable = false;
                        $missingServices[] = $service;
                    }
                }
                
                if (!$allServicesAvailable) {
                    $this->logger->middleware("Skipping middleware {$middlewareClass} due to missing services", [
                        'missing' => $missingServices
                    ]);
                    continue;
                }

                // Prepare constructor arguments with proper type handling
                $constructorArgs = $this->resolveConstructorArgs($middlewareClass, $requiredServices, $config['constructor_args'] ?? []);
                $middleware = $this->createMiddleware($middlewareClass, $constructorArgs);
                
                if ($middleware) {
                    $registeredMiddlewares[] = $middleware;
                    $this->logger->middleware("Successfully registered middleware: " . get_class($middleware));
                }
            } catch (\Throwable $e) {
                $this->logger->error("Middleware registration error", [
                    'middleware' => $middlewareClass,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return $registeredMiddlewares;
    }

    /**
     * Resolves constructor arguments with proper type handling
     * 
     * @param string $middlewareClass The middleware class
     * @param array $requiredServices Required services to inject
     * @param array $extraArgs Additional constructor arguments
     * @return array Fully resolved constructor arguments
     */
    private function resolveConstructorArgs(string $middlewareClass, array $requiredServices, array $extraArgs = []): array
    {
        $constructorArgs = [];
        
        // Prepare service arguments
        foreach ($requiredServices as $service) {
            switch ($service) {
                case 'container':
                    $constructorArgs[] = $this->container;
                    break;
                case 'logger':
                    $constructorArgs[] = $this->logger;
                    break;
                default:
                    // Check if service exists before retrieving it
                    if ($this->container->has($service)) {
                        $constructorArgs[] = $this->container->get($service);
                    } else {
                        $this->logger->middleware("Warning: Required service '$service' not found in container");
                        $constructorArgs[] = null;
                    }
            }
        }
        
        // Handle any explicitly defined positional arguments
        foreach ($extraArgs as $position => $value) {
            if (is_int($position)) {
                // Ensure array is large enough
                while (count($constructorArgs) <= $position) {
                    $constructorArgs[] = null;
                }
                $constructorArgs[$position] = $value;
            }
        }
        
        // Use reflection to analyze constructor parameters and ensure proper type compatibility
        try {
            $reflectionClass = new \ReflectionClass($middlewareClass);
            $constructor = $reflectionClass->getConstructor();
            
            if ($constructor) {
                $parameters = $constructor->getParameters();
                
                // Log parameter details for diagnostics
                $paramDetails = [];
                foreach ($parameters as $i => $param) {
                    $paramType = $param->getType();
                    $typeName = $paramType ? $paramType->getName() : 'mixed';
                    $allowsNull = $paramType ? $paramType->allowsNull() : true;
                    
                    $paramDetails[] = [
                        'position' => $i,
                        'name' => $param->getName(),
                        'type' => $typeName,
                        'allows_null' => $allowsNull,
                        'optional' => $param->isOptional(),
                        'default_value' => $param->isOptional() ? 
                            ($param->isDefaultValueAvailable() ? 'available' : 'not available') : 'n/a'
                    ];
                }
                
                $this->logger->middleware("Analyzing constructor parameters for $middlewareClass", [
                    'parameters' => $paramDetails,
                    'provided_args' => count($constructorArgs)
                ]);
                
                // Check and fill missing arguments with type-safe defaults
                for ($i = 0; $i < count($parameters); $i++) {
                    $param = $parameters[$i];
                    $paramType = $param->getType();
                    
                    // Skip if argument already exists and is not null when null is not allowed
                    if (isset($constructorArgs[$i]) && ($constructorArgs[$i] !== null || !$paramType || $paramType->allowsNull())) {
                        continue;
                    }
                    
                    // For missing or null arguments, provide type-safe defaults
                    if (!isset($constructorArgs[$i]) || ($constructorArgs[$i] === null && $paramType && !$paramType->allowsNull())) {
                        if ($param->isDefaultValueAvailable()) {
                            // Use parameter's default value if available
                            $constructorArgs[$i] = $param->getDefaultValue();
                        } elseif ($paramType && !$paramType->allowsNull()) {
                            // Provide type-specific defaults for non-nullable types
                            $typeName = $paramType->getName();
                            $constructorArgs[$i] = match($typeName) {
                                'bool' => false,
                                'int' => 0,
                                'float' => 0.0,
                                'string' => '',
                                'array' => [],
                                default => null // For classes and other types
                            };
                            
                            $this->logger->middleware("Applied type-safe default for parameter", [
                                'middleware' => $middlewareClass,
                                'parameter' => $param->getName(),
                                'position' => $i,
                                'type' => $typeName,
                                'default_value' => $constructorArgs[$i]
                            ]);
                        }
                    }
                }
            }
        } catch (\ReflectionException $e) {
            $this->logger->error("Failed to analyze constructor for $middlewareClass", [
                'error' => $e->getMessage()
            ]);
        }
        
        return $constructorArgs;
    }

    /**
     * Creates a middleware instance with proper error handling
     * 
     * @param string $middlewareClass Middleware class name
     * @param array $constructorArgs Constructor arguments
     * @return MiddlewareInterface|null Created middleware or null on failure
     */
    private function createMiddleware(string $middlewareClass, array $constructorArgs)
    {
        $serviceId = lcfirst(substr($middlewareClass, strrpos($middlewareClass, '\\') + 1));
        
        if ($this->container->has($serviceId)) {
            return $this->container->get($serviceId);
        }

        try {
            $reflectionClass = new \ReflectionClass($middlewareClass);
            
            // Handle special cases for specific middleware types
            $this->handleSpecialMiddlewareRequirements($middlewareClass, $constructorArgs);
            
            // Log the final set of constructor arguments
            $this->logger->middleware("Creating middleware with arguments", [
                'class' => $middlewareClass,
                'arg_count' => count($constructorArgs),
                'arg_types' => array_map(function($arg) {
                    return is_object($arg) ? get_class($arg) : gettype($arg);
                }, $constructorArgs)
            ]);
            
            // Create the middleware instance
            $middleware = $reflectionClass->newInstanceArgs($constructorArgs);
            
            // Register in container
            $this->container->register($serviceId, $middleware);
            
            return $middleware;
        } catch (\Throwable $e) {
            $this->logger->error("Failed to create middleware: $middlewareClass", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }
    
    /**
     * Handles special requirements for specific middleware types
     * 
     * @param string $middlewareClass Middleware class name
     * @param array &$constructorArgs Constructor arguments (passed by reference)
     * @return void
     */
    private function handleSpecialMiddlewareRequirements(string $middlewareClass, array &$constructorArgs): void
    {
        // Session middleware - ensure lazyMode is boolean
        if (strpos($middlewareClass, 'SessionMiddleware') !== false) {
            // Find the lazyMode parameter (typically position 2)
            if (isset($constructorArgs[2]) && $constructorArgs[2] === null) {
                $constructorArgs[2] = false; // Default to false if null
                $this->logger->middleware("Setting default lazyMode=false for SessionMiddleware");
            }
        }
        
        // CSRF middleware - ensure failedRedirectUrl is string
        elseif (strpos($middlewareClass, 'CsrfMiddleware') !== false) {
            // Find the failedRedirectUrl parameter (typically position 2)
            if (isset($constructorArgs[2]) && $constructorArgs[2] === null) {
                $constructorArgs[2] = ''; // Default to empty string if null
                $this->logger->middleware("Setting default failedRedirectUrl='' for CsrfMiddleware");
            }
        }
    }

    /**
     * Normalizira ime klase radi dosljedne usporedbe
     * 
     * @param string $className Ime klase za normalizaciju
     * @return string Normalizirano ime klase
     */
    private function normalizeClassName(string $className): string
    {
        // Dodajte namespace ako nema
        if (strpos($className, '\\') === false) {
            $className = "\\baseKRIZAN\\Http\\Middleware\\{$className}";
        }
        
        // Osigurajte da uvijek počinje s \
        if ($className[0] !== '\\') {
            $className = '\\' . $className;
        }
        
        return $className;
    }

    /**
     * Provjerava je li middleware već registriran
     * 
     * @param string $middlewareClass Ime klase middleware-a
     * @param string $group Grupa middleware-a
     * @return bool True ako je middleware već registriran
     */
    public function isMiddlewareRegistered(string $middlewareClass, string $group = 'global'): bool
    {
        $normalizedClass = $this->normalizeClassName($middlewareClass);
        
        // Provjeri sve grupe
        foreach ($this->middlewareDefinitions as $definedGroup => $middlewares) {
            // Ako je grupa specificirana, provjeri samo tu grupu
            if ($group !== null && $group !== $definedGroup) {
                continue;
            }
            
            foreach ($middlewares as $class => $config) {
                if ($this->normalizeClassName($class) === $normalizedClass) {
                    return true;
                }
            }
        }
        
        return false;
    }


    public function addMiddleware(string $middlewareClass, array $config): void
    {
        $group = $config['group'] ?? 'global';
        $normalizedClass = $this->normalizeClassName($middlewareClass);
        
        // Provjeri je li middleware već registriran
        if ($this->isMiddlewareRegistered($normalizedClass, $group)) {
            $this->logger->middleware("Middleware already registered, skipping: $normalizedClass", [
                'group' => $group
            ]);
            return;
        }
        
        // Dodajemo automatsko prepoznavanje potrebnih servisa
        if (!isset($config['requiredServices'])) {
            $config['requiredServices'] = [];
        }
    
        // Automatski dodajemo logger svim middleware-ima
        if (!in_array('logger', $config['requiredServices'])) {
            $config['requiredServices'][] = 'logger';
        }
        
        // Automatski dodajemo logger svim middleware-ima
        if (!in_array('logger', $config['requiredServices'])) {
            $config['requiredServices'][] = 'logger';
        }
        
        // Dodajemo specifične servise za određene middleware-e
        if (strpos($middlewareClass, 'SessionMiddleware') !== false && !in_array('sessionManager', $config['requiredServices'])) {
            $config['requiredServices'][] = 'sessionManager';
        }
        
        if (strpos($middlewareClass, 'RateLimiterMiddleware') !== false && !in_array('templateRenderer', $config['requiredServices'])) {
            $config['requiredServices'][] = 'templateRenderer';
        }
        
        if (strpos($middlewareClass, 'SanitizationMiddleware') !== false && !in_array('container', $config['requiredServices'])) {
            array_unshift($config['requiredServices'], 'container'); // Dodajemo na početak jer je prvi parametar
        }
        
        if (strpos($middlewareClass, 'CsrfMiddleware') !== false) {
            // Osiguraj da je sessionManager prvi u listi, a ne csrfProtection
            if (!in_array('sessionManager', $config['requiredServices'])) {
                array_unshift($config['requiredServices'], 'sessionManager');
            } else {
                // Ako već postoji, presloži da bude prvi
                $key = array_search('sessionManager', $config['requiredServices']);
                if ($key !== 0) {
                    unset($config['requiredServices'][$key]);
                    array_unshift($config['requiredServices'], 'sessionManager');
                }
            }
            
            // Izbaci csrfProtection iz liste ako postoji
            $csrfKey = array_search('csrfProtection', $config['requiredServices']);
            if ($csrfKey !== false) {
                unset($config['requiredServices'][$csrfKey]);
            }
        }
        
        $this->middlewareDefinitions[$group][$middlewareClass] = $config;
        
        // Log registraciju
        $this->logger->middleware("Middleware registered", [
            'class' => $normalizedClass,
            'group' => $group,
            'priority' => $config['priority'] ?? 100,
            'required_services' => $config['requiredServices']
        ]);
    }
    
    /**
     * Metoda za ispis svih registriranih middleware-ova
     * 
     * @return array Definirani middleware-ovi po grupama
     */
    public function listMiddlewares(): array
    {
        $result = [];
        
        foreach ($this->middlewareDefinitions as $group => $middlewares) {
            $result[$group] = [];
            
            foreach ($middlewares as $class => $config) {
                $result[$group][] = [
                    'class' => $class,
                    'priority' => $config['priority'] ?? 100,
                    'services' => $config['requiredServices'] ?? []
                ];
            }
        }
        
        return $result;
    }
    
    /**
     * Dohvaća konfiguraciju za određeni middleware
     * 
     * @param string $middlewareClass Puno ime klase middleware-a
     * @return array|null Konfiguracija ili null ako ne postoji
     */
    public function getMiddlewareConfig(string $middlewareClass): ?array
    {
        foreach ($this->middlewareDefinitions as $group => $middlewares) {
            if (isset($middlewares[$middlewareClass])) {
                return $middlewares[$middlewareClass];
            }
        }
        
        return null;
    }
}