<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Security\RateLimiter;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Template\TemplateRenderer;
use baseKRIZAN\Http\RequestClassifier;
use baseKRIZAN\Cache\CacheInterface;

/**
 * Middleware for rate limiting requests
 */
class RateLimiterMiddleware extends Middleware
{
    /**
     * Rate limiter instance
     */
    private RateLimiter $rateLimiter;

    /**
     * Constructor
     */
    public function __construct(
        TemplateRenderer $templateRenderer,
        ?Logger $logger = null,
        ?RequestClassifier $requestClassifier = null,
        ?CacheInterface $cache = null,
        ?\PDO $pdo = null
    ) {
        parent::__construct($logger);
        
        // Set lower priority, run after security-critical middleware
        $this->priority = 60;
        
        // Initialize Rate Limiter
        $this->rateLimiter = new RateLimiter(
            $templateRenderer,
            $logger,
            $requestClassifier,
            $cache,
            $pdo
        );
    }

    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response 
    {
        // Periodic cleanup (1% chance per request)
        if (rand(1, 100) === 1) {
            $this->rateLimiter->cleanOldData();
        }

        // Check if route is whitelisted
        $route = $this->rateLimiter->resolveRoute($request);
        if (in_array($route, $this->rateLimiter->getWhitelistedRoutes())) {
            return $next($request);
        }

        // Determine if this is a significant route for logging purposes
        $isSignificantRoute = $this->rateLimiter->shouldRateLimit($route);
        
        // Log request details for significant routes
        if (($isSignificantRoute || $request->getMethod() === 'POST') && $this->logger) {
            $this->logger->middleware('RateLimiterMiddleware: Request analysis', [
                'resolved_route' => $route,
                'method' => $request->getMethod(),
                'ip' => $request->getIp(),
                'email' => $request->getPost('email') ?? 'N/A',
            ]);
        }
        
        // Check for IP rotation attack patterns
        if ($this->rateLimiter->detectIpRotation($request)) {
            $this->rateLimiter->logIpRotationBlocked($request);
            $blockMinutes = $this->rateLimiter->getIpRotationDecay() / 60; // Convert to minutes
            return $this->rateLimiter->buildRateLimitResponse($blockMinutes);
        }
        
        // Apply rate limiting for protected routes with POST method
        if ($this->rateLimiter->shouldRateLimit($route) && $request->getMethod() === 'POST') {
            $key = $this->rateLimiter->getRequestSignature($request, $route);
            $limits = $this->rateLimiter->getProtectedRouteConfig($route);
            
            if ($this->rateLimiter->tooManyAttempts($key, $limits['max'], $limits['window'], $request, $route)) {
                $this->rateLimiter->logBlockedAttempt($request, $route);
                $retryAfter = $this->rateLimiter->getRetryAfterTime($key);
                return $this->rateLimiter->buildRateLimitResponse($retryAfter / 60); // Convert to minutes
            }
            
            $this->rateLimiter->addAttempt($key, $limits['window'], $request, $route);
            
            // Track IP addresses for identifiers (usernames/emails)
            $identifier = $this->rateLimiter->getIdentifier($request);
            if ($identifier) {
                $this->rateLimiter->trackIpForIdentifier($identifier, $request);
            }
        }
        
        return $next($request);
    }
    
    /**
     * Get the rate limiter instance
     */
    public function getRateLimiter(): RateLimiter
    {
        return $this->rateLimiter;
    }
}