<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Error\Logger;

/**
 * Middleware for input data sanitization
 */
class SanitizationMiddleware extends Middleware
{
    /**
     * Sanitization rules by type
     */
    private array $sanitizationRules = [
        'email' => FILTER_SANITIZE_EMAIL,
        'url' => FILTER_SANITIZE_URL,
        'int' => FILTER_SANITIZE_NUMBER_INT,
        'float' => [
            'filter' => FILTER_SANITIZE_NUMBER_FLOAT,
            'flags' => FILTER_FLAG_ALLOW_FRACTION
        ],
        'html' => [
            'filter' => FILTER_SANITIZE_SPECIAL_CHARS,
            'flags' => FILTER_FLAG_ENCODE_HIGH
        ]
    ];

    /**
     * Default field types based on field names
     */
    private array $defaultFieldTypes = [
        'email' => 'email',
        'password' => 'string',
        'username' => 'string',
        'name' => 'string',
        'description' => 'html',
        'url' => 'url',
        'id' => 'int',
        'amount' => 'float',
        'price' => 'float'
    ];

    /**
     * Container instance
     */
    private ?Container $container;

    /**
     * Constructor
     */
    public function __construct(?Container $container = null, ?Logger $logger = null)
    {
        parent::__construct($logger);
        $this->container = $container;
        $this->priority = 35; // Medium priority
    }

    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response
    {
        // Sanitize input data
        $this->sanitizeRequestData($request);
        
        // Set validator in request attributes if available in container
        if ($this->container?->has('validator')) {
            $validator = $this->container->get('validator');
            $request->setAttribute('validator', $validator);
        }
        
        return $next($request);
    }

    /**
     * Sanitize request data
     */
    private function sanitizeRequestData(Request $request): void
    {
        // Sanitize POST data
        $post = $request->getPost();
        if (!empty($post)) {
            $sanitizedPost = $this->sanitizeData($post);
            $request->setPost($sanitizedPost);
        }

        // Sanitize GET data
        $query = $request->getQuery();
        if (!empty($query)) {
            $sanitizedQuery = $this->sanitizeData($query);
            $request->setQuery($sanitizedQuery);
        }
    }

    /**
     * Sanitize data recursively
     */
    private function sanitizeData(array $data): array
    {
        $sanitized = [];
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $sanitized[$key] = $this->sanitizeData($value);
            } else {
                $type = $this->determineFieldType($key);
                $sanitized[$key] = $this->applySanitizationRule($value, $type);
            }
        }
        return $sanitized;
    }

    /**
     * Determine field type based on name
     */
    private function determineFieldType(string $key): string
    {
        // Check if we have a predefined type for this field name
        if (isset($this->defaultFieldTypes[$key])) {
            return $this->defaultFieldTypes[$key];
        }

        // Check if the field name contains a type indicator
        foreach ($this->defaultFieldTypes as $indicator => $type) {
            if (stripos($key, $indicator) !== false) {
                return $type;
            }
        }

        // Default to basic string sanitization
        return 'string';
    }

    /**
     * Apply sanitization rule based on field type
     * 
     * @param mixed $value Value to sanitize
     * @param string $type Field type
     * @return mixed Sanitized value
     */
    private function applySanitizationRule($value, string $type)
    {
        if ($value === null || $value === '') {
            return $value;
        }

        // For strings, use htmlspecialchars
        if ($type === 'string') {
            return $this->sanitizeString($value);
        }

        $rule = $this->sanitizationRules[$type] ?? null;
        if (!$rule) {
            return $this->sanitizeString($value);
        }

        if (is_array($rule)) {
            return filter_var($value, $rule['filter'], $rule['flags'] ?? 0);
        }

        return filter_var($value, $rule);
    }

    /**
     * Sanitize string
     */
    private function sanitizeString(string $value): string
    {
        // Remove disallowed characters and HTML tags
        $value = strip_tags($value);
        
        // Convert special characters to HTML entities
        $value = htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Trim spaces from beginning and end
        return trim($value);
    }
}