<?php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Assets\ResourceLoader;

/**
 * Middleware for security HTTP headers
 */
class SecurityHeadersMiddleware extends Middleware
{
    /**
     * Security headers configuration
     */
    private array $config;
    
    /**
     * Constructor
     */
    public function __construct(?Logger $logger = null)
    {
        parent::__construct($logger);
        $this->config = \baseKRIZAN\Config\Config::get('security.headers') ?? [];
        $this->priority = 30; // Medium priority
    }

    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response 
    {
        // Inicijaliziraj ResourceLoader ako još nije, da se generira nonce
        // VAŽNO: Ovo mora biti prije obrade zahtjeva kako bi CSP header koristio isti nonce
        if (method_exists(ResourceLoader::class, 'init') && !ResourceLoader::isInitialized()) {
            ResourceLoader::init();
        }
        
        // Zamijeni {NONCE} placeholdere u CSP konfiguraciji
        if (method_exists(ResourceLoader::class, 'getNonce')) {
            $nonce = ResourceLoader::getNonce();
            if ($this->logger) {
                $this->logger->middleware("Using nonce from ResourceLoader: " . $nonce);
            }
            $this->replaceNoncePlaceholders($nonce);
        }
        
        // Zatim obradi zahtjev
        $response = $next($request);
        
        // Skip if security headers are disabled
        if (!($this->config['enabled'] ?? true)) {
            return $response;
        }
        
        // Optimize headers based on request type
        $requestType = $request->getAttribute('request_type', 'web');
        
        match ($requestType) {
            'asset' => $this->setAssetHeaders($response),
            'api' => $this->setApiHeaders($response),
            default => $this->setWebHeaders($response)
        };
        
        if ($this->logger) {
            $this->logger->middleware('Security headers applied for ' . $requestType);
        }
        
        return $response;
    }
    
    /**
     * Zamjenjuje {NONCE} placeholder u CSP direktivama s pravom nonce vrijednosti
     */
    private function replaceNoncePlaceholders(string $nonce): void
    {
        // Dohvati direktno iz Config
        $cspDirectives = [
            'CSP_DEFAULT_SRC', 
            'CSP_SCRIPT_SRC', 
            'CSP_STYLE_SRC',
            'CSP_IMG_SRC',
            'CSP_FONT_SRC',
            'CSP_CONNECT_SRC',
            'CSP_FRAME_SRC',
            'CSP_FRAME_ANCESTORS',
            'CSP_FORM_ACTION'
        ];
        
        foreach ($cspDirectives as $directive) {
            $value = \baseKRIZAN\Config\Config::get($directive);
            
            if (is_string($value)) {
                $newValue = str_replace('{NONCE}', $nonce, $value);
                \baseKRIZAN\Config\Config::set($directive, $newValue);
                
                if ($this->logger) {
                    $this->logger->middleware("Replaced nonce in $directive: $newValue");
                }
            }
        }
    }
    
    /**
     * Set headers for web requests
     */
    private function setWebHeaders(Response $response): void
    {
        // Standard security headers
        $response->setHeader('X-Content-Type-Options', 'nosniff');
        $response->setHeader('X-Frame-Options', 'SAMEORIGIN');
        $response->setHeader('X-XSS-Protection', '1; mode=block');
        $response->setHeader('Referrer-Policy', 'strict-origin-when-cross-origin');
        
        // Add HSTS header for HTTPS connections
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') {
            $response->setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
        }
        
        // Add Content-Security-Policy header
        $cspDirectives = $this->getCSPDirectives();
        $response->setHeader('Content-Security-Policy', $cspDirectives);
        
        if ($this->logger) {
            $this->logger->middleware("Set CSP header: $cspDirectives");
        }
        
        // Add Permissions-Policy header
        $response->setHeader('Permissions-Policy', \baseKRIZAN\Config\Config::get('PERMISSIONS_POLICY') ?? 
            $this->config['permissions_policy'] ?? $this->getDefaultPermissionsPolicy());
    }
    
    /**
     * Set headers for API requests
     */
    private function setApiHeaders(Response $response): void
    {
        // Basic security headers
        $response->setHeader('X-Content-Type-Options', 'nosniff');
        
        // CORS headers for API
        $response->setHeader('Access-Control-Allow-Origin', $this->config['api']['allow_origin'] ?? '*');
        $response->setHeader('Access-Control-Allow-Methods', $this->config['api']['allow_methods'] ?? 'GET, POST, PUT, DELETE, OPTIONS');
        $response->setHeader('Access-Control-Allow-Headers', $this->config['api']['allow_headers'] ?? 'Content-Type, Authorization, X-Requested-With');
        
        // HSTS for API
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') {
            $response->setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
        }
    }
    
    /**
     * Set headers for asset requests
     */
    private function setAssetHeaders(Response $response): void
    {
        // Minimal headers for assets
        $response->setHeader('X-Content-Type-Options', 'nosniff');
        
        // Cache headers for assets
        if (!$response->hasHeader('Cache-Control')) {
            $response->setHeader('Cache-Control', 'public, max-age=86400'); // 1 day
        }
    }
    
    /**
     * Get CSP directives - prioritetno koristi direktne vrijednosti iz .env
     */
    private function getCSPDirectives(): string
    {
        // Prioritetno koristi direktne CSP postavke iz .env kroz Config
        if (\baseKRIZAN\Config\Config::get('CSP_DEFAULT_SRC')) {
            return $this->getCSPDirectivesFromConfig();
        }
        
        // Ako direktne CSP postavke nisu prisutne, koristi postavke iz $this->config
        $cspConfig = $this->config['csp'] ?? [];
        $basePath = \baseKRIZAN\Config\Config::get('paths.app_url') ?? '';
        $reportUri = isset($this->config['csp_report_uri']) && $this->config['csp_report_uri'] ?
            "report-uri " . $basePath . $this->config['csp_report_uri'] : '';
            
        return implode('; ', array_filter([
            "default-src " . ($cspConfig['default_src'] ?? "'self'"),
            "script-src " . ($cspConfig['script_src'] ?? "'self'"),
            "style-src " . ($cspConfig['style_src'] ?? "'self'"),
            "img-src " . ($cspConfig['img_src'] ?? "'self'"),
            "font-src " . ($cspConfig['font_src'] ?? "'self'"),
            "connect-src " . ($cspConfig['connect_src'] ?? "'self'"),
            "frame-src " . ($cspConfig['frame_src'] ?? "'self'"),
            "frame-ancestors " . ($cspConfig['frame_ancestors'] ?? "'self'"),
            "form-action " . ($cspConfig['form_action'] ?? "'self'"),
            $reportUri
        ]));
    }
    
    /**
     * Get CSP directives directly from config (.env)
     */
    private function getCSPDirectivesFromConfig(): string
    {
        $directives = [];
        
        // Map config keys to CSP directive names
        $cspMappings = [
            'CSP_DEFAULT_SRC' => 'default-src',
            'CSP_SCRIPT_SRC' => 'script-src',
            'CSP_STYLE_SRC' => 'style-src',
            'CSP_IMG_SRC' => 'img-src',
            'CSP_FONT_SRC' => 'font-src',
            'CSP_CONNECT_SRC' => 'connect-src',
            'CSP_FRAME_SRC' => 'frame-src',
            'CSP_FRAME_ANCESTORS' => 'frame-ancestors',
            'CSP_FORM_ACTION' => 'form-action'
        ];
        
        foreach ($cspMappings as $configKey => $directive) {
            $value = \baseKRIZAN\Config\Config::get($configKey);
            if ($value) {
                $directives[] = "$directive $value";
            }
        }
        
        return implode('; ', $directives);
    }
    
    /**
     * Get default permissions policy
     */
    private function getDefaultPermissionsPolicy(): string
    {
        return "geolocation=(), microphone=(), camera=(), payment=(), usb=(), accelerometer=(), gyroscope=(), magnetometer=(), midi=(), notifications=self, push=self, sync-xhr=self, interest-cohort=()";
    }
}