<?php
// app/baseKRIZAN/Http/Middleware/SessionMiddleware.php

namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

/**
 * Middleware za inicijalizaciju i upravljanje sesijom
 */
class SessionMiddleware extends Middleware
{
    /**
     * @var SessionManager
     */
    private SessionManager $sessionManager;
    
    /**
     * @var bool
     */
    private bool $lazyMode;
    
    /**
     * Konstruktor
     * 
     * @param SessionManager $sessionManager Session manager
     * @param Logger|null $logger Logger
     * @param bool $lazyMode True za lazy inicijalizaciju sesije
     */
    public function __construct(
        SessionManager $sessionManager, 
        ?Logger $logger = null,
        bool $lazyMode = false
    ) {
        parent::__construct($logger);
        $this->sessionManager = $sessionManager;
        $this->lazyMode = $lazyMode;
        $this->priority = 10; // Visok prioritet - izvršava se rano
        
        if ($this->logger) {
            $this->logger->middleware("SessionMiddleware created with lazyMode: " . ($lazyMode ? 'true' : 'false'));
        }
    }
 
    /**
     * @inheritDoc
     */
    public function process(Request $request, callable $next): Response
    {
        $requestType = $request->getAttribute('request_type', '');
        
        if ($this->logger) {
            $this->logger->middleware("SessionMiddleware processing request", [
                'path' => $request->getPath(),
                'request_type' => $requestType,
                'lazy_mode' => $this->lazyMode
            ]);
        }
        
        // Za asset zahtjeve možemo potpuno preskočiti pokretanje sesije
        if ($requestType === 'asset' && $this->lazyMode) {
            if ($this->logger) {
                $this->logger->middleware('Session: Skipping for asset request');
            }
            return $next($request);
        }
        
        // Za API zahtjeve možemo preskočiti sesiju ako nije potrebna autentikacija
        if ($requestType === 'api' && $this->lazyMode && 
            !$request->getAttribute('requires_auth', false)) {
            if ($this->logger) {
                $this->logger->middleware('Session: Skipping for stateless API request');
            }
            return $next($request);
        }
        
        // Only start if not already started
        if (!$this->sessionManager->isStarted()) {
            if ($this->logger) {
                $this->logger->middleware('Session: Starting new session');
            }
            $this->sessionManager->start();
        }
        
        // Dodaj session manager u atribute zahtjeva za kasnije korištenje
        $request->setAttribute('session_manager', $this->sessionManager);
        
        // Execute next middleware in chain
        $response = $next($request);
        
        return $response;
    }
    
    /**
     * @inheritDoc
     */
    public function terminate(Request $request, Response $response): void
    {
        // Zatvori sesiju na kraju zahtjeva ako je pokrenuta
        if ($this->sessionManager->isStarted() && session_status() === PHP_SESSION_ACTIVE) {
            if ($this->logger) {
                $this->logger->middleware('Session: Closing active session');
            }
            session_write_close();
        }
    }
}