<?php
// app/baseKRIZAN/Http/Request.php

namespace baseKRIZAN\Http;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Services\Container;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Implementacija HTTP zahtjeva s PSR-7 kompatibilnošću
 * 
 * @package baseKRIZAN\Http
 */
class Request implements ServerRequestInterface
{
    private static ?Request $currentInstance = null;
    private string $path;
    private string $uri;
    private string $method;
    private array $query;
    private array $post;
    private array $attributes = [];
    private array $headers;
    private array $files;
    private bool $isAjax;
    private array $params = [];
    private ?Logger $logger = null;
    private ?Container $container = null;
    private bool $hasLoggedCreation = false;
    
    // PSR-7 svojstva
    private string $protocolVersion = '1.1';
    private ?string $requestTarget = null;
    private array $serverParams = [];
    private array $cookieParams = [];
    private $body = null;
    private array $uploadedFiles = [];

    /**
     * Konstruktor za Request objekt
     * 
     * Inicijalizira zahtjev s trenutnim parametrima ili zadanim vrijednostima
     */
    public function __construct()
    {
        $this->serverParams = $_SERVER;
        $this->cookieParams = $_COOKIE;
        $this->path = trim($_SERVER['PATH_INFO'] ?? '/', '/');
        $this->method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $this->query = $this->sanitizeInput($_GET);
        $this->post = $this->sanitizeInput($_POST);
        $this->attributes = [];
        $this->files = $_FILES;
        $this->headers = $this->getHeadersFromServer();
        $this->isAjax = isset($this->headers['X-Requested-With']) && 
                       $this->headers['X-Requested-With'] === 'XMLHttpRequest';

        // Postavi HTTP protokol
        if (isset($_SERVER['SERVER_PROTOCOL'])) {
            $this->protocolVersion = str_replace('HTTP/', '', $_SERVER['SERVER_PROTOCOL']);
        }

        // Učitaj JSON podatke ako je potrebno
        if ($this->isJsonRequest()) {
            $jsonData = $this->getJsonInput();
            if ($jsonData !== null) {
                $this->post = $this->sanitizeInput($jsonData);
            }
        }
        
        // Inicijaliziraj URI
        $this->uri = $this->buildUri();
        
        // Normaliziraj uploadane datoteke
        $this->uploadedFiles = $this->normalizeUploadedFiles($this->files);
        
        // Postavi trenutnu instancu
        self::$currentInstance = $this;
    }
    
    /**
     * Postavlja logger instancu
     *
     * @param Logger|null $logger Logger instanca
     * @return void
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
        
        // Logiraj kreiranje zahtjeva ako je logger dostupan i ako još nije logirano
        if ($this->logger && !$this->hasLoggedCreation) {
            // Provjeri je li ovo primarni Request objekt (iz EntryPoint) ili se stvara negdje drugdje
            $isPrimary = $this->getAttribute('primary_request', false);
            
            // Samo logiraj za primarne zahtjeve ili ako nije postavljen atribut 
            // (kompatibilnost sa starim kodom)
            if ($isPrimary || !isset($this->attributes['primary_request'])) {
                $this->logger->http('Request object created', [
                    'path' => $this->path,
                    'method' => $this->method,
                    'ajax' => $this->isAjax ? 'yes' : 'no',
                    'json' => $this->isJsonRequest() ? 'yes' : 'no',
                    'request_type' => $this->getAttribute('request_type', '')
                ]);
                
                $this->hasLoggedCreation = true;
            }
        }
    }
    
    /**
     * Dohvaća trenutnu instancu zahtjeva
     *
     * @return self|null Trenutna instanca ili null ako nije inicijalizirana
     */
    public static function getCurrent(): ?self
    {
        return self::$currentInstance;
    }

    /**
     * Dohvaća putanju zahtjeva
     *
     * @return string Putanja zahtjeva
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * Postavlja putanju zahtjeva
     *
     * @param string $path Nova putanja
     * @return void
     */
    public function setPath(string $path): void
    {
        $this->path = $path;
    }

    /**
     * {@inheritdoc}
     */
    public function getUri(): \Psr\Http\Message\UriInterface
    {
        // Trebate implementirati vlastitu implementaciju Uri klase
        // ili koristiti vanjsku implementaciju
        return new Uri($this->uri);
    }

    /**
     * Dohvaća HTTP metodu zahtjeva
     *
     * @return string HTTP metoda (GET, POST, PUT, DELETE itd.)
     */
    public function getMethod(): string
    {
        return $this->method;
    }

    /**
     * Dohvaća query parametre
     *
     * @param string|null $key Ključ za dohvat ili null za sve parametre
     * @return mixed Vrijednost parametra ili svi query parametri
     */
    public function getQuery(?string $key = null)
    {
        if ($key === null) {
            return $this->query;
        }
        return $this->query[$key] ?? null;
    }

    /**
     * Dohvaća POST parametre
     *
     * @param string|null $key Ključ za dohvat ili null za sve parametre
     * @return mixed Vrijednost parametra ili svi POST parametri
     */
    public function getPost(?string $key = null)
    {
        if ($key === null) {
            return $this->post;
        }
        return $this->post[$key] ?? null;
    }

    /**
     * Dohvaća informacije o uploadanoj datoteci
     *
     * @param string $key Ključ datoteke
     * @return array|null Informacije o datoteci ili null ako ne postoji
     */
    public function getFile(string $key)
    {
        return $this->files[$key] ?? null;
    }

    /**
     * Provjerava postoji li uploadana datoteka pod zadanim ključem
     *
     * @param string $key Ključ datoteke
     * @return bool True ako datoteka postoji i nije UPLOAD_ERR_NO_FILE
     */
    public function hasFile(string $key): bool
    {
        return isset($this->files[$key]) && 
               $this->files[$key]['error'] !== UPLOAD_ERR_NO_FILE;
    }

    /**
     * Postavlja atribut na zahtjev
     *
     * @param string $key Ključ atributa
     * @param mixed $value Vrijednost atributa
     * @return void
     */
    public function setAttribute(string $key, $value): void
    {
        $this->attributes[$key] = $value;
    }

    /**
     * {@inheritdoc}
     */
    public function getHeader(string $name): array
    {
        $value = $this->headers[$name] ?? null;
        return $value ? (is_array($value) ? $value : [$value]) : [];
    }

    /**
     * Dohvaća IP adresu klijenta
     *
     * @return string IP adresa ili "0.0.0.0" ako nije moguće utvrditi
     */
    public function getIp(): string
    {
        $ip = '';
        
        if (isset($this->headers['HTTP_CLIENT_IP'])) {
            $ip = $this->headers['HTTP_CLIENT_IP'];
        } elseif (isset($this->headers['HTTP_X_FORWARDED_FOR'])) {
            // Uzima prvu IP adresu ako ih ima više
            $ip = strpos($this->headers['HTTP_X_FORWARDED_FOR'], ',') !== false 
                ? explode(',', $this->headers['HTTP_X_FORWARDED_FOR'])[0] 
                : $this->headers['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        }

        return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '0.0.0.0';
    }

    /**
     * Provjerava je li zahtjev AJAX
     *
     * @return bool True ako je AJAX zahtjev
     */
    public function isAjax(): bool
    {
        return $this->isAjax;
    }

    /**
     * Provjerava je li zahtjev JSON
     *
     * @return bool True ako je JSON zahtjev
     */
    public function isJson(): bool
    {
        return $this->isJsonRequest();
    }

    /**
     * Dohvaća HTTP headere iz $_SERVER
     *
     * @return array Asocijativni niz headera
     */
    private function getHeadersFromServer(): array
    {
        if (function_exists('getallheaders')) {
            return getallheaders();
        }

        $headers = [];
        foreach ($_SERVER as $name => $value) {
            if (substr($name, 0, 5) === 'HTTP_') {
                $headerName = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                $headers[$headerName] = $value;
            } elseif (in_array($name, ['CONTENT_TYPE', 'CONTENT_LENGTH', 'CONTENT_MD5'])) {
                $headerName = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', $name))));
                $headers[$headerName] = $value;
            }
        }
        return $headers;
    }

    /**
     * Provjerava je li zahtjev JSON na temelju Content-Type headera
     *
     * @return bool True ako je Content-Type application/json
     */
    private function isJsonRequest(): bool
    {
        $contentTypeHeader = $this->getHeader('Content-Type');
        if (empty($contentTypeHeader)) {
            return false;
        }
        
        // Provjeri svaku vrijednost Content-Type headera
        foreach ($contentTypeHeader as $contentType) {
            if (str_contains($contentType, 'application/json')) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Dohvaća i parsira JSON podatke iz tijela zahtjeva
     *
     * @return array|null Parsirani JSON podaci ili null ako parsirarnje nije uspjelo
     */
    private function getJsonInput(): ?array
    {
        $jsonData = file_get_contents('php://input');
        if (empty($jsonData)) {
            return null;
        }

        $data = json_decode($jsonData, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            if ($this->logger) {
                $this->logger->error('JSON decode error', [
                    'error' => json_last_error_msg(),
                    'input_length' => strlen($jsonData)
                ]);
            }
            return null;
        }

        return $data;
    }

    /**
     * Sanitizira ulazne podatke
     *
     * @param array $data Podaci za sanitizaciju
     * @return array Sanitizirani podaci
     */
    private function sanitizeInput(array $data): array
    {
        $sanitized = [];
        foreach ($data as $key => $value) {
            $sanitizedKey = $this->sanitizeKey($key);
            
            if (is_array($value)) {
                $sanitized[$sanitizedKey] = $this->sanitizeInput($value);
            } else {
                $sanitized[$sanitizedKey] = $this->sanitizeValue($value);
            }
        }
        return $sanitized;
    }

    /**
     * Sanitizira putanju
     *
     * @param string $path Putanja za sanitizaciju
     * @return string Sanitizirana putanja
     */
    private function sanitizePath(string $path): string
    {
        // Ukloni višestruke kose crte
        $path = preg_replace('|[/\\\\]+|', '/', $path);
        // Ukloni '../' pokušaje
        $path = str_replace('../', '', $path);
        return htmlspecialchars($path, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Sanitizira ključ
     *
     * @param string $key Ključ za sanitizaciju
     * @return string Sanitizirani ključ
     */
    private function sanitizeKey(string $key): string
    {
        return preg_replace('/[^a-zA-Z0-9_-]/', '', $key);
    }

    /**
     * Sanitizira vrijednost
     *
     * @param mixed $value Vrijednost za sanitizaciju
     * @return string Sanitizirana vrijednost
     */
    private function sanitizeValue($value): string
    {
        if ($value === null) {
            return '';
        }
        return htmlspecialchars((string)$value, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Postavlja POST podatke
     *
     * @param array $post Novi POST podaci
     * @return void
     */
    public function setPost(array $post): void
    {
        $this->post = $post;
        
        if ($this->logger) {
            $this->logger->http('POST data set on request');
        }
    }

    /**
     * Postavlja query parametre
     *
     * @param array $query Novi query parametri
     * @return void
     */
    public function setQuery(array $query): void
    {
        $this->query = $query;
        
        if ($this->logger) {
            $this->logger->http('Query parameters set on request');
        }
    }

    /**
     * Postavlja parametre rute
     *
     * @param array $params Parametri rute
     * @return void
     */
    public function setParams(array $params): void 
    {
        $this->params = $params;
        
        if ($this->logger) {
            $this->logger->http('Route parameters set on request', [
                'params' => $params
            ]);
        }
    }
    
    /**
     * Dohvaća parametre rute
     *
     * @return array Parametri rute
     */
    public function getParams(): array 
    {
        return $this->params;
    }
    
    /**
     * Dohvaća parametar rute po imenu
     *
     * @param string $name Ime parametra
     * @param mixed $default Zadana vrijednost ako parametar ne postoji
     * @return mixed Vrijednost parametra ili zadana vrijednost
     */
    public function getParam(string $name, $default = null) 
    {
        return $this->params[$name] ?? $default;
    }

    /**
     * Dohvaća sve ulazne podatke (POST i GET)
     *
     * @return array Svi ulazni podaci
     */
    public function all(): array
    {
        return array_merge($this->getQuery(), $this->getPost());
    }
    
    /**
     * Dohvaća ulazni podatak s validacijom
     *
     * @param string $key Ključ podatka
     * @param mixed $default Zadana vrijednost ako podatak ne postoji
     * @return mixed Vrijednost podatka ili zadana vrijednost
     */
    public function input(string $key, $default = null)
    {
        return $this->getPost($key) ?? $this->getQuery($key) ?? $default;
    }
    
    /**
     * Provjerava postoji li ulazni podatak
     *
     * @param string $key Ključ podatka
     * @return bool True ako podatak postoji
     */
    public function has(string $key): bool
    {
        return isset($this->post[$key]) || isset($this->query[$key]);
    }
    
    /**
     * Dohvaća samo specificirane ključeve iz ulaznih podataka
     *
     * @param array $keys Ključevi za dohvat
     * @return array Filtrirani ulazni podaci
     */
    public function only(array $keys): array
    {
        $data = [];
        foreach ($keys as $key) {
            if ($this->has($key)) {
                $data[$key] = $this->input($key);
            }
        }
        return $data;
    }
    
    /**
     * Dohvaća sve ulazne podatke osim specificiranih ključeva
     *
     * @param array $keys Ključevi za isključivanje
     * @return array Filtrirani ulazni podaci
     */
    public function except(array $keys): array
    {
        $data = $this->all();
        foreach ($keys as $key) {
            unset($data[$key]);
        }
        return $data;
    }
    
    /**
     * Provjerava je li zahtjev za specifičnu rutu
     *
     * @param string $route Ruta za provjeru
     * @return bool True ako je zahtjev za specificiranu rutu
     */
    public function isRoute(string $route): bool
    {
        return $this->getPath() === $route || $this->getQuery('route') === $route;
    }
    
    /**
     * Provjerava je li zahtjev specifične metode
     *
     * @param string $method HTTP metoda
     * @return bool True ako je zahtjev specificirane metode
     */
    public function isMethod(string $method): bool
    {
        return strtoupper($this->method) === strtoupper($method);
    }
    
    /**
     * Provjerava prihvaća li zahtjev specifični content type
     *
     * @param string $contentType Content type za provjeru
     * @return bool True ako zahtjev prihvaća specificirani content type
     */
    public function accepts(string $contentType): bool
    {
        $acceptHeader = $this->getHeaderLine('Accept');
        return strpos($acceptHeader, $contentType) !== false || $acceptHeader === '*/*';
    }
    
    /**
     * Provjerava želi li zahtjev JSON odgovor
     *
     * @return bool True ako zahtjev želi JSON odgovor
     */
    public function wantsJson(): bool
    {
        return $this->accepts('application/json') || $this->isAjax();
    }
    
    /**
     * Dohvaća URL zahtjeva
     *
     * @return string Puni URL zahtjeva
     */
    public function url(): string
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $uri = $_SERVER['REQUEST_URI'] ?? '/';
        return "$protocol://$host$uri";
    }

    /**
     * {@inheritdoc}
     */
    public function withUri(\Psr\Http\Message\UriInterface $uri, bool $preserveHost = false): \Psr\Http\Message\RequestInterface
    {
        $new = clone $this;
        
        // Uri je immutable objekt, pa samo postavljamo novi uri
        if ($preserveHost && $this->hasHeader('Host')) {
            // Ako treba sačuvati Host header, ne diramo ga
            return $new;
        }
        
        // Inače postavljamo Host header iz URI-ja
        if ($uri->getHost() !== '') {
            $host = $uri->getHost();
            if ($uri->getPort() !== null) {
                $host .= ':' . $uri->getPort();
            }
            
            // Ako postoji metoda za postavljanje pojedinačnog headera, koristi nju
            if (method_exists($new, 'setHeader')) {
                $new->setHeader('Host', $host);
            } else {
                // Inače koristimo withHeader metodu
                $new = $new->withHeader('Host', $host);
            }
        }
        
        return $new;
    }
    
    /**
     * Dohvaća bazni URL
     *
     * @return string Bazni URL (protokol + host)
     */
    public function baseUrl(): string
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return "$protocol://$host";
    }
    
    /**
     * Dohvaća User-Agent
     *
     * @return string User-Agent header
     */
    public function userAgent(): string
    {
        $acceptHeader = $this->getHeaderLine('User-Agent');
        return $acceptHeader;
    }

    /**
     * Postavlja DI container
     * 
     * @param Container|null $container Container instanca
     * @return void
     */
    public function setContainer(?Container $container): void
    {
        $this->container = $container;
    }

    /**
     * Dohvaća DI container
     * 
     * @return Container|null Container instanca
     */
    public function getContainer(): ?Container
    {
        return $this->container;
    }
    
    /**
     * Validira zahtjev prema pravilima
     * 
     * @param array $rules Pravila validacije
     * @param array $messages Prilagođene poruke grešaka
     * @return \baseKRIZAN\Validation\ValidationResult Rezultat validacije
     * @throws \RuntimeException Ako validator nije dostupan
     */
    public function validate(array $rules, array $messages = []): \baseKRIZAN\Validation\ValidationResult 
    {
        // Prvo pokušaj dohvatiti validator iz atributa zahtjeva
        $validator = $this->getAttribute('validator');
        
        // Ako validator nije u atributima, pokušaj dohvatiti iz containera priloženog zahtjevu
        if (!$validator && $this->container && $this->container->has('validator')) {
            $validator = $this->container->get('validator');
            // Spremi za buduću upotrebu
            $this->setAttribute('validator', $validator);
        }
        
        // Ako još uvijek nema validatora, pokušaj dohvatiti globalni container
        if (!$validator) {
            // Dohvati container iz Bootstrap instance
            $container = null;
            if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
                $bootstrap = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
                if ($bootstrap && method_exists($bootstrap, 'getContainer')) {
                    $container = $bootstrap->getContainer();
                }
            }
            
            if ($container && $container->has('validator')) {
                $validator = $container->get('validator');
                // Spremi za buduću upotrebu
                $this->setAttribute('validator', $validator);
            }
        }
        
        // Ako još uvijek nema validatora, baci iznimku
        if (!$validator) {
            throw new \RuntimeException('Validator service not available in request. Make sure validator is registered in container and ValidationInitializer is added to bootstrap process.');
        }
        
        // Logiraj pokušaj validacije ako je logger dostupan
        if ($this->logger) {
            $this->logger->http('Validating request data', [
                'rules_count' => count($rules),
                'path' => $this->getPath()
            ]);
        }
        
        // Izvrši validaciju
        return $validator->validate($this, $rules, $messages);
    }

    /**
     * Gradi URI za zahtjev
     * 
     * @return string URI
     */
    private function buildUri(): string
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $uri = $_SERVER['REQUEST_URI'] ?? '/';
        
        return "$protocol://$host$uri";
    }

    /**
     * Normalizira uploadane datoteke u standardni format
     * 
     * @param array $files Array uploadanih datoteka
     * @return array Normalizirani array
     */
    private function normalizeUploadedFiles(array $files): array
    {
        $normalized = [];
        
        foreach ($files as $key => $file) {
            if (isset($file['name'])) {
                if (is_array($file['name'])) {
                    $normalized[$key] = $this->normalizeNestedFiles($file);
                } else {
                    $normalized[$key] = $file;
                }
            }
        }
        
        return $normalized;
    }

    /**
     * Normalizira ugnježđene uploadane datoteke
     * 
     * @param array $file Array s ugnježđenim datotekama
     * @return array Normalizirani array
     */
    private function normalizeNestedFiles(array $file): array
    {
        $normalized = [];
        
        foreach (array_keys($file['name']) as $key) {
            $normalized[$key] = [
                'name' => $file['name'][$key],
                'type' => $file['type'][$key],
                'tmp_name' => $file['tmp_name'][$key],
                'error' => $file['error'][$key],
                'size' => $file['size'][$key],
            ];
        }
        
        return $normalized;
    }

    // PSR-7 ServerRequestInterface implementacija

    /**
     * {@inheritdoc}
     */
    public function getProtocolVersion(): string
    {
        return $this->protocolVersion;
    }

    /**
     * {@inheritdoc}
     */
    public function withProtocolVersion($version): \Psr\Http\Message\MessageInterface
    {
        $new = clone $this;
        $new->protocolVersion = $version;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getHeaders(): array
    {
        return $this->headers;
    }

    /**
     * {@inheritdoc}
     */
    public function hasHeader($name): bool
    {
        foreach ($this->headers as $headerName => $value) {
            if (strtolower($headerName) === strtolower($name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function getHeaderLine($name): string
    {
        $values = $this->getHeader($name);
        if (empty($values)) {
            return '';
        }
        return implode(', ', $values);
    }

    /**
     * {@inheritdoc}
     */
    public function withHeader($name, $value): \Psr\Http\Message\MessageInterface
    {
        $new = clone $this;
        $new->headers[$name] = is_array($value) ? $value : [$value];
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withAddedHeader($name, $value): \Psr\Http\Message\MessageInterface
    {
        $new = clone $this;
        if (isset($new->headers[$name])) {
            if (is_array($new->headers[$name])) {
                $new->headers[$name][] = $value;
            } else {
                $new->headers[$name] = [$new->headers[$name], $value];
            }
        } else {
            $new->headers[$name] = [$value];
        }
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withoutHeader($name): \Psr\Http\Message\MessageInterface
    {
        $new = clone $this;
        foreach ($new->headers as $headerName => $value) {
            if (strtolower($headerName) === strtolower($name)) {
                unset($new->headers[$headerName]);
            }
        }
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getBody(): \Psr\Http\Message\StreamInterface
    {
        if ($this->body === null) {
            $this->body = new Stream('php://temp', 'r+');
        }
        return $this->body;
    }

    /**
     * {@inheritdoc}
     */
    public function withBody(\Psr\Http\Message\StreamInterface $body): \Psr\Http\Message\MessageInterface
    {
        $new = clone $this;
        $new->body = $body;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getRequestTarget(): string
    {
        if ($this->requestTarget !== null) {
            return $this->requestTarget;
        }
        
        $target = '/' . $this->path;
        if (!empty($this->query)) {
            $target .= '?' . http_build_query($this->query);
        }
        
        return $target === '/' ? '/' : ltrim($target, '/');
    }

    /**
     * {@inheritdoc}
     */
    public function withRequestTarget($requestTarget): \Psr\Http\Message\RequestInterface
    {
        $new = clone $this;
        $new->requestTarget = $requestTarget;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withMethod($method): \Psr\Http\Message\RequestInterface
    {
        $new = clone $this;
        $new->method = $method;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getServerParams(): array
    {
        return $this->serverParams;
    }

    /**
     * {@inheritdoc}
     */
    public function getCookieParams(): array
    {
        return $this->cookieParams;
    }

    /**
     * {@inheritdoc}
     */
    public function withCookieParams(array $cookies): \Psr\Http\Message\ServerRequestInterface
    {
        $new = clone $this;
        $new->cookieParams = $cookies;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getQueryParams(): array
    {
        return $this->query;
    }

    /**
     * {@inheritdoc}
     */
    public function withQueryParams(array $query): \Psr\Http\Message\ServerRequestInterface
    {
        $new = clone $this;
        $new->query = $query;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getUploadedFiles(): array
    {
        return $this->uploadedFiles;
    }

    /**
     * {@inheritdoc}
     */
    public function withUploadedFiles(array $uploadedFiles): \Psr\Http\Message\ServerRequestInterface
    {
        $new = clone $this;
        $new->uploadedFiles = $uploadedFiles;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getParsedBody()
    {
        return $this->post;
    }

    /**
     * {@inheritdoc}
     */
    public function withParsedBody($data): \Psr\Http\Message\ServerRequestInterface
    {
        $new = clone $this;
        $new->post = $data;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function getAttributes(): array
    {
        return $this->attributes;
    }

    /**
     * {@inheritdoc}
     */
    public function getAttribute($name, $default = null)
    {
        return $this->attributes[$name] ?? $default;
    }

    /**
     * {@inheritdoc}
     */
    public function withAttribute($name, $value): \Psr\Http\Message\ServerRequestInterface
    {
        $new = clone $this;
        $new->attributes[$name] = $value;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withoutAttribute($name): \Psr\Http\Message\ServerRequestInterface
    {
        $new = clone $this;
        unset($new->attributes[$name]);
        return $new;
    }
}