<?php
// app/baseKRIZAN/Http/RequestClassifier.php

namespace baseKRIZAN\Http;

use baseKRIZAN\Error\Logger;

/**
 * Centralizirana klasa za klasifikaciju zahtjeva
 * 
 * Ova klasa je jedini izvor istine za određivanje tipa zahtjeva.
 * Tip zahtjeva se određuje jednom i zatim se koristi kroz cijelu aplikaciju.
 * Za API/Web zahtjeve koristi informacije iz router-a kada su dostupne.
 * 
 * @package baseKRIZAN\Http
 */
class RequestClassifier
{
    /**
     * Logger instanca
     * 
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Tip zahtjeva definiran od strane router-a
     * 
     * @var string|null
     */
    private ?string $routerDefinedType = null;
    
    /**
     * Paterni za prepoznavanje API zahtjeva
     * 
     * @var array
     */
    private array $apiPatterns = [
        '/^api\//',           // Počinje s api/
        '/\.json$/',          // Završava s .json
    ];
    
    /**
     * Ekstenzije za assetse
     * 
     * @var array
     */
    private array $assetExtensions = [
        'js', 'css', 'jpg', 'jpeg', 'png', 'gif', 'ico', 'svg', 
        'woff', 'ttf', 'eot', 'map'
    ];
    
    /**
     * Paterni za prepoznavanje redirect stranica
     * 
     * @var array
     */
    private array $redirectPatterns = [
        '/\/success$/',
        '/\/error$/',
        '/\/callback$/'
    ];
    
    /**
     * Cache za rezultate klasifikacije
     * 
     * @var array
     */
    private array $classificationCache = [];
    
    /**
     * Singleton instanca
     * 
     * @var self|null
     */
    private static ?self $instance = null;

    /**
     * Dohvaća singleton instancu
     * 
     * @param Logger|null $logger Logger instanca
     * @return self
     */
    public static function getInstance(?Logger $logger = null): self
    {
        if (self::$instance === null) {
            self::$instance = new self($logger);
        } elseif ($logger !== null && self::$instance->logger !== $logger) {
            // Ako je proslijeđen novi logger, ažuriraj postojeću instancu
            self::$instance->logger = $logger;
        }
        
        return self::$instance;
    }

    /**
     * Konstruktor
     *
     * @param Logger|null $logger Logger instanca
     */
    public function __construct(?Logger $logger = null)
    {
        $this->logger = $logger;
    }

    /**
     * Postavlja tip zahtjeva iz router-a
     * 
     * @param string $type Tip zahtjeva (web, api)
     * @return void
     */
    public function setRouterDefinedType(string $type): void
    {
        $this->routerDefinedType = $type;
        $this->classificationCache = []; // Poništi cache
        $this->log("Router defined request type: {$type}");
    }

    /**
     * Pomoćna metoda za logiranje
     *
     * @param string $message Poruka za logiranje
     * @param array $context Kontekstni podaci
     * @return void
     */
    private function log(string $message, array $context = []): void
    {
        if ($this->logger) {
            $this->logger->http($message, $context);
        }
    }

    /**
     * Dodaje pattern za prepoznavanje API zahtjeva
     *
     * @param string $pattern Regularni izraz
     * @return void
     */
    public function addApiPattern(string $pattern): void
    {
        if (!in_array($pattern, $this->apiPatterns)) {
            $this->apiPatterns[] = $pattern;
            $this->classificationCache = []; // Invalidate cache
        }
    }
    
    /**
     * Dodaje ekstenziju za assetse
     *
     * @param string $extension Ekstenzija
     * @return void
     */
    public function addAssetExtension(string $extension): void
    {
        $extension = strtolower(trim($extension, '.'));
        if (!in_array($extension, $this->assetExtensions)) {
            $this->assetExtensions[] = $extension;
            $this->classificationCache = []; // Invalidate cache
        }
    }
    
    /**
     * Dodaje pattern za prepoznavanje redirect stranica
     *
     * @param string $pattern Regularni izraz
     * @return void
     */
    public function addRedirectPattern(string $pattern): void
    {
        if (!in_array($pattern, $this->redirectPatterns)) {
            $this->redirectPatterns[] = $pattern;
            $this->classificationCache = []; // Invalidate cache
        }
    }

    /**
     * Određuje je li zahtjev API/AJAX
     *
     * @param Request|string $request Zahtjev ili putanja zahtjeva
     * @return bool True ako je API zahtjev
     */
    public function isApiRequest($request): bool
    {
        // Provjeri je li parametar Request ili string
        $path = is_string($request) ? $request : $request->getPath();
        $method = is_string($request) ? 'GET' : $request->getMethod();
        
        $cacheKey = 'api_' . md5($path . '_' . $method);
        
        if (isset($this->classificationCache[$cacheKey])) {
            return $this->classificationCache[$cacheKey];
        }
        
        // Provjeri je li već određen router tip
        if ($this->routerDefinedType === 'api') {
            $this->classificationCache[$cacheKey] = true;
            return true;
        }
        
        // Za Request objekte, provjeri headere
        if ($request instanceof Request) {
            // Provjeri X-Requested-With header (AJAX zahtjev)
            $xRequestedWith = $request->getHeader('X-Requested-With');
            if (!empty($xRequestedWith)) {
                $xRequestedWithValue = is_array($xRequestedWith) ? $xRequestedWith[0] : $xRequestedWith;
                if ($xRequestedWithValue === 'XMLHttpRequest') {
                    $this->log('Request classified as AJAX based on X-Requested-With header');
                    $this->classificationCache[$cacheKey] = true;
                    return true;
                }
            }
            
            // Provjeri Accept header (želi JSON)
            $acceptHeader = $request->getHeader('Accept');
            if (!empty($acceptHeader)) {
                $acceptHeaderStr = is_array($acceptHeader) ? implode(', ', $acceptHeader) : $acceptHeader;
                
                // Provjeri različite varijacije JSON accept headera
                if (strpos($acceptHeaderStr, 'application/json') !== false ||
                    strpos($acceptHeaderStr, 'text/javascript') !== false ||
                    strpos($acceptHeaderStr, 'application/javascript') !== false) {
                    
                    $this->log('Request classified as API based on Accept header', [
                        'accept' => $acceptHeaderStr
                    ]);
                    $this->classificationCache[$cacheKey] = true;
                    return true;
                }
            }
            
            // Provjeri Content-Type header (šalje JSON)
            $contentType = $request->getHeader('Content-Type');
            if (!empty($contentType)) {
                $contentTypeStr = is_array($contentType) ? implode(', ', $contentType) : $contentType;
                
                if (strpos($contentTypeStr, 'application/json') !== false) {
                    $this->log('Request classified as API based on Content-Type header', [
                        'content_type' => $contentTypeStr
                    ]);
                    $this->classificationCache[$cacheKey] = true;
                    return true;
                }
            }
            
            // Provjeri metodu zahtjeva
            if ($method !== 'GET' && $method !== 'POST') {
                // PUT, DELETE, PATCH su često API zahtjevi
                $this->log('Request classified as API based on HTTP method', [
                    'method' => $method
                ]);
                $this->classificationCache[$cacheKey] = true;
                return true;
            }
        } else {
            // Za stringove, provjeri $_SERVER varijable
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                
                $this->log('Request classified as AJAX based on X-Requested-With header');
                $this->classificationCache[$cacheKey] = true;
                return true;
            }
            
            if (isset($_SERVER['HTTP_ACCEPT'])) {
                $acceptHeader = $_SERVER['HTTP_ACCEPT'];
                if (strpos($acceptHeader, 'application/json') !== false || 
                    strpos($acceptHeader, 'application/javascript') !== false) {
                    
                    $this->log('Request classified as API based on Accept header', [
                        'accept' => $acceptHeader
                    ]);
                    $this->classificationCache[$cacheKey] = true;
                    return true;
                }
            }
            
            if (isset($_SERVER['HTTP_CONTENT_TYPE']) && 
                strpos($_SERVER['HTTP_CONTENT_TYPE'], 'application/json') !== false) {
                
                $this->log('Request classified as API based on Content-Type header');
                $this->classificationCache[$cacheKey] = true;
                return true;
            }
        }
        
        // Osnovni paterni za API endpointe, zadržavamo samo osnovne patterne
        $basicApiPatterns = [
            '/^api\//'  // Počinje s api/
        ];
        
        // Provjeri URL patterne
        foreach ($basicApiPatterns as $pattern) {
            if (preg_match($pattern, $path)) {
                $this->log('Request classified as API based on URL pattern', [
                    'path' => $path,
                    'pattern' => $pattern
                ]);
                $this->classificationCache[$cacheKey] = true;
                return true;
            }
        }
        
        $this->log('Request classified as standard web request');
        $this->classificationCache[$cacheKey] = false;
        return false;
    }
    
    /**
     * Određuje je li zahtjev za statički asset
     *
     * @param Request|string $request Zahtjev ili putanja zahtjeva
     * @return bool True ako je zahtjev za asset
     */
    public function isAssetRequest($request): bool
    {
        // Provjeri je li parametar Request ili string
        $path = is_string($request) ? $request : $request->getPath();
        
        $cacheKey = 'asset_' . md5($path);
        
        if (isset($this->classificationCache[$cacheKey])) {
            return $this->classificationCache[$cacheKey];
        }
        
        // Izgradi regex pattern iz ekstenzija
        $pattern = '/\.(' . implode('|', $this->assetExtensions) . ')(\?.*)?$/i';
        
        $result = preg_match($pattern, $path) === 1;
        $this->classificationCache[$cacheKey] = $result;
        return $result;
    }
    
    /**
     * Određuje je li zahtjev iz konzole
     * 
     * @return bool True ako je zahtjev iz CLI
     */
    public function isConsoleRequest(): bool
    {
        return PHP_SAPI === 'cli';
    }
    
    /**
     * Određuje je li zahtjev za redirect stranicu
     *
     * @param Request|string $request Zahtjev ili putanja zahtjeva
     * @return bool True ako je zahtjev za redirect stranicu
     */
    public function isRedirectPage($request): bool
    {
        // Provjeri je li parametar Request ili string
        $path = is_string($request) ? $request : $request->getPath();
        
        $cacheKey = 'redirect_' . md5($path);
        
        if (isset($this->classificationCache[$cacheKey])) {
            return $this->classificationCache[$cacheKey];
        }
        
        foreach ($this->redirectPatterns as $pattern) {
            if (preg_match($pattern, $path)) {
                $this->classificationCache[$cacheKey] = true;
                return true;
            }
        }
        
        $this->classificationCache[$cacheKey] = false;
        return false;
    }
    
    /**
     * Analizira zahtjev i vraća detalje
     *
     * @param Request|string $request Zahtjev ili putanja zahtjeva
     * @return array Detalji analize
     */
    public function analyzeRequest($request): array
    {
        // Provjeri je li parametar Request ili string
        if (is_string($request)) {
            $tempRequest = new Request();
            $tempRequest->setPath($request);
            $request = $tempRequest;
        }
        
        $analysis = [
            'path' => $request->getPath(),
            'method' => $request->getMethod(),
            'is_console' => $this->isConsoleRequest(),
            'is_ajax' => $request->isAjax(),
            'is_api' => $this->isApiRequest($request),
            'is_asset' => $this->isAssetRequest($request),
            'is_redirect' => $this->isRedirectPage($request),
            'router_defined_type' => $this->routerDefinedType,
            'bootstrap_type' => $this->determineBootstrapType($request),
            'request_type' => $this->getRequestCategory($request),
            'accept_header' => $request->getHeader('Accept'),
            'content_type' => $request->getHeader('Content-Type'),
            'client_ip' => $request->getIp(),
            'user_agent' => $request->userAgent(),
            'timestamp' => date('Y-m-d H:i:s'),
            'wants_json' => $this->wantsJson($request)
        ];
        
        $this->log('Request analysis', $analysis);
        
        return $analysis;
    }
    
    /**
     * Resetira cache
     *
     * @return void
     */
    public function resetCache(): void
    {
        $this->classificationCache = [];
    }
    
    /**
    * Određuje kategoriju zahtjeva
    *
    * @param Request|string $request Zahtjev ili putanja zahtjeva
    * @return string Kategorija zahtjeva ('api', 'asset', 'web', 'redirect', 'console')
    */
   public function getRequestCategory($request): string
   {
       // Ako je konzolni zahtjev, vrati odmah
       if ($this->isConsoleRequest()) {
           return 'console';
       }
       
       // Provjeri je li parametar Request ili string
       $path = is_string($request) ? $request : $request->getPath();
       
       // Za asset zahtjeve, uvijek koristi postojeću logiku jer router ne rukuje statičkim datotekama
       if ($this->isAssetRequest($request)) {
           return 'asset';
       }
       
       // Ako je tip definiran od strane router-a, koristi ga za api/web
       if ($this->routerDefinedType !== null) {
           if (in_array($this->routerDefinedType, ['api', 'web'])) {
               return $this->routerDefinedType;
           }
       }
       
       // Eksplicitna provjera za API putanje - ako sadrži /api/ u putanji
       if (strpos($path, '/api/') !== false) {
           $this->log("Request classified as API based on path containing /api/", [
               'path' => $path
           ]);
           return 'api';
       }
       
       // Za API zahtjeve, koristi API detekciju
       if ($this->isApiRequest($request)) {
           return 'api';
       }
       
       // Za redirect stranice
       if ($this->isRedirectPage($request)) {
           return 'redirect';
       }
       
       // Default je web
       return 'web';
   }
   
   /**
    * Određuje bootstrap mod na temelju zahtjeva
    * 
    * @param Request|string $request Zahtjev ili putanja zahtjeva
    * @return string Bootstrap mod ('api', 'minimal', 'console', 'full')
    */
   public function determineBootstrapType($request): string
   {
       if ($this->isConsoleRequest()) {
           return 'console';
       }
       
       if ($this->isAssetRequest($request)) {
           return 'minimal';
       }
       
       // Ako je tip definiran od strane router-a, koristi ga za api/web
       if ($this->routerDefinedType !== null && $this->routerDefinedType === 'api') {
           return 'api';
       }
       
       if ($this->isApiRequest($request)) {
           return 'api';
       }
       
       return 'full'; // Default bootstrap type
   }
   
   /**
    * Check if request wants JSON response
    * Premješteno iz BaseController
    *
    * @param Request $request Zahtjev za provjeru
    * @return bool True ako zahtjev želi JSON
    */
   public function wantsJson(Request $request): bool 
   {
       $cacheKey = 'wants_json_' . md5($request->getPath() . '_' . $request->getMethod());
       
       if (isset($this->classificationCache[$cacheKey])) {
           return $this->classificationCache[$cacheKey];
       }
       
       $result = $request->wantsJson() || 
                $request->isJson() || 
                $this->isAjax($request) || 
                str_contains($request->getHeaderLine('Accept') ?? '', 'application/json');
                
       $this->classificationCache[$cacheKey] = $result;
       return $result;
   }
   
   /**
    * Check if request is an AJAX request
    * Premješteno iz BaseController
    *
    * @param Request $request Zahtjev za provjeru
    * @return bool True ako je AJAX zahtjev
    */
   public function isAjax(Request $request): bool 
   {
       $cacheKey = 'is_ajax_' . md5($request->getPath() . '_' . $request->getMethod());
       
       if (isset($this->classificationCache[$cacheKey])) {
           return $this->classificationCache[$cacheKey];
       }
       
       $result = $request->isAjax() || 
                $request->getHeaderLine('X-Requested-With') === 'XMLHttpRequest';
       
       $this->classificationCache[$cacheKey] = $result;
       return $result;
   }
}