<?php

namespace baseKRIZAN\Http;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Template\TemplateRenderer;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

/**
 * Implementacija HTTP odgovora s PSR-7 kompatibilnošću
 * 
 * @package baseKRIZAN\Http
 */
class Response implements ResponseInterface
{
    /**
     * HTTP statusni kod
     * 
     * @var int
     */
    private int $statusCode;
    
    /**
     * HTTP statusna poruka
     * 
     * @var string
     */
    private string $reasonPhrase = '';
    
    /**
     * HTTP headeri
     * 
     * @var array
     */
    private array $headers;
    
    /**
     * Tijelo odgovora
     * 
     * @var mixed
     */
    private $content;
    
    /**
     * Protokol verzija
     * 
     * @var string
     */
    private string $protocolVersion = '1.1';
    
    /**
     * Je li odgovor redirect
     * 
     * @var bool
     */
    private bool $isRedirect = false;
    
    /**
     * URL za preusmjeravanje
     * 
     * @var string|null
     */
    private ?string $redirectUrl = null;
    
    /**
     * Kolačići za postavljanje
     * 
     * @var array
     */
    private array $cookies = [];
    
    /**
     * Logger instance
     * 
     * @var Logger|null
     */
    private ?Logger $logger = null;
    
    /**
     * Tijelo kao Stream objekt
     * 
     * @var StreamInterface|null
     */
    private ?StreamInterface $body = null;

    /**
     * Konstruktor
     *
     * @param mixed $content Tijelo odgovora
     * @param int $statusCode HTTP statusni kod
     * @param array $headers HTTP headeri
     */
    public function __construct($content = '', int $statusCode = 200, array $headers = [])
    {
        $this->content = $content;
        $this->statusCode = $statusCode;
        $this->headers = array_merge([
            'Content-Type' => 'text/html; charset=UTF-8',
            'X-Frame-Options' => 'SAMEORIGIN',
            'X-XSS-Protection' => '1; mode=block',
            'X-Content-Type-Options' => 'nosniff'
        ], $headers);
        
        $this->reasonPhrase = $this->getReasonPhraseForStatusCode($statusCode);
    }
    
    /**
     * Postavlja logger
     *
     * @param Logger|null $logger Logger instanca
     * @return void
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function getBody(): \Psr\Http\Message\StreamInterface
    {
        if ($this->body === null) {
            $resource = fopen('php://temp', 'r+');
            fwrite($resource, (string)$this->content);
            rewind($resource);
            $this->body = new Stream($resource);
        }
        
        return $this->body;
    }

    /**
     * Šalje odgovor klijentu
     *
     * @return never Završava izvršavanje skripte
     */
    public function send(): never
    {
        // Logiraj odgovor prije slanja ako je logger dostupan
        if ($this->logger) {
            $this->logger->http('Sending response', [
                'status' => $this->statusCode,
                'content_type' => $this->headers['Content-Type'] ?? 'unknown',
                'is_redirect' => $this->isRedirect ? 'yes' : 'no'
            ]);
        }

        if (!headers_sent()) {
            // Postavi security headers
            $this->setSecurityHeaders();
            
            // Postavi HTTP status code
            http_response_code($this->statusCode);

            // Postavi cookies
            $this->sendCookies();

            // Postavi headers
            foreach ($this->headers as $name => $value) {
                if (is_array($value)) {
                    foreach ($value as $v) {
                        header("$name: $v", false);
                    }
                } else {
                    header("$name: $value");
                }
            }
        }

        if ($this->isRedirect) {
            header("Location: " . $this->redirectUrl);
            exit;
        }

        echo $this->content;
        exit;
    }

    /**
     * Postavlja tijelo odgovora
     *
     * @param mixed $content Novo tijelo odgovora
     * @return self Fluent interface
     */
    public function setContent($content): self
    {
        $this->content = $content;
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getHeader($name): array
    {
        foreach ($this->headers as $headerName => $value) {
            if (strtolower($headerName) === strtolower($name)) {
                // Pretvaramo u array ako je string
                return is_array($value) ? $value : [$value];
            }
        }
        return [];
    }

    /**
     * Postavlja HTTP header
     *
     * @param string $name Ime headera
     * @param string $value Vrijednost headera
     * @return self Fluent interface
     */
    public function setHeader(string $name, string $value): self {
        $this->headers[$name] = $value;
        return $this;
    }

    /**
     * Uklanja header
     *
     * @param string $name Ime headera za uklanjanje
     * @return self Fluent interface
     */
    public function removeHeader(string $name): self
    {
        unset($this->headers[$name]);
        return $this;
    }
    
    /**
     * Dohvaća sve headere
     *
     * @return array Svi headeri
     */
    public function getHeaders(): array {
        return $this->headers;
    }

    /**
     * Spaja headere s postojećima
     *
     * @param array $headers Headeri za spajanje
     * @return self Fluent interface
     */
    public function mergeHeaders(array $headers): self {
        $this->headers = array_merge($this->headers, $headers);
        return $this;
    }

    /**
     * Postavlja preusmjeravanje
     *
     * @param string $url URL za preusmjeravanje
     * @param int $statusCode HTTP statusni kod
     * @return self Fluent interface
     */
    public function setRedirect(string $url, int $statusCode = 302): self
    {
        $this->isRedirect = true;
        $this->redirectUrl = route($url);
        $this->statusCode = $statusCode;
        
        if ($this->logger) {
            $this->logger->http('Setting redirect', [
                'url' => $this->redirectUrl,
                'status' => $statusCode
            ]);
        }
        
        return $this;
    }

    /**
     * Postavlja kolačić
     *
     * @param string $name Ime kolačića
     * @param string $value Vrijednost kolačića
     * @param int $expire Vrijeme isteka
     * @param string $path Putanja
     * @param string $domain Domena
     * @param bool $secure Secure flag
     * @param bool $httpOnly HttpOnly flag
     * @param string $sameSite SameSite opcija
     * @return self Fluent interface
     */
    public function setCookie(
        string $name, 
        string $value, 
        int $expire = 0, 
        string $path = '/', 
        string $domain = '', 
        bool $secure = true, 
        bool $httpOnly = true,
        string $sameSite = 'Lax'
    ): self {
        $this->cookies[] = [
            'name' => $name,
            'value' => $value,
            'expire' => $expire,
            'path' => $path,
            'domain' => $domain,
            'secure' => $secure,
            'httpOnly' => $httpOnly,
            'sameSite' => $sameSite
        ];
        
        if ($this->logger) {
            $this->logger->http('Setting cookie', [
                'name' => $name,
                'path' => $path,
                'expire' => $expire > 0 ? date('Y-m-d H:i:s', $expire) : 'session'
            ]);
        }
        
        return $this;
    }

    /**
     * Šalje kolačiće
     *
     * @return void
     */
    private function sendCookies(): void
    {
        foreach ($this->cookies as $cookie) {
            setcookie(
                $cookie['name'],
                $cookie['value'],
                [
                    'expires' => $cookie['expire'],
                    'path' => $cookie['path'],
                    'domain' => $cookie['domain'],
                    'secure' => $cookie['secure'],
                    'httponly' => $cookie['httpOnly'],
                    'samesite' => $cookie['sameSite']
                ]
            );
        }
        
        if ($this->cookies && $this->logger) {
            $this->logger->http('Cookies sent', [
                'count' => count($this->cookies)
            ]);
        }
    }

    /**
     * Postavlja sigurnosne headere
     *
     * @return void
     */
    private function setSecurityHeaders(): void
    {
        // Only set headers that aren't already set
        $securityHeaders = [
            'X-Frame-Options' => 'SAMEORIGIN',
            'X-XSS-Protection' => '1; mode=block',
            'X-Content-Type-Options' => 'nosniff',
            'Referrer-Policy' => 'strict-origin-when-cross-origin',
            'Permissions-Policy' => 'geolocation=(), microphone=(), camera=()'
        ];

        foreach ($securityHeaders as $header => $value) {
            if (!isset($this->headers[$header])) {
                $this->headers[$header] = $value;
            }
        }
        
        if ($this->logger) {
            $this->logger->http('Security headers applied');
        }
    }

    /**
     * Dohvaća odgovarajuću reason phrase za statusni kod
     *
     * @param int $statusCode HTTP statusni kod
     * @return string Odgovarajuća reason phrase
     */
    private function getReasonPhraseForStatusCode(int $statusCode): string
    {
        $phrases = [
            100 => 'Continue',
            101 => 'Switching Protocols',
            102 => 'Processing',
            200 => 'OK',
            201 => 'Created',
            202 => 'Accepted',
            203 => 'Non-Authoritative Information',
            204 => 'No Content',
            205 => 'Reset Content',
            206 => 'Partial Content',
            207 => 'Multi-Status',
            208 => 'Already Reported',
            226 => 'IM Used',
            300 => 'Multiple Choices',
            301 => 'Moved Permanently',
            302 => 'Found',
            303 => 'See Other',
            304 => 'Not Modified',
            305 => 'Use Proxy',
            306 => 'Switch Proxy',
            307 => 'Temporary Redirect',
            308 => 'Permanent Redirect',
            400 => 'Bad Request',
            401 => 'Unauthorized',
            402 => 'Payment Required',
            403 => 'Forbidden',
            404 => 'Not Found',
            405 => 'Method Not Allowed',
            406 => 'Not Acceptable',
            407 => 'Proxy Authentication Required',
            408 => 'Request Timeout',
            409 => 'Conflict',
            410 => 'Gone',
            411 => 'Length Required',
            412 => 'Precondition Failed',
            413 => 'Payload Too Large',
            414 => 'URI Too Long',
            415 => 'Unsupported Media Type',
            416 => 'Range Not Satisfiable',
            417 => 'Expectation Failed',
            418 => "I'm a teapot",
            421 => 'Misdirected Request',
            422 => 'Unprocessable Entity',
            423 => 'Locked',
            424 => 'Failed Dependency',
            426 => 'Upgrade Required',
            428 => 'Precondition Required',
            429 => 'Too Many Requests',
            431 => 'Request Header Fields Too Large',
            451 => 'Unavailable For Legal Reasons',
            500 => 'Internal Server Error',
            501 => 'Not Implemented',
            502 => 'Bad Gateway',
            503 => 'Service Unavailable',
            504 => 'Gateway Timeout',
            505 => 'HTTP Version Not Supported',
            506 => 'Variant Also Negotiates',
            507 => 'Insufficient Storage',
            508 => 'Loop Detected',
            510 => 'Not Extended',
            511 => 'Network Authentication Required',
        ];
        
        return $phrases[$statusCode] ?? '';
    }

    // Statičke metode za kreiranje odgovora

    /**
     * Kreira JSON odgovor
     *
     * @param mixed $data Podaci za JSON
     * @param int $statusCode HTTP statusni kod
     * @return self Novi Response objekt
     */
    public static function json($data, int $statusCode = 200): self
    {
        return new self(
            json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
            $statusCode,
            ['Content-Type' => 'application/json; charset=UTF-8']
        );
    }

    /**
     * Kreira standardizirani API odgovor
     *
     * @param mixed $data Podaci
     * @param string $message Poruka
     * @param bool $success Uspjeh
     * @return self Novi Response objekt
     */
    public static function api($data = null, string $message = '', bool $success = true): self
    {
        return self::json([
            'success' => $success,
            'message' => $message,
            'data' => $data
        ]);
    }

    /**
     * Kreira HTML odgovor
     *
     * @param string $content HTML sadržaj
     * @param int $statusCode HTTP statusni kod
     * @return self Novi Response objekt
     */
    public static function html(string $content, int $statusCode = 200): self
    {
        return new self($content, $statusCode, [
            'Content-Type' => 'text/html; charset=UTF-8'
        ]);
    }

    /**
     * Kreira tekstualni odgovor
     *
     * @param string $content Tekst
     * @param int $statusCode HTTP statusni kod
     * @return self Novi Response objekt
     */
    public static function text(string $content, int $statusCode = 200): self
    {
        return new self($content, $statusCode, [
            'Content-Type' => 'text/plain; charset=UTF-8'
        ]);
    }

    /**
     * Kreira view odgovor
     *
     * @param string $template Putanja do templatea
     * @param array $data Podaci za template
     * @param string $title Naslov
     * @param int $status HTTP statusni kod
     * @param Container|null $container DI container
     * @return self Novi Response objekt
     */
    public static function view(string $template, array $data = [], string $title = '', int $status = 200, ?Container $container = null): self
    {
        // Dohvati potrebne servise iz containera ili kreiraj nove
        if ($container) {
            $logger = $container->get('logger');
            $sessionManager = $container->get('sessionManager');
            
            // Dohvati renderer iz containera ako postoji
            if ($container->has('templateRenderer')) {
                $templateRenderer = $container->get('templateRenderer');
            } else {
                // Kreiraj potrebne ovisnosti za renderer
                $loader = new \baseKRIZAN\Template\TemplateLoader([
                    \baseKRIZAN\Config\Config::get('paths.resourcesviews') . '/',
                    \baseKRIZAN\Config\Config::get('paths.moduli') . '/'
                ], $logger);
                
                $processor = new \baseKRIZAN\Template\DirectiveProcessor($logger);
                
                $cache = new \baseKRIZAN\Template\TemplateCache(
                    \baseKRIZAN\Config\Config::get('paths.cacheviews'),
                    \baseKRIZAN\Config\Config::get('environment') !== 'development',
                    $logger
                );
                
                $config = new \baseKRIZAN\Config\Config();
                
                // Kreiraj novi renderer umjesto korištenja singletona
                $templateRenderer = new \baseKRIZAN\Template\TemplateRenderer(
                    $logger,
                    $sessionManager,
                    $loader,
                    $processor,
                    $cache,
                    $config
                );
            }
        } else {
            // Fallback za slučaj da container nije dostupan
            $logger = new Logger(\baseKRIZAN\Config\Config::get('paths.logs'), true);
            $sessionManager = new \baseKRIZAN\Session\SessionManager($logger);
            
            // Kreiraj potrebne ovisnosti za renderer
            $loader = new \baseKRIZAN\Template\TemplateLoader([
                \baseKRIZAN\Config\Config::get('paths.resourcesviews') . '/',
                \baseKRIZAN\Config\Config::get('paths.moduli') . '/'
            ], $logger);
            
            $processor = new \baseKRIZAN\Template\DirectiveProcessor($logger);
            
            $cache = new \baseKRIZAN\Template\TemplateCache(
                \baseKRIZAN\Config\Config::get('paths.cacheviews'),
                \baseKRIZAN\Config\Config::get('environment') !== 'development',
                $logger
            );
            
            $config = new \baseKRIZAN\Config\Config();
            
            // Kreiraj novi renderer umjesto korištenja singletona
            $templateRenderer = new \baseKRIZAN\Template\TemplateRenderer(
                $logger,
                $sessionManager,
                $loader,
                $processor,
                $cache,
                $config
            );
        }
        
        $content = $templateRenderer->render($template, $data);
        
        $response = new self($content, $status);
        
        // Postavi logger na novi response objekt
        $response->setLogger($logger);
        
        return $response;
    }

    /**
     * Render a template with the standard layout
     * 
     * @param string $template Template path
     * @param array $data Template data
     * @param string $title Page title
     * @param int $statusCode HTTP status code
     * @param array $commonViewData Common view data
     * @param string $layout Layout template path
     * @return self Fluent interface
     */
    public function render(
        string $template, 
        array $data = [], 
        string $title = '', 
        int $statusCode = 200,
        array $commonViewData = [],
        string $layout = 'layout.html.php',
        ?TemplateRenderer $templateRenderer = null
    ): self {
        // Ako commonViewData nije proslijeđen ili je prazan, dohvati ga automatski
        if (empty($commonViewData)) {
            $commonViewData = $this->getCommonViewData();
        }
        
        if (!$templateRenderer) {
            // Pokušaj dohvatiti TemplateRenderer iz container-a
            if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
                $bootstrap = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
                if ($bootstrap && method_exists($bootstrap, 'getContainer')) {
                    $container = $bootstrap->getContainer();
                    if ($container && $container->has('templateRenderer')) {
                        $templateRenderer = $container->get('templateRenderer');
                    }
                }
            }
            
            if (!$templateRenderer) {
                throw new \RuntimeException('TemplateRenderer not available for rendering');
            }
        }
        
        try {
            // Spoji s korisničkim podacima
            $mergedData = array_merge($data, $commonViewData);
            
            // Očisti stare resurse prije renderiranja
            if (class_exists('\\baseKRIZAN\\Assets\\ResourceLoader')) {
                \baseKRIZAN\Assets\ResourceLoader::clear();
            }
            
            // Prvo renderiraj template
            $content = $templateRenderer->render($template, $mergedData);
            
            // Zatim renderiraj layout s contentom
            $fullPage = $templateRenderer->render($layout, array_merge([
                'output' => $content,
                'title' => $title,
            ], $commonViewData));
            
            // Postavi sadržaj i status kod odgovora
            $this->content = $fullPage;
            $this->statusCode = $statusCode;
            
            return $this;
        } catch (\Throwable $e) {
            return $this->handleRenderingError($e, $template);
        }
    }

    /**
     * Get common data for rendering views
     * 
     * @return array Common view data
     */
    private function getCommonViewData(): array
    {
        $loggedIn = false;
        $user = null;
        $appName = '';
        $environment = '';
        
        // Pokušaj dohvatiti authentication iz containera
        if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
            $bootstrap = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
            if ($bootstrap && method_exists($bootstrap, 'getContainer')) {
                $container = $bootstrap->getContainer();
                
                // Dohvati authentication ako postoji
                if ($container && $container->has('authentication')) {
                    $authentication = $container->get('authentication');
                    $loggedIn = $authentication->isLoggedIn();
                    $user = $authentication->getUser();
                }
                
                // Dohvati config ako postoji
                if ($container && $container->has('config')) {
                    $config = $container->get('config');
                    $appName = $config->get('appname');
                    $environment = $config->get('environment');
                }
            }
        }
        
        // Ako ne možemo dohvatiti putem containera, pokušamo direktno iz Config klase
        if (empty($appName) && class_exists('\\baseKRIZAN\\Config\\Config')) {
            $appName = \baseKRIZAN\Config\Config::get('appname');
            $environment = \baseKRIZAN\Config\Config::get('environment');
        }
        
        return [
            'loggedIn' => $loggedIn,
            'user' => $user,
            'currentYear' => date('Y'),
            'appName' => $appName,
            'environment' => $environment
        ];
    }

    /**
     * Handle rendering errors
     * 
     * @param \Throwable $exception Exception that occurred
     * @param string $context Context information
     * @return self Fluent interface
     */
    private function handleRenderingError(\Throwable $exception, string $context): self {
        if ($this->logger) {
            $this->logger->error('Template rendering error', [
                'context' => $context,
                'error' => $exception->getMessage(),
                'trace' => $exception->getTraceAsString()
            ]);
        }
        
        // U development modu, prikaži detaljnu grešku
        $environment = '';
        
        // Pokušaj dohvatiti environment iz Config-a
        if (class_exists('\\baseKRIZAN\\Config\\Config')) {
            $environment = \baseKRIZAN\Config\Config::get('environment');
        }
        
        if ($environment === 'development') {
            $this->content = "Error rendering template: " . $exception->getMessage() . 
                "<pre>" . htmlspecialchars($exception->getTraceAsString()) . "</pre>";
            $this->statusCode = 500;
        } else {
            // U produkciji, prikaži generičku grešku
            $this->content = "An error occurred while processing your request.";
            $this->statusCode = 500;
        }
        
        return $this;
    }
    
    /**
     * Render a form with validation errors
     * 
     * @param array $formData Form data
     * @param array $errors Validation errors
     * @param string $template Template path
     * @param string $title Page title
     * @param int $statusCode HTTP status code
     * @param array $commonViewData Common view data
     * @return self Fluent interface
     */
    public function renderForm(
        array $formData, 
        array $errors, 
        string $template, 
        string $title, 
        int $statusCode = 200,
        array $commonViewData = [],
        ?TemplateRenderer $templateRenderer = null
    ): self {
        // Ako commonViewData nije proslijeđen ili je prazan, dohvati ga automatski
        if (empty($commonViewData)) {
            $commonViewData = $this->getCommonViewData();
        }
        
        if (!$templateRenderer) {
            // Pokušaj dohvatiti TemplateRenderer iz container-a
            if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
                $bootstrap = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
                if ($bootstrap && method_exists($bootstrap, 'getContainer')) {
                    $container = $bootstrap->getContainer();
                    if ($container && $container->has('templateRenderer')) {
                        $templateRenderer = $container->get('templateRenderer');
                    }
                }
            }
            
            if (!$templateRenderer) {
                throw new \RuntimeException('TemplateRenderer not available for rendering form');
            }
        }
        
        try {
            // Kombiniraj podatke za renderiranje templatea
            $templateData = [
                'korisnik' => $formData,
                'errors' => $errors,
            ] + $commonViewData;
            
            // Renderiraj form template
            $content = $templateRenderer->render($template, $templateData);
            
            // Renderiraj layout s contentom
            $layoutData = array_merge([
                'output' => $content,
                'title' => $title,
            ], $commonViewData);
            
            $fullPage = $templateRenderer->render('layout.html.php', $layoutData);
            
            $this->content = $fullPage;
            $this->statusCode = $statusCode;
            
            return $this;
        } catch (\Throwable $e) {
            if ($this->logger) {
                $this->logger->error('Error rendering form template', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
            
            // Format grešaka za fallback odgovor
            $errorMessage = $this->formatErrorsAsString($errors);
            $this->content = "Form Error: " . $errorMessage;
            $this->statusCode = $statusCode;
            
            return $this;
        }
    }
   
   /**
    * Format errors array as string
    */
   private function formatErrorsAsString(array $errors): string
   {
       $errorLines = [];
       
       foreach ($errors as $field => $error) {
           if (is_array($error)) {
               $errorLines[] = $field . ': ' . implode(', ', $error);
           } else {
               $errorLines[] = $error;
           }
       }
       
       return implode('; ', $errorLines);
   }

   /**
     * Render an informational message
     */
    public function renderInfo(
        string $message, 
        string $submessage = '', 
        string $title = 'Information', 
        int $statusCode = 200,
        array $commonViewData = [],
        ?TemplateRenderer $templateRenderer = null
    ): self {
        // Ako commonViewData nije proslijeđen ili je prazan, dohvati ga automatski
        if (empty($commonViewData)) {
            $commonViewData = $this->getCommonViewData();
        }
        
        return $this->renderMessageTemplate(
            'Error/staticerrortemplate.html.php', // Koristimo template za info poruke
            $message,
            $submessage,
            $title,
            $statusCode,
            $commonViewData,
            $templateRenderer
        );
    }

    /**
     * Render an error message
     */
    public function renderError(
        string $message, 
        string $title = 'Error', 
        int $statusCode = 400,
        array $commonViewData = [],
        ?TemplateRenderer $templateRenderer = null
    ): self {
        // Ako commonViewData nije proslijeđen ili je prazan, dohvati ga automatski
        if (empty($commonViewData)) {
            $commonViewData = $this->getCommonViewData();
        }
        
        return $this->renderMessageTemplate(
            'Error/errortemplate.html.php', // Koristimo template za greške
            $title,
            $message,
            $title,
            $statusCode,
            $commonViewData,
            $templateRenderer
        );
    }

    /**
     * Render a message using a template
     */
    private function renderMessageTemplate(
        string $template,
        string $message,
        string $submessage,
        string $title,
        int $statusCode,
        array $commonViewData = [],
        ?TemplateRenderer $templateRenderer = null
    ): self {
        // Ako commonViewData nije proslijeđen ili je prazan, dohvati ga automatski
        if (empty($commonViewData)) {
            $commonViewData = $this->getCommonViewData();
        }
        
        if (!$templateRenderer) {
            // Pokušaj dohvatiti TemplateRenderer iz container-a
            if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
                $bootstrap = \baseKRIZAN\Bootstrap\Bootstrap::getInstance();
                if ($bootstrap && method_exists($bootstrap, 'getContainer')) {
                    $container = $bootstrap->getContainer();
                    if ($container && $container->has('templateRenderer')) {
                        $templateRenderer = $container->get('templateRenderer');
                    }
                }
            }
            
            if (!$templateRenderer) {
                throw new \RuntimeException('TemplateRenderer not available for rendering message');
            }
        }
        
        try {
            // Renderiraj message template
            $content = $templateRenderer->render(
                $template,
                [
                    'msg' => $message,
                    'msg2' => $submessage
                ]
            );
            
            // Renderiraj layout s contentom
            $fullPage = $templateRenderer->render('layout.html.php', array_merge([
                'output' => $content,
                'title' => $title,
            ], $commonViewData));
            
            $this->content = $fullPage;
            $this->statusCode = $statusCode;
            
            return $this;
        } catch (\Throwable $e) {
            if ($this->logger) {
                $this->logger->error('Error rendering message template', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
            
            $this->content = "Error: " . $message;
            $this->statusCode = $statusCode;
            
            return $this;
        }
    }
   
   /**
    * Create success API response
    */
   public function success(
       mixed $data = null, 
       string $message = 'Operation completed successfully'
   ): self {
       return $this->apiResponse($data, $message, true, 200);
   }
   
   /**
    * Create error API response
    */
   public function error(
       string $message, 
       mixed $data = null, 
       int $status = 400
   ): self {
       return $this->apiResponse($data, $message, false, $status);
   }
   
   /**
    * Create API response with standardized structure
    */
   public function apiResponse(
       mixed $data = null, 
       string $message = '', 
       bool $success = true, 
       int $status = 200
   ): self {
       $this->content = json_encode([
           'success' => $success,
           'message' => $message,
           'data' => $data,
           'timestamp' => time()
       ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
       
       $this->statusCode = $status;
       $this->headers['Content-Type'] = 'application/json; charset=UTF-8';
       
       return $this;
   }
   
   /**
   * Kreira preusmjeravanje
   *
   * @param string $url URL za preusmjeravanje
   * @param int $status HTTP statusni kod
   * @return self Novi Response objekt
   */
   public static function redirect(string $url, int $status = 302): self
   {
       $response = new self('', $status);
       return $response->setRedirect($url, $status);
   }
  
   /**
   * Kreira trajno preusmjeravanje (301)
   *
   * @param string $url URL za preusmjeravanje
   * @return self Novi Response objekt
   */
   public static function permanentRedirect(string $url): self
   {
       return self::redirect($url, 301);
   }
  
   /**
   * Kreira privremeno preusmjeravanje (307)
   *
   * @param string $url URL za preusmjeravanje
   * @return self Novi Response objekt
   */
   public static function temporaryRedirect(string $url): self
   {
       return self::redirect($url, 307);
   }
   
   /**
    * Kreira odgovor za preuzimanje datoteke
    *
    * @param string $content Sadržaj datoteke
    * @param string $filename Ime datoteke
    * @param string $contentType MIME tip
    * @return self Novi Response objekt
    */
   public static function download(string $content, string $filename, string $contentType = 'application/octet-stream'): self
   {
       return new self($content, 200, [
           'Content-Type' => $contentType,
           'Content-Disposition' => 'attachment; filename="' . $filename . '"',
           'Content-Length' => strlen($content),
           'Cache-Control' => 'private, no-cache, no-store, must-revalidate',
           'Pragma' => 'no-cache',
           'Expires' => '0'
       ]);
   }
   
   /**
    * Kreira odgovor s datotekom
    *
    * @param string $path Putanja do datoteke
    * @param string|null $contentType MIME tip ili null za automatsko određivanje
    * @return self Novi Response objekt
    */
   public static function file(string $path, string $contentType = null): self
   {
       if (!file_exists($path)) {
           throw new \RuntimeException("File not found: $path");
       }
       
       $content = file_get_contents($path);
       
       if ($contentType === null) {
           $contentType = mime_content_type($path) ?: 'application/octet-stream';
       }
       
       return new self($content, 200, [
           'Content-Type' => $contentType,
           'Content-Length' => strlen($content)
       ]);
   }
  
   /**
   * Dodaje cache control headere
   *
   * @param int $seconds Vrijeme trajanja cachea u sekundama
   * @return self Fluent interface
   */
   public function cache(int $seconds): self
   {
       $this->setHeader('Cache-Control', "public, max-age=$seconds");
       $this->setHeader('Expires', gmdate('D, d M Y H:i:s', time() + $seconds) . ' GMT');
       
       if ($this->logger) {
           $this->logger->http('Cache headers added', [
               'seconds' => $seconds
           ]);
       }
       
       return $this;
   }
  
   /**
   * Dodaje no-cache headere
   *
   * @return self Fluent interface
   */
   public function noCache(): self
   {
       $this->setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
       $this->setHeader('Pragma', 'no-cache');
       $this->setHeader('Expires', '0');
       
       if ($this->logger) {
           $this->logger->http('No-cache headers added');
       }
       
       return $this;
   }

  // PSR-7 ResponseInterface implementacija

   /**
    * {@inheritdoc}
    */
   public function getProtocolVersion(): string
   {
       return $this->protocolVersion;
   }

  /**
    * {@inheritdoc}
    */
   public function withProtocolVersion($version): \Psr\Http\Message\MessageInterface
   {
       $new = clone $this;
       $new->protocolVersion = $version;
       return $new;
   }

   /**
    * {@inheritdoc}
    */
   public function hasHeader($name): bool
   {
       foreach ($this->headers as $headerName => $value) {
           if (strtolower($headerName) === strtolower($name)) {
               return true;
           }
       }
       return false;
   }

   /**
    * {@inheritdoc}
    */
   public function withHeader($name, $value): \Psr\Http\Message\MessageInterface
   {
       $new = clone $this;
       $new->headers[$name] = is_array($value) ? $value : [$value];
       return $new;
   }

   /**
    * {@inheritdoc}
    */
   public function withAddedHeader($name, $value): \Psr\Http\Message\MessageInterface
   {
       $new = clone $this;
       if (isset($new->headers[$name])) {
           if (is_array($new->headers[$name])) {
               $new->headers[$name][] = $value;
           } else {
               $new->headers[$name] = [$new->headers[$name], $value];
           }
       } else {
           $new->headers[$name] = [$value];
       }
       return $new;
   }

   /**
    * {@inheritdoc}
    */
   public function withoutHeader($name): \Psr\Http\Message\MessageInterface
   {
       $new = clone $this;
       foreach ($new->headers as $headerName => $value) {
           if (strtolower($headerName) === strtolower($name)) {
               unset($new->headers[$headerName]);
           }
       }
       return $new;
   }

   /**
    * {@inheritdoc}
    */
   public function getHeaderLine($name): string
   {
       $values = $this->getHeader($name);
       if (empty($values)) {
           return '';
       }
       return implode(', ', $values);
   }

   /**
    * {@inheritdoc}
    */
   public function withBody(\Psr\Http\Message\StreamInterface $body): \Psr\Http\Message\MessageInterface
   {
       $new = clone $this;
       $new->body = $body;
       $new->content = (string) $body;
       return $new;
   }

   /**
    * {@inheritdoc}
    */
   public function getStatusCode(): int
   {
       return $this->statusCode;
   }

   /**
    * {@inheritdoc}
    */
   public function withStatus($code, $reasonPhrase = ''): \Psr\Http\Message\ResponseInterface
   {
       $new = clone $this;
       $new->statusCode = (int) $code;
       $new->reasonPhrase = $reasonPhrase ?: $this->getReasonPhraseForStatusCode($code);
       return $new;
   }

   /**
    * {@inheritdoc}
    */
   public function getReasonPhrase(): string
   {
       return $this->reasonPhrase;
   }
}