<?php

namespace baseKRIZAN\Http;

use Psr\Http\Message\StreamInterface;

/**
 * Osnovna implementacija PSR-7 StreamInterface
 * 
 * @package baseKRIZAN\Http
 */
class Stream implements StreamInterface
{
    /**
     * @var resource|null Stream resurs
     */
    private $stream;
    
    /**
     * @var bool Je li stream čitljiv
     */
    private bool $readable;
    
    /**
     * @var bool Je li stream zapisiv
     */
    private bool $writable;
    
    /**
     * @var bool Je li stream seekable
     */
    private bool $seekable;
    
    /**
     * @var int|null Veličina streama
     */
    private ?int $size = null;
    
    /**
     * @var string|null URI streama
     */
    private ?string $uri = null;
    
    /**
     * Konstruktor
     *
     * @param string|resource $stream Stream resurs ili string (npr. 'php://input')
     * @param string $mode Mode za otvaranje streama (npr. 'r', 'w', 'a')
     */
    public function __construct($stream, string $mode = 'r')
    {
        if (is_string($stream)) {
            set_error_handler(function () use ($stream) {
                throw new \RuntimeException(sprintf('Unable to open %s stream', $stream));
            });
            
            $resource = @fopen($stream, $mode);
            
            restore_error_handler();
            
            if ($resource === false) {
                throw new \RuntimeException(sprintf('Unable to open %s stream', $stream));
            }
            
            $this->stream = $resource;
        } elseif (is_resource($stream)) {
            $this->stream = $stream;
        } else {
            throw new \InvalidArgumentException('Stream must be a string stream identifier or resource');
        }
        
        $this->determineProperties();
    }
    
    /**
     * Određuje svojstva streama (readable, writable, seekable)
     */
    private function determineProperties(): void
    {
        $metadata = $this->getMetadata();
        $this->uri = $metadata['uri'] ?? null;
        
        // Provjeri je li stream seekable
        $this->seekable = $metadata['seekable'] ?? false;
        
        // Provjeri je li stream readable i writable na temelju mode-a
        $mode = $metadata['mode'] ?? '';
        
        // Readable modes: r, r+, w+, a+, x+, c+
        $this->readable = $mode && (
            strpos($mode, 'r') !== false || 
            strpos($mode, '+') !== false
        );
        
        // Writable modes: w, w+, r+, a, a+, x, x+, c, c+
        $this->writable = $mode && (
            strpos($mode, 'w') !== false ||
            strpos($mode, 'a') !== false ||
            strpos($mode, 'x') !== false ||
            strpos($mode, 'c') !== false ||
            strpos($mode, '+') !== false
        );
    }
    
    /**
     * {@inheritdoc}
     */
    /**
     * {@inheritdoc}
     */
    public function __toString(): string
    {
        try {
            $this->rewind();
            return $this->getContents();
        } catch (\Throwable $e) {
            return '';
        }
    }

    /**
     * {@inheritdoc}
     */
    public function close(): void
    {
        if ($this->stream !== null) {
            if (is_resource($this->stream)) {
                fclose($this->stream);
            }
            $this->stream = null;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function detach()
    {
        $resource = $this->stream;
        $this->stream = null;
        $this->size = null;
        $this->uri = null;
        $this->readable = false;
        $this->writable = false;
        $this->seekable = false;
        
        return $resource;
    }

    /**
     * {@inheritdoc}
     */
    public function getSize(): ?int
    {
        if ($this->stream === null) {
            return null;
        }
        
        // Ako već imamo izračunatu veličinu, vrati je
        if ($this->size !== null) {
            return $this->size;
        }
        
        // Pokušaj dohvatiti veličinu iz stat-a
        $stats = fstat($this->stream);
        $this->size = $stats['size'] ?? null;
        
        return $this->size;
    }

    /**
     * {@inheritdoc}
     */
    public function tell(): int
    {
        if (!is_resource($this->stream)) {
            throw new \RuntimeException('Stream is not a resource');
        }
        
        $position = ftell($this->stream);
        if ($position === false) {
            throw new \RuntimeException('Unable to determine stream position');
        }
        
        return $position;
    }

    /**
     * {@inheritdoc}
     */
    public function eof(): bool
    {
        return !is_resource($this->stream) || feof($this->stream);
    }

    /**
     * {@inheritdoc}
     */
    public function isSeekable(): bool
    {
        return $this->seekable;
    }

    /**
     * {@inheritdoc}
     */
    public function seek($offset, $whence = SEEK_SET): void
    {
        if (!$this->seekable) {
            throw new \RuntimeException('Stream is not seekable');
        }
        
        if (!is_resource($this->stream)) {
            throw new \RuntimeException('Stream is not a resource');
        }
        
        if (fseek($this->stream, $offset, $whence) !== 0) {
            throw new \RuntimeException('Unable to seek to stream position ' . $offset);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function rewind(): void
    {
        $this->seek(0);
    }

    /**
     * {@inheritdoc}
     */
    public function isWritable(): bool
    {
        return $this->writable;
    }

    /**
     * {@inheritdoc}
     */
    public function write($string): int
    {
        if (!$this->writable) {
            throw new \RuntimeException('Stream is not writable');
        }
        
        if (!is_resource($this->stream)) {
            throw new \RuntimeException('Stream is not a resource');
        }
        
        $result = fwrite($this->stream, $string);
        if ($result === false) {
            throw new \RuntimeException('Unable to write to stream');
        }
        
        // Resetiraj veličinu jer smo modificirali stream
        $this->size = null;
        
        return $result;
    }

    /**
     * {@inheritdoc}
     */
    public function isReadable(): bool
    {
        return $this->readable;
    }

    /**
     * {@inheritdoc}
     */
    public function read($length): string
    {
        if (!$this->readable) {
            throw new \RuntimeException('Stream is not readable');
        }
        
        if (!is_resource($this->stream)) {
            throw new \RuntimeException('Stream is not a resource');
        }
        
        $result = fread($this->stream, $length);
        if ($result === false) {
            throw new \RuntimeException('Unable to read from stream');
        }
        
        return $result;
    }

    /**
     * {@inheritdoc}
     */
    public function getContents(): string
    {
        if (!$this->readable) {
            throw new \RuntimeException('Stream is not readable');
        }
        
        if (!is_resource($this->stream)) {
            throw new \RuntimeException('Stream is not a resource');
        }
        
        $contents = stream_get_contents($this->stream);
        if ($contents === false) {
            throw new \RuntimeException('Unable to read stream contents');
        }
        
        return $contents;
    }

    /**
     * {@inheritdoc}
     */
    public function getMetadata($key = null)
    {
        if (!is_resource($this->stream)) {
            return $key ? null : [];
        }
        
        $meta = stream_get_meta_data($this->stream);
        
        if ($key === null) {
            return $meta;
        }
        
        return $meta[$key] ?? null;
    }
    
    /**
     * Desctructor koji automatski zatvara stream
     */
    public function __destruct()
    {
        $this->close();
    }
}