<?php

namespace baseKRIZAN\Http;

use Psr\Http\Message\UriInterface;

/**
 * Implementacija URI-ja prema PSR-7 standardu
 * 
 * @package baseKRIZAN\Http
 */
class Uri implements UriInterface
{
    private string $scheme = '';
    private string $host = '';
    private ?int $port = null;
    private string $user = '';
    private string $password = '';
    private string $path = '';
    private string $query = '';
    private string $fragment = '';

    /**
     * Konstruktor
     *
     * @param string $uri URI kao string
     */
    public function __construct(string $uri = '')
    {
        if (!empty($uri)) {
            $parts = parse_url($uri);
            
            if ($parts === false) {
                throw new \InvalidArgumentException('Invalid URI provided');
            }
            
            $this->scheme = $parts['scheme'] ?? '';
            $this->host = $parts['host'] ?? '';
            $this->port = isset($parts['port']) ? (int)$parts['port'] : null;
            $this->user = $parts['user'] ?? '';
            $this->password = $parts['pass'] ?? '';
            $this->path = $parts['path'] ?? '';
            $this->query = $parts['query'] ?? '';
            $this->fragment = $parts['fragment'] ?? '';
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getScheme(): string
    {
        return $this->scheme;
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthority(): string
    {
        if (empty($this->host)) {
            return '';
        }
        
        $authority = $this->host;
        
        if ($this->port !== null) {
            $authority .= ':' . $this->port;
        }
        
        if (!empty($this->user)) {
            $userInfo = $this->user;
            if (!empty($this->password)) {
                $userInfo .= ':' . $this->password;
            }
            $authority = $userInfo . '@' . $authority;
        }
        
        return $authority;
    }

    /**
     * {@inheritdoc}
     */
    public function getUserInfo(): string
    {
        if (empty($this->user)) {
            return '';
        }
        
        $userInfo = $this->user;
        if (!empty($this->password)) {
            $userInfo .= ':' . $this->password;
        }
        
        return $userInfo;
    }

    /**
     * {@inheritdoc}
     */
    public function getHost(): string
    {
        return $this->host;
    }

    /**
     * {@inheritdoc}
     */
    public function getPort(): ?int
    {
        // Ako je port standardni za scheme, vratiti null
        if ($this->port !== null) {
            if (($this->scheme === 'http' && $this->port === 80) ||
                ($this->scheme === 'https' && $this->port === 443)) {
                return null;
            }
        }
        
        return $this->port;
    }

    /**
     * {@inheritdoc}
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * {@inheritdoc}
     */
    public function getQuery(): string
    {
        return $this->query;
    }

    /**
     * {@inheritdoc}
     */
    public function getFragment(): string
    {
        return $this->fragment;
    }

    /**
     * {@inheritdoc}
     */
    public function withScheme($scheme): UriInterface
    {
        $new = clone $this;
        $new->scheme = strtolower($scheme);
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withUserInfo($user, $password = null): UriInterface
    {
        $new = clone $this;
        $new->user = $user;
        $new->password = $password ?? '';
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withHost($host): UriInterface
    {
        $new = clone $this;
        $new->host = strtolower($host);
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withPort($port): UriInterface
    {
        $new = clone $this;
        $new->port = $port !== null ? (int)$port : null;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withPath($path): UriInterface
    {
        $new = clone $this;
        $new->path = $path;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withQuery($query): UriInterface
    {
        $new = clone $this;
        $new->query = $query;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function withFragment($fragment): UriInterface
    {
        $new = clone $this;
        $new->fragment = $fragment;
        return $new;
    }

    /**
     * {@inheritdoc}
     */
    public function __toString(): string
    {
        $uri = '';
        
        if (!empty($this->scheme)) {
            $uri .= $this->scheme . ':';
        }
        
        $authority = $this->getAuthority();
        if (!empty($authority)) {
            $uri .= '//' . $authority;
        }
        
        $path = $this->path;
        if (!empty($path)) {
            // Osiguraj da path počinje sa /
            if (!empty($authority) && substr($path, 0, 1) !== '/') {
                $path = '/' . $path;
            }
            $uri .= $path;
        }
        
        if (!empty($this->query)) {
            $uri .= '?' . $this->query;
        }
        
        if (!empty($this->fragment)) {
            $uri .= '#' . $this->fragment;
        }
        
        return $uri;
    }
}