<?php

namespace baseKRIZAN\LUKA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Events\EventDispatcher;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * Database Query Hook za snimanje SQL upita
 */
class DatabaseHook
{
    private Logger $logger;
    private MetricsCollector $collector;
    private bool $enabled = true;
    private ?EventDispatcher $eventDispatcher = null;
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, MetricsCollector $collector, ?EventDispatcher $eventDispatcher = null)
    {
        $this->logger = $logger;
        $this->collector = $collector;
        $this->eventDispatcher = $eventDispatcher;
        
        // Registriramo hook na bazu
        $this->registerDatabaseHook();
    }
    
    /**
     * Registrira hook na bazu podataka
     */
    private function registerDatabaseHook(): void
    {
        // Koristimo EventDispatcher ako je dostupan
        if ($this->eventDispatcher) {
            $this->eventDispatcher->addListener('database.query', [$this, 'onQuery']);
            
            // Postavimo eventDispatcher i logger na DatabaseConnection
            if (class_exists('\baseKRIZAN\Database\DatabaseConnection')) {
                $dbConnection = DatabaseConnection::getInstance();
                $dbConnection->setServices($this->eventDispatcher, $this->logger);
                
                $this->logger->luka('Database hook registered using EventDispatcher and connected to DatabaseConnection');
            } else {
                $this->logger->luka('Database hook registered using EventDispatcher but DatabaseConnection not found');
            }
        } else {
            $this->logger->luka('Database hook could not register automatically', [
                'info' => 'No EventDispatcher available - database queries will not be tracked'
            ]);
        }
    }
    
    /**
     * Callback za SQL upite
     */
    public function onQuery(\baseKRIZAN\Events\EventInterface $event): void
    {
        if (!$this->enabled) {
            $this->logger->luka('Database hook disabled');
            return;
        }
        
        // Dohvatimo podatke iz eventa
        $query = $event->get('query', '');
        $executionTime = $event->get('execution_time', 0.0);
        $params = $event->get('params', []);
        
        // Debug: Provjeri podatke iz eventa
        $this->logger->luka('Database hook received query event', [
            'query_preview' => substr($query, 0, 50) . (strlen($query) > 50 ? '...' : ''),
            'execution_time' => $executionTime,
            'has_params' => !empty($params) ? 'yes' : 'no'
        ]);
        
        // Zapisat ćemo u kolektor metrika
        $this->collector->recordDatabaseQuery($query, $executionTime, $params);
    }
    
    /**
     * Uključi snimanje upita
     */
    public function enable(): void
    {
        $this->enabled = true;
    }
    
    /**
     * Isključi snimanje upita
     */
    public function disable(): void
    {
        $this->enabled = false;
    }
}