<?php

namespace baseKRIZAN\LUKA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\LUKA\Storage\MetricsStorage;
use baseKRIZAN\Shared\WebSocketServer;

/**
 * Publishes events to WebSocket subscribers for real-time updates
 */
class EventPublisher
{
    private Logger $logger;
    private MetricsStorage $storage;
    private int $websocketPort;
    private ?WebSocketServer $server = null;
    private bool $serverStarted = false;
    private string $serverPidFile;

    /**
     * Constructor
     */
    public function __construct(Logger $logger, MetricsStorage $storage, int $websocketPort)
    {
        $this->logger = $logger;
        $this->storage = $storage;
        $this->websocketPort = $websocketPort;
        
        // Koristimo stvarnu putanju aplikacije i dodajemo poddirektorij za pohranu
        $lukaPath = \baseKRIZAN\Config\Config::get('paths.logs') . '/luka';
        
        $this->serverPidFile = $lukaPath . '/websocket.pid';
        
        // Try to start the WebSocket server if needed
        $this->ensureServerRunning();
    }

    /**
     * Ensure the WebSocket server is running
     */
    private function ensureServerRunning(): void
    {
        if ($this->isServerRunning()) {
            $this->serverStarted = true;
            return;
        }
        
        // Check if we're in CLI mode - only start server in CLI
        if (php_sapi_name() === 'cli') {
            try {
                $this->startServer();
            } catch (\Exception $e) {
                $this->logger->error('Failed to start WebSocket server', [
                    'error' => $e->getMessage()
                ]);
            }
        } else {
            // In web context, just log a message
            $this->logger->luka('WebSocket server not running. Start manually using CLI command.');
        }
    }

    /**
     * Check if the WebSocket server is already running
     */
    private function isServerRunning(): bool
    {
        if (!file_exists($this->serverPidFile)) {
            return false;
        }
        
        $pid = (int)file_get_contents($this->serverPidFile);
        
        if (!$pid) {
            return false;
        }
        
        // On Linux, check if process exists
        if (PHP_OS_FAMILY === 'Linux') {
            return file_exists("/proc/{$pid}");
        }
        
        // On other systems, try a different approach
        try {
            $result = shell_exec("ps -p {$pid}");
            return strpos($result, (string)$pid) !== false;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Start the WebSocket server
     */
    private function startServer(): void
    {
        // Make sure parent directory exists
        $serverPidDir = dirname($this->serverPidFile);
        if (!is_dir($serverPidDir)) {
            mkdir($serverPidDir, 0755, true);
        }
        
        // Initialize WebSocket server
        $this->server = new WebSocketServer($this->logger, $this->websocketPort);
        
        // Fork and detach the WebSocket server on Linux
        if (PHP_OS_FAMILY === 'Linux' && function_exists('pcntl_fork')) {
            $pid = pcntl_fork();
            
            if ($pid == -1) {
                // Fork failed
                throw new \RuntimeException('Failed to fork WebSocket server process');
            } else if ($pid) {
                // Parent process
                $this->serverStarted = true;
                file_put_contents($this->serverPidFile, $pid);
                $this->logger->luka('WebSocket server forked', [
                    'pid' => $pid,
                    'port' => $this->websocketPort
                ]);
                return;
            } else {
                // Child process
                // Use pcntl_exec if available or just run in same process
                $this->server->run();
                exit(0);
            }
        } else {
            // Non-Linux system, just start in current process for development
            $this->logger->luka('WebSocket server started in the same process (non-detached)', [
                'port' => $this->websocketPort
            ]);
            $this->server->run();
            $this->serverStarted = true;
        }
    }

    /**
     * Publish event to WebSocket clients
     */
    public function publish(string $eventType, array $eventData = []): bool
    {
        if (!$this->serverStarted) {
            return false;
        }
        
        $payload = [
            'type' => $eventType,
            'data' => $eventData,
            'timestamp' => microtime(true)
        ];
        
        try {
            // Use shared implementation to access running server
            return WebSocketServer::publishToAll(json_encode($payload));
        } catch (\Exception $e) {
            $this->logger->error('Failed to publish event to WebSocket', [
                'error' => $e->getMessage(),
                'event_type' => $eventType
            ]);
            return false;
        }
    }
    
    /**
     * Stop the WebSocket server
     */
    public function stopServer(): bool
    {
        if (!file_exists($this->serverPidFile)) {
            return true; // Already stopped
        }
        
        $pid = (int)file_get_contents($this->serverPidFile);
        
        if (!$pid) {
            return true;
        }
        
        // Send termination signal
        if (PHP_OS_FAMILY === 'Linux') {
            // Try using kill command instead of posix functions
            exec("kill {$pid} 2>/dev/null");
            $this->logger->luka('WebSocket server stopping', [
                'pid' => $pid
            ]);
        } else {
            // Windows or other systems
            exec("taskkill /PID {$pid} /F");
        }
        
        // Remove PID file
        @unlink($this->serverPidFile);
        $this->serverStarted = false;
        
        return true;
    }
    
    /**
     * Get WebSocket connection URL for clients
     */
    public function getWebSocketUrl(): string
    {
        $protocol = 'ws';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Adjust host if it includes a port
        if (strpos($host, ':') !== false) {
            $host = explode(':', $host)[0];
        }
        
        return "{$protocol}://{$host}:{$this->websocketPort}";
    }
}