<?php

namespace baseKRIZAN\LUKA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Events\EventDispatcher;
use baseKRIZAN\LUKA\Storage\MetricsStorage;

/**
 * LoggerHook za integraciju s Logger klasom
 */
class LoggerHook
{
    private MetricsStorage $storage;
    private Logger $logger;
    private ?EventDispatcher $eventDispatcher = null;
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, MetricsStorage $storage, ?EventDispatcher $eventDispatcher = null)
    {
        $this->logger = $logger;
        $this->storage = $storage;
        $this->eventDispatcher = $eventDispatcher;
        
        // Registriramo hook na postojeći logger
        $this->registerHook();
    }
    
    /**
     * Registrira hook na logger
     */
    private function registerHook(): void
    {
        if ($this->eventDispatcher) {
            // Dodajemo listener za log događaj
            $this->eventDispatcher->addListener('logger.log', [$this, 'onLog']);
            $this->logger->luka('Logger Hook activated using EventDispatcher');
        } else {
            // Ako nema EventDispatcher-a, informiramo korisnika
            $this->logger->luka('Logger Hook activated - limited functionality without EventDispatcher');
        }
    }
    
    /**
     * Procesira log zapis i šalje ga u LUKA metriku
     */
    public function onLog(\baseKRIZAN\Events\EventInterface $event): void
    {
        // Dohvatimo podatke iz eventa
        $level = $event->get('level', 'info');
        $message = $event->get('message', '');
        $component = $event->get('component', 'System');
        $context = $event->get('context', []);
        
        // Konvertiraj log level u format za LUKA
        $logType = $this->mapLogLevel($level);
        
        // Pošalji u storage
        $this->storage->storeSystemLog($logType, $component, $message, $context);
    }
    
    /**
     * Procesira log zapis direktno
     */
    public function processLog(string $level, string $message, string $component, array $context = []): void
    {
        // Konvertiraj log level u format za LUKA
        $logType = $this->mapLogLevel($level);
        
        // Pošalji u storage
        $this->storage->storeSystemLog($logType, $component, $message, $context);
    }
    
    /**
     * Mapira PSR log level u LUKA log type
     */
    private function mapLogLevel(string $level): string
    {
        $level = strtolower($level);
        
        switch ($level) {
            case 'emergency':
            case 'alert':
            case 'critical':
            case 'error':
                return 'error';
            
            case 'warning':
                return 'warning';
            
            case 'notice':
            case 'info':
                return 'info';
            
            case 'debug':
            default:
                return 'debug';
        }
    }
}