<?php

namespace baseKRIZAN\LUKA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\LUKA\MetricsCollector;

/**
 * Logger proxy for capturing logs and sending them to LUKA metrics
 */
class LoggerProxy
{
    private static ?Logger $originalLogger = null;
    private static ?MetricsCollector $metricsCollector = null;
    
    /**
     * Initialize the proxy with the original logger and metrics collector
     */
    public static function initialize(Logger $logger, MetricsCollector $collector): void
    {
        self::$originalLogger = $logger;
        self::$metricsCollector = $collector;
    }
    
    /**
     * Intercept log calls and forward them to LUKA metrics
     */
    public static function captureLog($level, $message, array $context = []): void
    {
        if (self::$metricsCollector) {
            // Convert log level to our format
            $logType = self::mapLogLevel($level);
            
            // Extract component from context or default to 'System'
            $component = $context['component'] ?? ($context['category'] ?? 'System');
            
            // Store in database via MetricsCollector
            self::$metricsCollector->storeSystemLog($logType, $component, $message, $context);
        }
    }
    
    /**
     * Map PSR Log level to our system log types
     */
    private static function mapLogLevel(string $level): string
    {
        $level = strtolower($level);
        
        switch ($level) {
            case 'emergency':
            case 'alert':
            case 'critical':
            case 'error':
                return 'error';
            
            case 'warning':
                return 'warning';
            
            case 'notice':
            case 'info':
                return 'info';
            
            case 'debug':
            default:
                return 'debug';
        }
    }
}