<?php

namespace baseKRIZAN\LUKA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Services\Container;
use baseKRIZAN\LUKA\Storage\MetricsStorage;
use baseKRIZAN\LUKA\Storage\DatabaseStorage;
use baseKRIZAN\LUKA\MetricsCollector;
use baseKRIZAN\LUKA\EventPublisher;
use baseKRIZAN\Http\Middleware\MiddlewareRegistry;

/**
 * Core management class for the LUKA Digital Twin system
 * Handles initialization, configuration, and coordination between components
 */
class LukaManager
{
    private static ?self $instance = null;
    private Logger $logger;
    private Container $container;
    private MetricsStorage $storage;
    private MetricsCollector $collector;
    private ?EventPublisher $publisher = null;
    private bool $isEnabled = false;
    private bool $debugMode = false;
    private array $config = [];

    /**
     * Get singleton instance
     */
    public static function getInstance(Logger $logger, Container $container): self
    {
        if (self::$instance === null) {
            self::$instance = new self($logger, $container);
        }
        return self::$instance;
    }

    /**
     * Private constructor for singleton pattern
     */
    public function __construct(
        Logger $logger, 
        Container $container,
        ?MetricsStorage $storage = null
    ) {
        $this->logger = $logger;
        $this->container = $container;
        
        // Load configuration
        $this->loadConfig();
        
        // Configure storage
        $this->initializeStorage();
        
        // Initialize components if enabled
        if ($this->isEnabled()) {
            $this->initializeComponents();
        }
        
        $this->logger->luka('Digital Twin ' . ($this->isEnabled() ? 'enabled' : 'disabled'), [
            'debug_mode' => $this->debugMode
        ]);
    
        // Uklonite ovaj dio - registracija middleware-a će se pozvati eksplicitno iz CoreModuleServiceProvider
        /*
        if ($this->isEnabled() && $this->container->has('middlewareRegistry')) {
            $middlewareRegistry = $this->container->get('middlewareRegistry');
            $this->registerMiddleware($middlewareRegistry);
        }
        */
    
        // Register module-specific components if enabled
        if ($this->isEnabled()) {
            $this->initializeModuleSpecificComponents();
        }
    }

    /**
     * Register middleware for metrics collection
     * 
     * @param MiddlewareRegistry $registry MiddlewareRegistry instanca
     * @return void
     */
    public function registerMiddleware(MiddlewareRegistry $registry): void 
    {
        if (!$this->isEnabled()) {
            return;
        }

        $registry->addMiddleware(
            \baseKRIZAN\Http\Middleware\LukaMiddleware::class, 
            [
                'group' => 'global', 
                'priority' => 15,
                'requiredServices' => ['logger'],
                'conditionalLoad' => true
            ]
        );

        $this->logger->luka('LUKA middleware registration completed');
    }

    /**
     * Load configuration settings
     */
    private function loadConfig(): void
    {
        $this->isEnabled = \baseKRIZAN\Config\Config::get('luka.enabled', false);
        $this->debugMode = \baseKRIZAN\Config\Config::get('LUKA_DEBUG', false);
        
        $this->config = [
            'storage_type' => \baseKRIZAN\Config\Config::get('luka.storage_type', 'database'),
            'websocket_enabled' => \baseKRIZAN\Config\Config::get('luka.websocket_enabled', false),
            'websocket_port' => \baseKRIZAN\Config\Config::get('luka.websocket_port', 8080),
            'metrics_retention_days' => \baseKRIZAN\Config\Config::get('LUKA_METRICS_RETENTION_DAYS', 7),
            'dashboard_path' => \baseKRIZAN\Config\Config::get('LUKA_DASHBOARD_PATH', '/luka/digital-twin'),
            'sampling_rate' => \baseKRIZAN\Config\Config::get('luka.sampling_rate', 100), // Percentage of requests to monitor
            'track_database' => \baseKRIZAN\Config\Config::get('LUKA_TRACK_DATABASE', true),
            'track_template' => \baseKRIZAN\Config\Config::get('LUKA_TRACK_TEMPLATE', true),
            'track_session' => \baseKRIZAN\Config\Config::get('LUKA_TRACK_SESSION', true),
            'track_assets' => \baseKRIZAN\Config\Config::get('LUKA_TRACK_ASSETS', true)
        ];
    }

    /**
     * Initialize module-specific components
     *
     * @return void
     */
    private function initializeModuleSpecificComponents(): void
    {
        $this->logger->luka('Initializing LUKA specific components');

        // Specifično za LUKA modul
        if (method_exists($this, 'getMetricsCollector')) {
            $metricsCollector = $this->getMetricsCollector();
            if ($metricsCollector) {
                $this->container->register('lukaMetricsCollector', $metricsCollector);
            }
        }

        // Registriraj kontroler ako postoji
        if (!$this->container->has('lukaController') && class_exists('\\Controllers\\LUKA')) {
            $this->container->register('lukaController', function($container) {
                return new \Controllers\LUKA($container);
            });
        }

        $this->logger->luka('LUKA module specific components initialized');
    }
    
    /**
     * Initialize core components
     */
    private function initializeComponents(): void
    {
        // Debug: Praćenje inicijalizacije
        $this->logger->luka('Initializing components');
        
        // Initialize storage
        $this->initializeStorage();
        
        // Initialize metrics collector
        $this->collector = new MetricsCollector($this->logger, $this->storage, $this->config);
        $this->container->register('lukaMetricsCollector', $this->collector);
        
        // Initialize event publisher if WebSocket is enabled
        if ($this->config['websocket_enabled']) {
            $this->publisher = new EventPublisher(
                $this->logger,
                $this->storage,
                $this->config['websocket_port']
            );
            $this->container->register('lukaEventPublisher', $this->publisher);
        }
        
        // Debug: Prije hookova
        $this->logger->luka('Setting up hooks', [
            'has_event_dispatcher' => $this->container->has('eventDispatcher') ? 'yes' : 'no'
        ]);
        
        // Hook logger to capture system logs
        $this->hookLogger();
        
        // Initialize database hook
        $this->initializeDatabaseHook();
    }

    /**
     * Initialize storage based on configuration
     */
    private function initializeStorage(): void
    {
        // Remove the 'sqlite' case
        switch ($this->config['storage_type']) {
            case 'database':
            default:
                $this->storage = new DatabaseStorage(
                    $this->logger,
                    $this->config['metrics_retention_days']
                );
                break;
        }
        
        $this->container->register('lukaMetricsStorage', $this->storage);
    }

    /**
     * Hook logger to capture system logs
     */
    private function hookLogger(): void
    {
        if (!isset($this->logger) || !isset($this->storage)) {
            return;
        }
        
        // Dohvati EventDispatcher iz container-a ako postoji
        $eventDispatcher = null;
        if ($this->container->has('eventDispatcher')) {
            $eventDispatcher = $this->container->get('eventDispatcher');
        }
        
        // Sada ćemo stvoriti LoggerHook koji će povezati logger i LUKA metriku
        $loggerHook = new \baseKRIZAN\LUKA\LoggerHook(
            $this->logger, 
            $this->storage,
            $eventDispatcher
        );
        
        // Registrirati hook u container kako bi bio dostupan kroz aplikaciju
        $this->container->register('lukaLoggerHook', $loggerHook);
        
        $this->logger->luka('Logger integration enabled', [
            'metrics_storage' => get_class($this->storage)
        ]);
    }

    /**
     * Initialize database hook
     */
    private function initializeDatabaseHook(): void
    {
        if (!isset($this->collector) || !($this->config['track_database'] ?? true)) {
            return;
        }
        
        // Dohvati EventDispatcher iz container-a ako postoji
        $eventDispatcher = null;
        if ($this->container->has('eventDispatcher')) {
            $eventDispatcher = $this->container->get('eventDispatcher');
        }
        
        // Stvori database hook
        $databaseHook = new \baseKRIZAN\LUKA\DatabaseHook(
            $this->logger, 
            $this->collector,
            $eventDispatcher
        );
        
        // Registriraj u container
        $this->container->register('lukaDatabaseHook', $databaseHook);
        
        $this->logger->luka('Database hook enabled');
    }
    
    /**
     * Record application event for tracking
     */
    public function recordEvent(string $eventType, array $eventData = []): void
    {
        if (!$this->isEnabled) {
            return;
        }
        
        try {
            $this->collector->recordEvent($eventType, $eventData);
            
            // If WebSocket is enabled, publish the event
            if ($this->publisher !== null) {
                $this->publisher->publish($eventType, $eventData);
            }
        } catch (\Exception $e) {
            // Just log the error but don't throw - we don't want to disrupt normal operation
            $this->logger->error('Error recording event', [
                'event_type' => $eventType,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Get metrics for a specific time period
     */
    public function getMetrics(string $metricType, \DateTime $from, \DateTime $to): array
    {
        if (!$this->isEnabled) {
            return [];
        }
        
        return $this->storage->getMetrics($metricType, $from, $to);
    }

    /**
     * Get current application state information
     */
    public function getApplicationState(): array
    {
        if (!$this->isEnabled) {
            return ['status' => 'LUKA Digital Twin disabled'];
        }
        
        return [
            'status' => 'active',
            'metrics_count' => $this->storage->getMetricsCount(),
            'websocket_status' => $this->publisher ? 'active' : 'disabled',
            'uptime' => $this->getSystemUptime(),
            'php_version' => PHP_VERSION,
            'memory_usage' => $this->getMemoryUsage(),
            'debug_mode' => $this->debugMode
        ];
    }
    
    /**
     * Get system uptime
     */
    private function getSystemUptime(): string
    {
        if (PHP_OS_FAMILY === 'Linux') {
            $uptime = shell_exec('uptime -p');
            return $uptime ? trim($uptime) : 'unknown';
        }
        
        return 'unknown';
    }
    
    /**
     * Get current memory usage in a formatted string
     */
    private function getMemoryUsage(): string
    {
        $memory = memory_get_usage(true);
        if ($memory < 1024) {
            return $memory . ' B';
        } elseif ($memory < 1048576) {
            return round($memory / 1024, 2) . ' KB';
        } else {
            return round($memory / 1048576, 2) . ' MB';
        }
    }

    /**
     * Get the metrics collector instance
     * 
     * @return MetricsCollector|null
     */
    public function getMetricsCollector(): ?MetricsCollector
    {
        return $this->collector ?? null;
    }
    
    /**
     * Check if LUKA is enabled
     */
    public function isEnabled(): bool
    {
        return $this->isEnabled;
    }
    
    /**
     * Get configuration settings
     */
    public function getConfig(): array
    {
        return $this->config;
    }
}