<?php

namespace baseKRIZAN\LUKA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\LUKA\Storage\MetricsStorage;

/**
 * Collects performance metrics and application state data
 */
class MetricsCollector
{
    private Logger $logger;
    private MetricsStorage $storage;
    private array $config;
    private array $currentRequestMetrics = [];
    private float $requestStartTime;
    private int $requestStartMemory;
    private array $eventCounts = [];
    private array $pendingEvents = [];
    
    // Tracking WebSocket metrics
    private int $wsMessagesSent = 0;
    private int $wsMessagesReceived = 0;
    private int $wsBytesSent = 0;
    private int $wsBytesReceived = 0;
    private int $wsErrors = 0;
    private array $uniqueQueries = [];

    /**
     * Constructor
     */
    public function __construct(Logger $logger, MetricsStorage $storage, array $config)
    {
        $this->logger = $logger;
        $this->storage = $storage;
        $this->config = $config;
        $this->requestStartTime = microtime(true);
        $this->requestStartMemory = memory_get_usage(true);
        
        // Register shutdown function to ensure metrics are saved
        register_shutdown_function([$this, 'onShutdown']);
        
        // Register logger extension
        $this->registerLoggerExtension();
        
        $this->logger->luka('Metrics Collector initialized');
    }
    
    /**
     * Register extension to capture system logs
     */
    private function registerLoggerExtension(): void
    {
        // Hook into the logger to capture logs to our storage
        $captureHandler = function(string $level, string $message, array $context = []) {
            // Convert log level to our format
            $logType = $this->mapLogLevel($level);
            
            // Extract component from context or default to 'System'
            $component = $context['component'] ?? ($context['category'] ?? 'System');
            
            // Store in database
            $this->storeSystemLog($logType, $component, $message, $context);
        };
        
        // If we could directly extend Logger class, we would add this method there
        // For now, we'll rely on the logs being captured when we directly log
        // through the MetricsCollector class
    }
    
    /**
     * Map PSR Log level to our system log types
     */
    private function mapLogLevel(string $level): string
    {
        $level = strtolower($level);
        
        switch ($level) {
            case 'emergency':
            case 'alert':
            case 'critical':
            case 'error':
                return 'error';
            
            case 'warning':
                return 'warning';
            
            case 'notice':
            case 'info':
                return 'info';
            
            case 'debug':
            default:
                return 'debug';
        }
    }
    
    /**
     * Store system log
     */
    public function storeSystemLog(string $logType, string $component, string $message, ?array $context = null): void
    {
        if (empty($this->config['track_system_logs'] ?? true)) {
            return;
        }
        
        // Store in database
        $this->storage->storeSystemLog($logType, $component, $message, $context);
    }

    /**
     * Start tracking a new request
     */
    public function startRequest(string $path, string $method, array $headers = []): void
    {
        // Check if we should sample this request based on sampling rate
        if (rand(1, 100) > $this->config['sampling_rate']) {
            return;
        }

        $this->currentRequestMetrics = [
            'path' => $path,
            'method' => $method,
            'start_time' => microtime(true),
            'timestamp' => date('Y-m-d H:i:s'),
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'session_id' => session_id() ?: 'none',
            'user_id' => $_SESSION['user_id'] ?? null,
            'components' => [],
            'database_queries' => [],
            'memory_start' => memory_get_usage(true),
            'events' => []
        ];
        
        // Reset unique queries tracking for this request
        $this->uniqueQueries = [];
    }

    /**
     * Record component execution time
     */
    public function recordComponentExecution(string $component, float $executionTime, array $metadata = []): void
    {
        if (empty($this->currentRequestMetrics)) {
            return;
        }
        
        if (!isset($this->currentRequestMetrics['components'][$component])) {
            $this->currentRequestMetrics['components'][$component] = [
                'total_time' => 0,
                'count' => 0,
                'instances' => []
            ];
        }
        
        $this->currentRequestMetrics['components'][$component]['total_time'] += $executionTime;
        $this->currentRequestMetrics['components'][$component]['count']++;
        
        // Store individual execution instance
        $this->currentRequestMetrics['components'][$component]['instances'][] = [
            'execution_time' => $executionTime,
            'timestamp' => microtime(true),
            'metadata' => $metadata
        ];
    }

    /**
     * Record database query
     */
    public function recordDatabaseQuery(string $query, float $executionTime, array $params = []): void
    {
        if (empty($this->currentRequestMetrics) || !($this->config['track_database'] ?? true)) {
            return;
        }
        
        // Anonymize sensitive data in queries and parameters
        $anonymizedQuery = $this->anonymizeSensitiveData($query);
        $anonymizedParams = $this->anonymizeQueryParams($params);
        
        $this->currentRequestMetrics['database_queries'][] = [
            'query' => $anonymizedQuery,
            'execution_time' => $executionTime,
            'params' => $anonymizedParams,
            'timestamp' => microtime(true)
        ];
        
        // Add to unique queries for tracking in database_queries table
        $queryHash = md5($anonymizedQuery);
        
        // Only track each unique query once per request
        if (!isset($this->uniqueQueries[$queryHash])) {
            $this->uniqueQueries[$queryHash] = [
                'query' => $anonymizedQuery,
                'execution_time' => $executionTime,
                'params' => empty($anonymizedParams) ? null : $anonymizedParams
            ];
        }
    }
    
    /**
     * Record template rendering
     */
    public function recordTemplateRendering(string $template, float $renderTime, int $outputSize): void
    {
        if (empty($this->currentRequestMetrics) || !($this->config['track_template'] ?? true)) {
            return;
        }
        
        if (!isset($this->currentRequestMetrics['templates'])) {
            $this->currentRequestMetrics['templates'] = [];
        }
        
        $this->currentRequestMetrics['templates'][] = [
            'template' => $template,
            'render_time' => $renderTime,
            'output_size' => $outputSize,
            'timestamp' => microtime(true)
        ];
    }
    
    /**
     * Record session operation
     */
    public function recordSessionOperation(string $operation, float $operationTime): void
    {
        if (empty($this->currentRequestMetrics) || !($this->config['track_session'] ?? true)) {
            return;
        }
        
        if (!isset($this->currentRequestMetrics['session_operations'])) {
            $this->currentRequestMetrics['session_operations'] = [];
        }
        
        $this->currentRequestMetrics['session_operations'][] = [
            'operation' => $operation,
            'operation_time' => $operationTime,
            'timestamp' => microtime(true)
        ];
    }
    
    /**
     * Record asset loading
     */
    public function recordAssetLoading(string $asset, float $loadTime, int $size): void
    {
        if (empty($this->currentRequestMetrics) || !($this->config['track_assets'] ?? true)) {
            return;
        }
        
        if (!isset($this->currentRequestMetrics['assets'])) {
            $this->currentRequestMetrics['assets'] = [];
        }
        
        $this->currentRequestMetrics['assets'][] = [
            'asset' => $asset,
            'load_time' => $loadTime,
            'size' => $size,
            'timestamp' => microtime(true)
        ];
    }

    /**
     * Record WebSocket message sent
     */
    public function recordWebSocketMessageSent(string $message, int $size): void
    {
        $this->wsMessagesSent++;
        $this->wsBytesSent += $size;
        
        // Store metrics periodically (every 50 messages)
        if ($this->wsMessagesSent % 50 === 0) {
            $this->storeWebSocketMetrics();
        }
    }
    
    /**
     * Record WebSocket message received
     */
    public function recordWebSocketMessageReceived(string $message, int $size): void
    {
        $this->wsMessagesReceived++;
        $this->wsBytesReceived += $size;
        
        // Store metrics periodically (every 50 messages)
        if ($this->wsMessagesReceived % 50 === 0) {
            $this->storeWebSocketMetrics();
        }
    }
    
    /**
     * Record WebSocket error
     */
    public function recordWebSocketError(string $error): void
    {
        $this->wsErrors++;
        
        // Log the error
        $this->logger->error('WebSocket error: ' . $error);
        
        // Store system log
        $this->storeSystemLog('error', 'WebSocket', $error);
    }
    
    /**
     * Store WebSocket metrics to database
     */
    private function storeWebSocketMetrics(): void
    {
        if (!($this->config['websocket_enabled'] ?? false)) {
            return;
        }
        
        // Get count of active clients - we'd have to get this from the WebSocketServer
        $connectedClients = $this->getConnectedClientsCount();
        
        // Store in database
        $this->storage->storeWebsocketMetrics(
            $connectedClients,
            $this->wsMessagesSent,
            $this->wsMessagesReceived,
            $this->wsBytesSent,
            $this->wsBytesReceived,
            $this->wsErrors
        );
        
        // Reset counters (but keep client count)
        $this->wsMessagesSent = 0;
        $this->wsMessagesReceived = 0;
        $this->wsBytesSent = 0;
        $this->wsBytesReceived = 0;
        $this->wsErrors = 0;
    }
    
    /**
     * Get count of connected WebSocket clients
     */
    private function getConnectedClientsCount(): int
    {
        // In real implementation, this would get the count from the WebSocketServer
        // For now, return a placeholder value
        return 0;
    }

    /**
     * Record application event
     */
    public function recordEvent(string $eventType, array $eventData = []): void
    {
        if (!isset($this->eventCounts[$eventType])) {
            $this->eventCounts[$eventType] = 0;
        }
        
        $this->eventCounts[$eventType]++;
        
        // Add event to current request if request metrics are being collected
        if (!empty($this->currentRequestMetrics)) {
            $this->currentRequestMetrics['events'][] = [
                'type' => $eventType,
                'data' => $eventData,
                'timestamp' => microtime(true)
            ];
        } else {
            // Store for later processing if outside of a request context
            $this->pendingEvents[] = [
                'type' => $eventType,
                'data' => $eventData,
                'timestamp' => microtime(true)
            ];
        }
        
        // Store event in storage
        $this->storage->storeEvent($eventType, $eventData);
    }

    /**
     * Complete request tracking and store metrics
     */
    public function completeRequest(int $statusCode): void
    {
        if (empty($this->currentRequestMetrics)) {
            return;
        }
        
        if (isset($this->currentRequestMetrics['end_time'])) {
            $this->logger->luka('Request already completed, skipping duplicate completion', [
                'path' => $this->currentRequestMetrics['path'] ?? 'unknown'
            ]);
            return;
        }
        
        $endTime = microtime(true);
        $endMemory = memory_get_usage(true);
        
        $this->currentRequestMetrics['end_time'] = $endTime;
        $this->currentRequestMetrics['total_time'] = $endTime - $this->currentRequestMetrics['start_time'];
        $this->currentRequestMetrics['memory_end'] = $endMemory;
        $this->currentRequestMetrics['memory_peak'] = memory_get_peak_usage(true);
        $this->currentRequestMetrics['memory_used'] = $endMemory - $this->currentRequestMetrics['memory_start'];
        $this->currentRequestMetrics['status_code'] = $statusCode;
        
        // Store the completed metrics
        $this->storage->storeRequestMetrics($this->currentRequestMetrics);
        
        // Process unique database queries
        $this->processUniqueQueries();
        
        // Mark request as fully processed to avoid double storing in shutdown handler
        $this->currentRequestMetrics['_fully_processed'] = true;
        
        $this->logger->luka('Request metrics stored successfully', [
            'path' => $this->currentRequestMetrics['path'] ?? 'unknown'
        ]);
    }
    
    /**
     * Process and store unique database queries
     */
    private function processUniqueQueries(): void
    {
        if (empty($this->uniqueQueries)) {
            return;
        }
        
        // Process each unique query in this request
        foreach ($this->uniqueQueries as $queryData) {
            $this->processDatabaseQuery(
                $queryData['query'],
                $queryData['execution_time'],
                $queryData['params']
            );
        }
        
        // Clear the unique queries array
        $this->uniqueQueries = [];
    }
    
    /**
     * Process a single database query
     */
    private function processDatabaseQuery(string $query, float $executionTime, ?array $params = null): void
    {
        // This is just a placeholder - the actual storage already handles this in DatabaseStorage
        // We've implemented this logic in the DatabaseStorage::processDatabaseQueries method
    }
    
    /**
     * Process any pending events
     */
    private function processPendingEvents(): void
    {
        if (empty($this->pendingEvents)) {
            return;
        }
        
        foreach ($this->pendingEvents as $event) {
            $this->storage->storeEvent($event['type'], $event['data']);
        }
        
        $this->pendingEvents = [];
    }
 
    /**
     * Anonymize sensitive data in SQL queries
     */
    private function anonymizeSensitiveData(string $query): string
    {
        // Replace password values
        $query = preg_replace("/password\s*=\s*'[^']*'/i", "password='***'", $query);
        $query = preg_replace('/password\s*=\s*"[^"]*"/i', 'password="***"', $query);
        
        // Replace email addresses
        $query = preg_replace('/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/', 'email@redacted.com', $query);
        
        // Replace credit card numbers
        $query = preg_replace('/\b(?:\d[ -]*?){13,16}\b/', 'XXXX-XXXX-XXXX-XXXX', $query);
        
        return $query;
    }
    
    /**
     * Anonymize query parameters
     */
    private function anonymizeQueryParams(array $params): array
    {
        $sensitiveKeys = ['password', 'token', 'api_key', 'secret', 'credit_card', 'card_number'];
        
        foreach ($params as $key => $value) {
            if (is_string($key) && in_array(strtolower($key), $sensitiveKeys)) {
                $params[$key] = '***';
            } elseif (is_array($value)) {
                $params[$key] = $this->anonymizeQueryParams($value);
            }
        }
        
        return $params;
    }
    
    /**
     * Shutdown handler to ensure metrics are saved
     */
    public function onShutdown(): void
    {
        // Process pending events
        $this->processPendingEvents();
        
        // If request wasn't completed normally, try to capture metrics now
        if (!empty($this->currentRequestMetrics)) {
            $statusCode = http_response_code() ?: 0;
            $this->completeRequest($statusCode);
        }
        
        // Store event counts
        foreach ($this->eventCounts as $eventType => $count) {
            $this->storage->updateEventCount($eventType, $count);
        }
        
        // Store WebSocket metrics if needed
        if (($this->wsMessagesSent > 0 || $this->wsMessagesReceived > 0) && 
            ($this->config['websocket_enabled'] ?? false)) {
            $this->storeWebSocketMetrics();
        }
    }
    
    /**
     * Get event counts for all recorded event types
     */
    public function getEventCounts(): array
    {
        return $this->eventCounts;
    }
 }