<?php

namespace baseKRIZAN\LUKA\Storage;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * MySQL/MariaDB implementation of metrics storage
 */
class DatabaseStorage implements MetricsStorage
{
    private Logger $logger;
    private int $retentionDays;
    private DatabaseConnection $dbConnection;
    private bool $initialized = false;
    
    // Database tables
    private string $requestsTable = 'security_luka_requests';
    private string $eventsTable = 'security_luka_events';
    private string $eventCountsTable = 'security_luka_event_counts';
    private string $summaryTable = 'security_luka_summary_stats';
    private string $databaseQueriesTable = 'security_luka_database_queries';
    private string $systemLogsTable = 'security_luka_system_logs';
    private string $websocketMetricsTable = 'security_luka_websocket_metrics';
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, int $retentionDays)
    {
        $this->logger = $logger;
        $this->retentionDays = $retentionDays;
        
        try {
            $this->dbConnection = DatabaseConnection::getInstance();
        } catch (\Throwable $e) {
            $this->logger->error('Failed to create database connection for storage', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
        
        // Initialize database tables
        $this->initialize();
        
        // Schedule cleanup of old data
        if (rand(1, 100) <= 5) { // 5% chance to run cleanup on initialization
            $this->cleanupOldData($this->retentionDays);
        }
    }

    /**
     * Initialize database tables
     */
    public function initialize(): bool
    {
        if ($this->initialized) {
            return false;
        }
        
        try {
            // Create requests table
            $this->createRequestsTable();
            
            // Create events table
            $this->createEventsTable();
            
            // Create event_counts table
            $this->createEventCountsTable();
            
            // Create summary_stats table
            $this->createSummaryStatsTable();
            
            // Create database_queries table
            $this->createDatabaseQueriesTable();
            
            // Create system_logs table
            $this->createSystemLogsTable();
            
            // Create websocket_metrics table
            $this->createWebsocketMetricsTable();
            
            $this->initialized = true;
            $this->logger->luka('Metrics tables initialized');
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to initialize metrics tables', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Create requests table
     */
    private function createRequestsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->requestsTable}` (
            `id` BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `path` VARCHAR(255) NOT NULL,
            `method` VARCHAR(10) NOT NULL,
            `start_time` DOUBLE NOT NULL,
            `end_time` DOUBLE NOT NULL,
            `total_time` DOUBLE NOT NULL,
            `timestamp` DATETIME NOT NULL,
            `ip` VARCHAR(45),
            `user_agent` TEXT,
            `session_id` VARCHAR(128),
            `user_id` INT UNSIGNED,
            `status_code` INT UNSIGNED,
            `memory_start` BIGINT UNSIGNED,
            `memory_end` BIGINT UNSIGNED,
            `memory_peak` BIGINT UNSIGNED,
            `memory_used` BIGINT UNSIGNED,
            `components_data` JSON,
            `database_queries` JSON,
            `templates_data` JSON,
            `session_operations` JSON,
            `assets_data` JSON,
            `events_data` JSON,
            INDEX `idx_requests_timestamp` (`timestamp`),
            INDEX `idx_requests_path` (`path`),
            INDEX `idx_requests_status_code` (`status_code`),
            INDEX `idx_requests_session_id` (`session_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }
    
    /**
     * Create events table
     */
    private function createEventsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->eventsTable}` (
            `id` BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `event_type` VARCHAR(50) NOT NULL,
            `event_data` JSON,
            `timestamp` DATETIME NOT NULL,
            INDEX `idx_events_type` (`event_type`),
            INDEX `idx_events_timestamp` (`timestamp`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }
    
    /**
     * Create event counts table
     */
    private function createEventCountsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->eventCountsTable}` (
            `event_type` VARCHAR(50) PRIMARY KEY,
            `count` BIGINT UNSIGNED NOT NULL DEFAULT 0,
            `last_updated` DATETIME NOT NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }
    
    /**
     * Create summary stats table
     */
    private function createSummaryStatsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->summaryTable}` (
            `stat_key` VARCHAR(50) PRIMARY KEY,
            `stat_value` TEXT NOT NULL,
            `last_updated` DATETIME NOT NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }
    
    /**
     * Create database queries table
     */
    private function createDatabaseQueriesTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->databaseQueriesTable}` (
            `id` BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `query_text` TEXT NOT NULL,
            `first_seen` DATETIME NOT NULL,
            `last_seen` DATETIME NOT NULL,
            `avg_execution_time` DOUBLE NOT NULL,
            `max_execution_time` DOUBLE NOT NULL,
            `min_execution_time` DOUBLE NOT NULL,
            `execution_count` BIGINT UNSIGNED NOT NULL,
            `parameters_sample` JSON,
            `query_hash` VARCHAR(64) NOT NULL,
            INDEX `idx_query_hash` (`query_hash`),
            INDEX `idx_last_seen` (`last_seen`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }
    
    /**
     * Create system logs table
     */
    private function createSystemLogsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->systemLogsTable}` (
            `id` BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `timestamp` DATETIME NOT NULL,
            `log_type` ENUM('error', 'warning', 'info', 'debug') NOT NULL,
            `component` VARCHAR(50) NOT NULL,
            `message` TEXT NOT NULL,
            `context` JSON,
            INDEX `idx_timestamp` (`timestamp`),
            INDEX `idx_log_type` (`log_type`),
            INDEX `idx_component` (`component`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }
    
    /**
     * Create websocket metrics table
     */
    private function createWebsocketMetricsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->websocketMetricsTable}` (
            `id` BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `timestamp` DATETIME NOT NULL,
            `connected_clients` INT UNSIGNED NOT NULL,
            `messages_sent` BIGINT UNSIGNED NOT NULL,
            `messages_received` BIGINT UNSIGNED NOT NULL,
            `bytes_sent` BIGINT UNSIGNED NOT NULL,
            `bytes_received` BIGINT UNSIGNED NOT NULL,
            `errors_count` INT UNSIGNED NOT NULL,
            INDEX `idx_timestamp` (`timestamp`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->dbConnection->execute($sql);
    }

    /**
     * Store request metrics
     */
    public function storeRequestMetrics(array $metrics): bool
    {
        if (!$this->initialized) {
            return false;
        }
        
        try {
            $sql = "INSERT INTO `{$this->requestsTable}` (
                    path, method, start_time, end_time, total_time, timestamp,
                    ip, user_agent, session_id, user_id, status_code,
                    memory_start, memory_end, memory_peak, memory_used,
                    components_data, database_queries, templates_data,
                    session_operations, assets_data, events_data
                ) VALUES (
                    :path, :method, :start_time, :end_time, :total_time, :timestamp,
                    :ip, :user_agent, :session_id, :user_id, :status_code,
                    :memory_start, :memory_end, :memory_peak, :memory_used,
                    :components_data, :database_queries, :templates_data,
                    :session_operations, :assets_data, :events_data
                )";
            
            // Prepare serialized data
            $componentsData = isset($metrics['components']) ? json_encode($metrics['components']) : null;
            $databaseQueries = isset($metrics['database_queries']) ? json_encode($metrics['database_queries']) : null;
            $templatesData = isset($metrics['templates']) ? json_encode($metrics['templates']) : null;
            $sessionOperations = isset($metrics['session_operations']) ? json_encode($metrics['session_operations']) : null;
            $assetsData = isset($metrics['assets']) ? json_encode($metrics['assets']) : null;
            $eventsData = isset($metrics['events']) ? json_encode($metrics['events']) : null;
            
            $params = [
                'path' => $metrics['path'],
                'method' => $metrics['method'],
                'start_time' => $metrics['start_time'],
                'end_time' => $metrics['end_time'],
                'total_time' => $metrics['total_time'],
                'timestamp' => $metrics['timestamp'],
                'ip' => $metrics['ip'],
                'user_agent' => $metrics['user_agent'],
                'session_id' => $metrics['session_id'],
                'user_id' => $metrics['user_id'],
                'status_code' => $metrics['status_code'],
                'memory_start' => $metrics['memory_start'],
                'memory_end' => $metrics['memory_end'],
                'memory_peak' => $metrics['memory_peak'],
                'memory_used' => $metrics['memory_used'],
                'components_data' => $componentsData,
                'database_queries' => $databaseQueries,
                'templates_data' => $templatesData,
                'session_operations' => $sessionOperations,
                'assets_data' => $assetsData,
                'events_data' => $eventsData
            ];
            
            $result = $this->dbConnection->execute($sql, $params);
            
            // Process database queries if available
            if (!empty($metrics['database_queries']) && is_array($metrics['database_queries'])) {
                $this->processDatabaseQueries($metrics['database_queries']);
            }
            
            // Update summary stats
            $this->updateSummaryStats();
            
            return $result > 0;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to store request metrics', [
                'error' => $e->getMessage(),
                'path' => $metrics['path'] ?? 'unknown'
            ]);
            return false;
        }
    }
    
    /**
     * Process and store database queries
     */
    private function processDatabaseQueries(array $queries): void
    {
        foreach ($queries as $query) {
            // Skip if query text is empty
            if (empty($query['query'])) {
                continue;
            }
            
            // Create a hash of the query for grouping similar queries
            $queryHash = md5($query['query']);
            
            // Check if query already exists
            $existingSql = "SELECT id, execution_count, avg_execution_time, max_execution_time, min_execution_time 
                        FROM `{$this->databaseQueriesTable}` 
                        WHERE query_hash = :query_hash";
            
            $existingParams = ['query_hash' => $queryHash];
            $existingResult = $this->dbConnection->queryAndFetchAssoc($existingSql, $existingParams);
            
            $now = date('Y-m-d H:i:s');
            $executionTime = $query['execution_time'] ?? 0;
            $params = isset($query['params']) && !empty($query['params']) ? json_encode($query['params']) : null;
            
            if ($existingResult) {
                // Update existing query
                $newCount = $existingResult['execution_count'] + 1;
                
                // Recalculate average
                $newAvg = (($existingResult['avg_execution_time'] * $existingResult['execution_count']) + $executionTime) / $newCount;
                
                // Update max and min
                $newMax = max($existingResult['max_execution_time'], $executionTime);
                $newMin = min($existingResult['min_execution_time'], $executionTime);
                
                $updateSql = "UPDATE `{$this->databaseQueriesTable}` 
                            SET execution_count = :count,
                                avg_execution_time = :avg_time,
                                max_execution_time = :max_time,
                                min_execution_time = :min_time,
                                last_seen = :last_seen,
                                parameters_sample = COALESCE(:params, parameters_sample)
                            WHERE id = :id";
                
                $updateParams = [
                    'count' => $newCount,
                    'avg_time' => $newAvg,
                    'max_time' => $newMax,
                    'min_time' => $newMin,
                    'last_seen' => $now,
                    'params' => $params,
                    'id' => $existingResult['id']
                ];
                
                $this->dbConnection->execute($updateSql, $updateParams);
            } else {
                // Insert new query
                $insertSql = "INSERT INTO `{$this->databaseQueriesTable}` (
                            query_text, query_hash, first_seen, last_seen, 
                            avg_execution_time, max_execution_time, min_execution_time, 
                            execution_count, parameters_sample
                            ) VALUES (
                            :query_text, :query_hash, :first_seen, :last_seen,
                            :avg_time, :max_time, :min_time,
                            :count, :params
                            )";
                
                $insertParams = [
                    'query_text' => $query['query'],
                    'query_hash' => $queryHash,
                    'first_seen' => $now,
                    'last_seen' => $now,
                    'avg_time' => $executionTime,
                    'max_time' => $executionTime,
                    'min_time' => $executionTime,
                    'count' => 1,
                    'params' => $params
                ];
                
                $this->dbConnection->execute($insertSql, $insertParams);
            }
        }
    }

    /**
     * Store application event
     */
    public function storeEvent(string $eventType, array $eventData): bool
    {
        if (!$this->initialized) {
            return false;
        }
        
        try {
            $sql = "INSERT INTO `{$this->eventsTable}` (event_type, event_data, timestamp)
                    VALUES (:event_type, :event_data, :timestamp)";
            
            $timestamp = date('Y-m-d H:i:s');
            $serializedData = json_encode($eventData);
            
            $params = [
                'event_type' => $eventType,
                'event_data' => $serializedData,
                'timestamp' => $timestamp
            ];
            
            return $this->dbConnection->execute($sql, $params) > 0;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to store event', [
                'error' => $e->getMessage(),
                'event_type' => $eventType
            ]);
            return false;
        }
    }

    /**
     * Store system log
     */
    public function storeSystemLog(string $logType, string $component, string $message, ?array $context = null): bool
    {
        if (!$this->initialized) {
            return false;
        }
        
        try {
            $sql = "INSERT INTO `{$this->systemLogsTable}` (timestamp, log_type, component, message, context)
                    VALUES (:timestamp, :log_type, :component, :message, :context)";
            
            $timestamp = date('Y-m-d H:i:s');
            $contextJson = $context ? json_encode($context) : null;
            
            $params = [
                'timestamp' => $timestamp,
                'log_type' => $logType,
                'component' => $component,
                'message' => $message,
                'context' => $contextJson
            ];
            
            return $this->dbConnection->execute($sql, $params) > 0;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to store system log', [
                'error' => $e->getMessage(),
                'component' => $component
            ]);
            return false;
        }
    }
    
    /**
     * Store WebSocket metrics
     */
    public function storeWebsocketMetrics(int $connectedClients, int $messagesSent, int $messagesReceived, 
                                        int $bytesSent, int $bytesReceived, int $errorsCount): bool
    {
        if (!$this->initialized) {
            return false;
        }
        
        try {
            $sql = "INSERT INTO `{$this->websocketMetricsTable}` (
                       timestamp, connected_clients, messages_sent, messages_received, 
                       bytes_sent, bytes_received, errors_count
                    ) VALUES (
                       :timestamp, :connected_clients, :messages_sent, :messages_received,
                       :bytes_sent, :bytes_received, :errors_count
                    )";
            
            $timestamp = date('Y-m-d H:i:s');
            
            $params = [
                'timestamp' => $timestamp,
                'connected_clients' => $connectedClients,
                'messages_sent' => $messagesSent,
                'messages_received' => $messagesReceived,
                'bytes_sent' => $bytesSent,
                'bytes_received' => $bytesReceived,
                'errors_count' => $errorsCount
            ];
            
            return $this->dbConnection->execute($sql, $params) > 0;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to store WebSocket metrics', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Update event count
     */
    public function updateEventCount(string $eventType, int $count): bool
    {
        // Existing implementation
        if (!$this->initialized) {
            return false;
        }
        
        try {
            // First check if the event type already exists
            $sql = "SELECT count FROM `{$this->eventCountsTable}` WHERE event_type = :event_type";
            $params = ['event_type' => $eventType];
            $currentCount = $this->dbConnection->querySingleValue($sql, $params);
            
            $timestamp = date('Y-m-d H:i:s');
            
            if ($currentCount !== null) {
                // Update existing record
                $newCount = (int)$currentCount + $count;
                
                $updateSql = "UPDATE `{$this->eventCountsTable}`
                        SET count = :count, last_updated = :timestamp
                        WHERE event_type = :event_type";
                
                $updateParams = [
                    'count' => $newCount,
                    'timestamp' => $timestamp,
                    'event_type' => $eventType
                ];
                
                return $this->dbConnection->execute($updateSql, $updateParams) > 0;
            } else {
                // Insert new record
                $insertSql = "INSERT INTO `{$this->eventCountsTable}` (event_type, count, last_updated)
                        VALUES (:event_type, :count, :timestamp)";
                
                $insertParams = [
                    'event_type' => $eventType,
                    'count' => $count,
                    'timestamp' => $timestamp
                ];
                
                return $this->dbConnection->execute($insertSql, $insertParams) > 0;
            }
        } catch (\PDOException $e) {
            $this->logger->error('Failed to update event count', [
                'error' => $e->getMessage(),
                'event_type' => $eventType
            ]);
            return false;
        }
    }

    /**
     * Get metrics for a specific time period
     */
    public function getMetrics(string $metricType, \DateTime $from, \DateTime $to): array
    {
        if (!$this->initialized) {
            return [];
        }
        
        try {
            $fromStr = $from->format('Y-m-d H:i:s');
            $toStr = $to->format('Y-m-d H:i:s');
            
            switch ($metricType) {
                case 'requests':
                    return $this->getRequestMetrics($fromStr, $toStr);
                    
                case 'events':
                    return $this->getEventMetrics($fromStr, $toStr);
                    
                case 'performance':
                    return $this->getPerformanceMetrics($fromStr, $toStr);
                    
                case 'memory':
                    return $this->getMemoryMetrics($fromStr, $toStr);
                    
                case 'errors':
                    return $this->getErrorMetrics($fromStr, $toStr);
                    
                case 'database':
                    return $this->getDatabaseMetrics($fromStr, $toStr);
                    
                case 'system_logs':
                    return $this->getSystemLogMetrics($fromStr, $toStr);
                    
                case 'websocket':
                    return $this->getWebsocketMetrics($fromStr, $toStr);
                    
                default:
                    $this->logger->luka('Unknown metric type requested', [
                        'metric_type' => $metricType
                    ]);
                    return [];
            }
        } catch (\PDOException $e) {
            $this->logger->error('Failed to get metrics', [
                'error' => $e->getMessage(),
                'metric_type' => $metricType
            ]);
            return [];
        }
    }

    /**
     * Get request metrics
     */
    private function getRequestMetrics(string $from, string $to): array
    {
        $sql = "SELECT path, method, status_code, COUNT(*) as count, AVG(total_time) as avg_time,
                    MAX(total_time) as max_time, MIN(total_time) as min_time
                FROM `{$this->requestsTable}`
                WHERE timestamp BETWEEN :from AND :to
                GROUP BY path, method, status_code
                ORDER BY count DESC";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get event metrics
     */
    private function getEventMetrics(string $from, string $to): array
    {
        $sql = "SELECT event_type, COUNT(*) as count
                FROM `{$this->eventsTable}`
                WHERE timestamp BETWEEN :from AND :to
                GROUP BY event_type
                ORDER BY count DESC";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get performance metrics
     */
    private function getPerformanceMetrics(string $from, string $to): array
    {
        // Use DATE_FORMAT to group by hour in MySQL
        $sql = "SELECT 
                    DATE_FORMAT(timestamp, '%Y-%m-%d %H:00:00') as hour,
                    COUNT(*) as request_count,
                    AVG(total_time) as avg_response_time,
                    MAX(total_time) as max_response_time
                FROM `{$this->requestsTable}`
                WHERE timestamp BETWEEN :from AND :to
                GROUP BY hour
                ORDER BY hour";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get memory metrics
     */
    private function getMemoryMetrics(string $from, string $to): array
    {
        $sql = "SELECT 
                    DATE_FORMAT(timestamp, '%Y-%m-%d %H:00:00') as hour,
                    AVG(memory_used) as avg_memory,
                    MAX(memory_peak) as peak_memory
                FROM `{$this->requestsTable}`
                WHERE timestamp BETWEEN :from AND :to
                GROUP BY hour
                ORDER BY hour";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get error metrics
     */
    private function getErrorMetrics(string $from, string $to): array
    {
        $sql = "SELECT 
                    path,
                    status_code,
                    COUNT(*) as error_count
                FROM `{$this->requestsTable}`
                WHERE timestamp BETWEEN :from AND :to
                    AND status_code >= 400
                GROUP BY path, status_code
                ORDER BY error_count DESC";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }
    
    /**
     * Get database metrics
     */
    private function getDatabaseMetrics(string $from, string $to): array
    {
        $sql = "SELECT 
                    query_text, 
                    avg_execution_time,
                    max_execution_time,
                    min_execution_time,
                    execution_count,
                    first_seen,
                    last_seen
                FROM `{$this->databaseQueriesTable}`
                WHERE last_seen BETWEEN :from AND :to
                ORDER BY avg_execution_time * execution_count DESC";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }
    
    /**
     * Get system log metrics
     */
    private function getSystemLogMetrics(string $from, string $to): array
    {
        $sql = "SELECT 
                    timestamp,
                    log_type,
                    component,
                    message,
                    context
                FROM `{$this->systemLogsTable}`
                WHERE timestamp BETWEEN :from AND :to
                ORDER BY timestamp DESC";
        
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        $logs = $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
        
        // Decode JSON context
        foreach ($logs as &$log) {
            if (!empty($log['context'])) {
                $log['context'] = json_decode($log['context'], true);
            }
        }
        
        return $logs;
    }
    
    /**
     * Get WebSocket metrics
     */
    private function getWebsocketMetrics(string $from, string $to): array
    {
        $sql = "SELECT 
                    timestamp,
                    connected_clients,
                    messages_sent,
                    messages_received,
                    bytes_sent,
                    bytes_received,
                    errors_count
                FROM `{$this->websocketMetricsTable}`
                WHERE timestamp BETWEEN :from AND :to ORDER BY timestamp";
       
        $params = [
            'from' => $from,
            'to' => $to
        ];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get total count of stored metrics
     */
    public function getMetricsCount(): int
    {
        if (!$this->initialized) {
            return 0;
        }
        
        try {
            $requestsCount = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->requestsTable}`");
            $eventsCount = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->eventsTable}`");
            $databaseQueriesCount = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->databaseQueriesTable}`");
            $systemLogsCount = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->systemLogsTable}`");
            $websocketMetricsCount = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->websocketMetricsTable}`");
            
            return (int)$requestsCount + (int)$eventsCount + (int)$databaseQueriesCount + 
                   (int)$systemLogsCount + (int)$websocketMetricsCount;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to get metrics count', [
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Clean up old metrics data
     */
    public function cleanupOldData(int $retentionDays): bool
    {
        if (!$this->initialized) {
            return false;
        }
        
        try {
            $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$retentionDays} days"));
            
            // Delete old requests
            // Delete old requests
            $requestsDeleted = $this->dbConnection->execute(
                "DELETE FROM `{$this->requestsTable}` WHERE timestamp < :cutoff_date",
                ['cutoff_date' => $cutoffDate]
            );
            
            // Delete old events
            $eventsDeleted = $this->dbConnection->execute(
                "DELETE FROM `{$this->eventsTable}` WHERE timestamp < :cutoff_date",
                ['cutoff_date' => $cutoffDate]
            );
            
            // Delete old database queries (based on last_seen)
            $queriesDeleted = $this->dbConnection->execute(
                "DELETE FROM `{$this->databaseQueriesTable}` WHERE last_seen < :cutoff_date",
                ['cutoff_date' => $cutoffDate]
            );
            
            // Delete old system logs
            $logsDeleted = $this->dbConnection->execute(
                "DELETE FROM `{$this->systemLogsTable}` WHERE timestamp < :cutoff_date",
                ['cutoff_date' => $cutoffDate]
            );
            
            // Delete old WebSocket metrics
            $websocketDeleted = $this->dbConnection->execute(
                "DELETE FROM `{$this->websocketMetricsTable}` WHERE timestamp < :cutoff_date",
                ['cutoff_date' => $cutoffDate]
            );
            
            // Log cleanup results
            $this->logger->luka('Metrics cleanup completed', [
                'requests_deleted' => $requestsDeleted,
                'events_deleted' => $eventsDeleted,
                'queries_deleted' => $queriesDeleted,
                'logs_deleted' => $logsDeleted,
                'websocket_metrics_deleted' => $websocketDeleted,
                'retention_days' => $retentionDays,
                'cutoff_date' => $cutoffDate
            ]);
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to cleanup old metrics data', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get all event types with their counts
     */
    public function getEventTypes(): array
    {
        if (!$this->initialized) {
            return [];
        }
        
        try {
            $sql = "SELECT event_type, count, last_updated
                   FROM `{$this->eventCountsTable}`
                   ORDER BY count DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($sql);
        } catch (\PDOException $e) {
            $this->logger->error('Failed to get event types', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Get a summary of metrics for dashboard display
     */
    public function getMetricsSummary(int $limit = 50): array
    {
        if (!$this->initialized) {
            return [];
        }
        
        try {
            // Get latest summary from cache if available (less than 5 minutes old)
            $sql = "SELECT stat_value FROM `{$this->summaryTable}`
                   WHERE stat_key = 'metrics_summary'
                   AND last_updated > DATE_SUB(NOW(), INTERVAL 5 MINUTE)";
            
            $cachedSummary = $this->dbConnection->querySingleValue($sql);
            
            if ($cachedSummary) {
                return json_decode($cachedSummary, true);
            }
            
            // Calculate summary data
            $summary = [
                'total_requests' => (int)$this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->requestsTable}`"),
                'total_events' => (int)$this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->eventsTable}`"),
                'total_database_queries' => (int)$this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->databaseQueriesTable}`"),
                'total_system_logs' => (int)$this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->systemLogsTable}`"),
                'average_response_time' => (float)$this->dbConnection->querySingleValue("SELECT AVG(total_time) FROM `{$this->requestsTable}`"),
                'error_rate' => $this->calculateErrorRate(),
                'top_paths' => $this->getTopPaths($limit),
                'recent_errors' => $this->getRecentErrors($limit),
                'slow_queries' => $this->getSlowQueries($limit),
                'performance_trend' => $this->getPerformanceTrend(),
                'websocket_stats' => $this->getWebSocketStats()
            ];
            
            // Cache the summary
            $this->updateSummaryStat('metrics_summary', json_encode($summary));
            
            return $summary;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to get metrics summary', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Calculate error rate percentage
     */
    private function calculateErrorRate(): float
    {
        $totalRequests = (int)$this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->requestsTable}`");
        
        if ($totalRequests == 0) {
            return 0.0;
        }
        
        $errorRequests = (int)$this->dbConnection->querySingleValue(
            "SELECT COUNT(*) FROM `{$this->requestsTable}` WHERE status_code >= 400"
        );
        
        return round(($errorRequests / $totalRequests) * 100, 2);
    }

    /**
     * Get top accessed paths
     */
    private function getTopPaths(int $limit): array
    {
        $sql = "SELECT path, COUNT(*) as count, AVG(total_time) as avg_time
               FROM `{$this->requestsTable}`
               GROUP BY path
               ORDER BY count DESC
               LIMIT :limit";
        
        $params = ['limit' => $limit];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get recent errors
     */
    private function getRecentErrors(int $limit): array
    {
        $sql = "SELECT id, path, method, status_code, timestamp, total_time
               FROM `{$this->requestsTable}`
               WHERE status_code >= 400
               ORDER BY timestamp DESC
               LIMIT :limit";
        
        $params = ['limit' => $limit];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }
    
    /**
     * Get slow queries
     */
    private function getSlowQueries(int $limit): array
    {
        $sql = "SELECT 
                   SUBSTRING(query_text, 1, 200) as query_text,
                   avg_execution_time,
                   max_execution_time,
                   execution_count,
                   last_seen
               FROM `{$this->databaseQueriesTable}`
               ORDER BY avg_execution_time DESC
               LIMIT :limit";
        
        $params = ['limit' => $limit];
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }

    /**
     * Get performance trend (hourly averages for the past 24 hours)
     */
    private function getPerformanceTrend(): array
    {
        $sql = "SELECT 
                   DATE_FORMAT(timestamp, '%Y-%m-%d %H:00:00') as hour,
                   COUNT(*) as request_count,
                   AVG(total_time) as avg_response_time
               FROM `{$this->requestsTable}`
               WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
               GROUP BY hour
               ORDER BY hour";
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql);
    }
    
    /**
     * Get WebSocket stats
     */
    private function getWebSocketStats(): array
    {
        $sql = "SELECT 
                   AVG(connected_clients) as avg_clients,
                   MAX(connected_clients) as max_clients,
                   SUM(messages_sent) as total_messages_sent,
                   SUM(messages_received) as total_messages_received,
                   SUM(bytes_sent) as total_bytes_sent,
                   SUM(bytes_received) as total_bytes_received,
                   SUM(errors_count) as total_errors
               FROM `{$this->websocketMetricsTable}`
               WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        
        $result = $this->dbConnection->queryAndFetchAssoc($sql);
        
        // Format bytes to human-readable
        if ($result) {
            $result['total_bytes_sent_formatted'] = $this->formatBytes($result['total_bytes_sent'] ?? 0);
            $result['total_bytes_received_formatted'] = $this->formatBytes($result['total_bytes_received'] ?? 0);
        }
        
        return $result ?: [
            'avg_clients' => 0,
            'max_clients' => 0,
            'total_messages_sent' => 0,
            'total_messages_received' => 0,
            'total_bytes_sent' => 0,
            'total_bytes_received' => 0,
            'total_errors' => 0,
            'total_bytes_sent_formatted' => '0 B',
            'total_bytes_received_formatted' => '0 B'
        ];
    }
    
    /**
     * Format bytes to human-readable format
     */
    private function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= (1 << (10 * $pow));
        
        return round($bytes, $precision) . ' ' . $units[$pow];
    }

    /**
     * Update summary statistics
     */
    private function updateSummaryStats(): void
    {
        try {
            // Update basic stats
            $totalRequests = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->requestsTable}`");
            $avgResponseTime = $this->dbConnection->querySingleValue("SELECT AVG(total_time) FROM `{$this->requestsTable}`");
            $totalErrors = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM `{$this->requestsTable}` WHERE status_code >= 400");
            $totalQueries = $this->dbConnection->querySingleValue("SELECT SUM(execution_count) FROM `{$this->databaseQueriesTable}`");
            
            $this->updateSummaryStat('total_requests', $totalRequests);
            $this->updateSummaryStat('avg_response_time', $avgResponseTime);
            $this->updateSummaryStat('total_errors', $totalErrors);
            $this->updateSummaryStat('total_queries', $totalQueries);
            
        } catch (\PDOException $e) {
            $this->logger->error('Failed to update summary stats', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Update a summary statistic
     */
    private function updateSummaryStat(string $key, $value): void
    {
        try {
            $sql = "INSERT INTO `{$this->summaryTable}` (stat_key, stat_value, last_updated)
                   VALUES (:key, :value, NOW())
                   ON DUPLICATE KEY UPDATE
                   stat_value = VALUES(stat_value),
                   last_updated = VALUES(last_updated)";
            
            $valueStr = is_string($value) ? $value : (string)$value;
            
            $params = [
                'key' => $key,
                'value' => $valueStr
            ];
            
            $this->dbConnection->execute($sql, $params);
        } catch (\PDOException $e) {
            $this->logger->error('Failed to update summary stat', [
                'error' => $e->getMessage(),
                'key' => $key
            ]);
        }
    }
    
    /**
     * Get database query details
     */
    public function getDatabaseQueryDetails(string $queryHash): ?array
    {
        if (!$this->initialized) {
            return null;
        }
        
        try {
            $sql = "SELECT 
                     query_text, 
                     first_seen, 
                     last_seen, 
                     avg_execution_time, 
                     max_execution_time, 
                     min_execution_time, 
                     execution_count, 
                     parameters_sample
                   FROM `{$this->databaseQueriesTable}`
                   WHERE query_hash = :query_hash";
            
            $params = ['query_hash' => $queryHash];
            
            $result = $this->dbConnection->queryAndFetchAssoc($sql, $params);
            
            if (!$result) {
                return null;
            }
            
            // Parse JSON parameters
            if (!empty($result['parameters_sample'])) {
                $result['parameters_sample'] = json_decode($result['parameters_sample'], true);
            }
            
            return $result;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to get database query details', [
                'error' => $e->getMessage(),
                'query_hash' => $queryHash
            ]);
            return null;
        }
    }
    
    /**
     * Get system logs by type
     */
    public function getSystemLogsByType(string $logType, int $limit = 100): array
    {
        if (!$this->initialized) {
            return [];
        }
        
        try {
            $sql = "SELECT 
                     id, 
                     timestamp, 
                     component, 
                     message, 
                     context
                   FROM `{$this->systemLogsTable}`
                   WHERE log_type = :log_type
                   ORDER BY timestamp DESC
                   LIMIT :limit";
            
            $params = [
                'log_type' => $logType,
                'limit' => $limit
            ];
            
            $logs = $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
            
            // Decode JSON context
            foreach ($logs as &$log) {
                if (!empty($log['context'])) {
                    $log['context'] = json_decode($log['context'], true);
                }
            }
            
            return $logs;
        } catch (\PDOException $e) {
            $this->logger->error('Failed to get system logs by type', [
                'error' => $e->getMessage(),
                'log_type' => $logType
            ]);
            return [];
        }
    }
}