<?php

namespace baseKRIZAN\LUKA\Storage;

/**
 * Interface for metrics storage implementations
 */
interface MetricsStorage
{
    /**
     * Store request metrics
     * 
     * @param array $metrics The metrics data to store
     * @return bool Success status
     */
    public function storeRequestMetrics(array $metrics): bool;
    
    /**
     * Store application event
     * 
     * @param string $eventType Type of event
     * @param array $eventData Event data
     * @return bool Success status
     */
    public function storeEvent(string $eventType, array $eventData): bool;
    
    /**
     * Update event count for a specific event type
     * 
     * @param string $eventType Type of event
     * @param int $count Number of occurrences
     * @return bool Success status
     */
    public function updateEventCount(string $eventType, int $count): bool;
    
    /**
     * Get metrics for a specific time period
     * 
     * @param string $metricType Type of metrics to retrieve
     * @param \DateTime $from Start time
     * @param \DateTime $to End time
     * @return array Metrics data
     */
    public function getMetrics(string $metricType, \DateTime $from, \DateTime $to): array;
    
    /**
     * Get total count of stored metrics
     * 
     * @return int Count of metrics
     */
    public function getMetricsCount(): int;
    
    /**
     * Clean up old metrics data
     * 
     * @param int $retentionDays Number of days to keep data
     * @return bool Success status
     */
    public function cleanupOldData(int $retentionDays): bool;
    
    /**
     * Initialize storage - create necessary tables/structures
     * 
     * @return bool Success status
     */
    public function initialize(): bool;
    
    /**
     * Get all event types with their counts
     * 
     * @return array Event types and counts
     */
    public function getEventTypes(): array;
    
    /**
     * Get a summary of metrics for dashboard display
     * 
     * @param int $limit Maximum number of records to return
     * @return array Summary data
     */
    public function getMetricsSummary(int $limit = 50): array;

    /**
     * Store system log
     * 
     * @param string $logType Type of log (error, warning, info, debug)
     * @param string $component Component that generated the log
     * @param string $message Log message
     * @param array|null $context Additional context data
     * @return bool Success status
     */
    public function storeSystemLog(string $logType, string $component, string $message, ?array $context = null): bool;
    
    /**
     * Store WebSocket metrics
     * 
     * @param int $connectedClients Number of connected clients
     * @param int $messagesSent Number of messages sent
     * @param int $messagesReceived Number of messages received
     * @param int $bytesSent Number of bytes sent
     * @param int $bytesReceived Number of bytes received
     * @param int $errorsCount Number of errors
     * @return bool Success status
     */
    public function storeWebsocketMetrics(int $connectedClients, int $messagesSent, int $messagesReceived, 
                                        int $bytesSent, int $bytesReceived, int $errorsCount): bool;
}