<?php
// app/baseKRIZAN/Modules/AbstractModule.php

namespace baseKRIZAN\Modules;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Events\EventDispatcher;

/**
 * Apstraktna bazna klasa za implementaciju modula
 * 
 * @package baseKRIZAN\Modules
 */
abstract class AbstractModule implements ModuleInterface
{
    /**
     * Meta podaci modula
     * 
     * @var array
     */
    protected array $metadata = [];
    
    /**
     * Putanja do direktorija modula
     * 
     * @var string
     */
    protected string $path;
    
    /**
     * Logger instanca
     * 
     * @var Logger|null
     */
    protected ?Logger $logger = null;
    
    /**
     * EventDispatcher instanca
     * 
     * @var EventDispatcher|null
     */
    protected ?EventDispatcher $eventDispatcher = null;
    
    /**
     * Status aktivacije modula
     * 
     * @var bool
     */
    protected bool $isEnabled = false;
    
    /**
     * Konstruktor
     * 
     * @param string $path Putanja do direktorija modula
     * @param array $metadata Meta podaci iz module.json
     */
    public function __construct(string $path, array $metadata = [])
    {
        $this->path = $path;
        $this->metadata = $metadata;
        
        // Popratni status aktivacije iz metapodataka
        $this->isEnabled = $metadata['enabled'] ?? false;
    }
    
    /**
     * @inheritDoc
     */
    public function getId(): string
    {
        return $this->metadata['id'] ?? basename($this->path);
    }
    
    /**
     * @inheritDoc
     */
    public function getName(): string
    {
        return $this->metadata['name'] ?? basename($this->path);
    }
    
    /**
     * @inheritDoc
     */
    public function getDescription(): string
    {
        return $this->metadata['description'] ?? '';
    }
    
    /**
     * @inheritDoc
     */
    public function getVersion(): string
    {
        return $this->metadata['version'] ?? '1.0.0';
    }
    
    /**
     * @inheritDoc
     */
    public function getPath(): string
    {
        return $this->path;
    }
    
    /**
     * @inheritDoc
     */
    public function getDependencies(): array
    {
        return $this->metadata['dependencies'] ?? [];
    }
    
    /**
     * @inheritDoc
     */
    public function register(Container $container, Logger $logger, EventDispatcher $eventDispatcher): bool
    {
        $this->logger = $logger;
        $this->eventDispatcher = $eventDispatcher;
        
        $logger->modules('Registering module', [
            'id' => $this->getId(),
            'name' => $this->getName(),
            'version' => $this->getVersion()
        ]);
        
        try {
            // Registriranje modula u kontejner
            $container->register('module.' . $this->getId(), $this);
            
            // Registriramo event listenere
            $this->registerEventListeners($eventDispatcher);
            
            return true;
        } catch (\Throwable $e) {
            $logger->error('Failed to register module', [
                'id' => $this->getId(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function bootstrap(Container $container): bool
    {
        if (!$this->isEnabled) {
            if ($this->logger) {
                $this->logger->modules('Skipping bootstrap for disabled module', [
                    'id' => $this->getId()
                ]);
            }
            return false;
        }
        
        // Učitavanje dependencies.php datoteke ako postoji
        $dependenciesFile = $this->path . '/' . $this->getId() . '_dependencies.php';
        
        if (file_exists($dependenciesFile)) {
            if ($this->logger) {
                $this->logger->modules('Loading module dependencies', [
                    'id' => $this->getId(),
                    'file' => $dependenciesFile
                ]);
            }
            
            try {
                $loader = include $dependenciesFile;
                if (is_callable($loader)) {
                    $loader($container);
                    
                    if ($this->logger) {
                        $this->logger->modules('Module dependencies loaded', [
                            'id' => $this->getId()
                        ]);
                    }
                }
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to load module dependencies', [
                        'id' => $this->getId(),
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                return false;
            }
        }
        
        // Učitavanje bootstrap.php datoteke ako postoji
        $bootstrapFile = $this->path . '/' . $this->getId() . '_bootstrap.php';
        
        if (file_exists($bootstrapFile)) {
            if ($this->logger) {
                $this->logger->modules('Loading module bootstrap', [
                    'id' => $this->getId(),
                    'file' => $bootstrapFile
                ]);
            }
            
            try {
                include $bootstrapFile;
                
                if ($this->logger) {
                    $this->logger->modules('Module bootstrap loaded', [
                        'id' => $this->getId()
                    ]);
                }
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to load module bootstrap', [
                        'id' => $this->getId(),
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * @inheritDoc
     */
    public function activate(Container $container): bool
    {
        if ($this->isEnabled) {
            return true; // Već je aktiviran
        }
        
        if ($this->logger) {
            $this->logger->modules('Activating module', [
                'id' => $this->getId()
            ]);
        }
        
        // Pozovite bootstrap za inicijalizaciju
        if ($this->bootstrap($container)) {
            $this->isEnabled = true;
            
            // Emitiranje događaja
            if ($this->eventDispatcher) {
                $this->eventDispatcher->dispatch('module.activated', [
                    'id' => $this->getId(),
                    'name' => $this->getName(),
                    'version' => $this->getVersion()
                ]);
            }
            
            if ($this->logger) {
                $this->logger->modules('Module activated successfully', [
                    'id' => $this->getId()
                ]);
            }
            
            return true;
        }
        
        return false;
    }
    
    /**
     * @inheritDoc
     */
    public function deactivate(Container $container): bool
    {
        if (!$this->isEnabled) {
            return true; // Već je deaktiviran
        }
        
        if ($this->logger) {
            $this->logger->modules('Deactivating module', [
                'id' => $this->getId()
            ]);
        }
        
        // Implementacija za deaktivaciju - potomci mogu proširiti
        $this->isEnabled = false;
        
        // Emitiranje događaja
        if ($this->eventDispatcher) {
            $this->eventDispatcher->dispatch('module.deactivated', [
                'id' => $this->getId(),
                'name' => $this->getName(),
                'version' => $this->getVersion()
            ]);
        }
        
        if ($this->logger) {
            $this->logger->modules('Module deactivated successfully', [
                'id' => $this->getId()
            ]);
        }
        
        return true;
    }
    
    /**
     * @inheritDoc
     */
    public function isEnabled(): bool
    {
        return $this->isEnabled;
    }
    
    /**
     * Dobiva metadata vrijednost prema ključu
     * 
     * @param string $key Ključ
     * @param mixed $default Zadana vrijednost ako ključ ne postoji
     * @return mixed
     */
    public function getMetadata(string $key, $default = null)
    {
        return $this->metadata[$key] ?? $default;
    }
    
    /**
     * Registrira event listenere za modul
     * 
     * @param EventDispatcher $eventDispatcher
     */
    protected function registerEventListeners(EventDispatcher $eventDispatcher): void
    {
        // Implementacija u potomcima
    }
}