<?php
// app/baseKRIZAN/Modules/GenericModule.php

namespace baseKRIZAN\Modules;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Events\EventDispatcher;

/**
 * Generička implementacija modula za module koji nemaju vlastitu implementaciju
 * 
 * @package baseKRIZAN\Modules
 */
class GenericModule extends AbstractModule
{
    /**
     * @inheritDoc
     */
    public function bootstrap(Container $container): bool
    {
        if (!$this->isEnabled) {
            if ($this->logger) {
                $this->logger->modules('Skipping bootstrap for disabled module', [
                    'id' => $this->getId()
                ]);
            }
            return false;
        }
        
        // Učitavanje dependencies.php datoteke ako postoji
        $dependenciesFile = $this->path . '/' . $this->getId() . '_dependencies.php';
        
        if (file_exists($dependenciesFile)) {
            if ($this->logger) {
                $this->logger->modules('Loading module dependencies', [
                    'id' => $this->getId(),
                    'file' => $dependenciesFile
                ]);
            }
            
            try {
                $loader = include $dependenciesFile;
                if (is_callable($loader)) {
                    $loader($container);
                    
                    if ($this->logger) {
                        $this->logger->modules('Module dependencies loaded', [
                            'id' => $this->getId()
                        ]);
                    }
                }
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to load module dependencies', [
                        'id' => $this->getId(),
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                return false;
            }
        }
        
        // Učitavanje bootstrap.php datoteke ako postoji
        $bootstrapFile = $this->path . '/' . $this->getId() . '_bootstrap.php';
        
        if (file_exists($bootstrapFile)) {
            if ($this->logger) {
                $this->logger->modules('Loading module bootstrap', [
                    'id' => $this->getId(),
                    'file' => $bootstrapFile
                ]);
            }
            
            try {
                include $bootstrapFile;
                
                if ($this->logger) {
                    $this->logger->modules('Module bootstrap loaded', [
                        'id' => $this->getId()
                    ]);
                }
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to load module bootstrap', [
                        'id' => $this->getId(),
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                return false;
            }
        }
        
        $routesFile = $this->path . '/' . $this->getId() . '_routes.php';
        
        if (file_exists($routesFile) && $container->has('routeBuilder')) {
            if ($this->logger) {
                $this->logger->modules('Loading module routes', [
                    'id' => $this->getId(),
                    'file' => $routesFile
                ]);
            }
            
            try {
                $builder = $container->get('routeBuilder');
                
                $routeBuilder = $container->get('routeBuilder');
                // Sad koristimo $routeBuilder umjesto $builder u include kontekstu
                $logger = $this->logger;
                $currentModule = $this->getId();
                
                include $routesFile;
                
                if ($this->logger) {
                    $this->logger->modules('Module routes loaded', [
                        'id' => $this->getId()
                    ]);
                }
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to load module routes', [
                        'id' => $this->getId(),
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                // Nastavite s učitavanjem - rute nisu kritične za funkcionalnost modula
            }
        }
        
        return true;
    }
    
    /**
     * @inheritDoc
     */
    protected function registerEventListeners(EventDispatcher $eventDispatcher): void
    {
        // Provjeri postoji li events.php datoteka
        $eventsFile = $this->path . '/' . $this->getId() . '_events.php';
        
        if (file_exists($eventsFile)) {
            if ($this->logger) {
                $this->logger->modules('Loading module event listeners', [
                    'id' => $this->getId(),
                    'file' => $eventsFile
                ]);
            }
            
            try {
                $events = include $eventsFile;
                
                if (is_array($events)) {
                    foreach ($events as $eventName => $listeners) {
                        if (is_array($listeners)) {
                            foreach ($listeners as $listener) {
                                if (is_callable($listener)) {
                                    $eventDispatcher->addListener($eventName, $listener);
                                    
                                    if ($this->logger) {
                                        $this->logger->modules('Registered event listener', [
                                            'id' => $this->getId(),
                                            'event' => $eventName
                                        ]);
                                    }
                                }
                            }
                        }
                    }
                }
            } catch (\Throwable $e) {
                if ($this->logger) {
                    $this->logger->error('Failed to load module event listeners', [
                        'id' => $this->getId(),
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }
        }
    }
}