<?php
// app/baseKRIZAN/Modules/ModuleManager.php

namespace baseKRIZAN\Modules;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Events\EventDispatcher;

/**
 * Upravlja modulima u sustavu - učitavanje, aktivacija, deaktivacija
 * 
 * @package baseKRIZAN\Modules
 */
class ModuleManager
{
    private static ?self $instance = null;
    
    /**
     * @var array<string, ModuleInterface> Učitani moduli (instancirani)
     */
    private array $modules = [];
    
    /**
     * @var array<string, array> Metapodaci o modulima
     */
    private array $moduleMetadata = [];
    
    /**
     * @var array<string> Lista aktiviranih modula
     */
    private array $enabledModules = [];
    
    /**
     * @var Logger Logger
     */
    private Logger $logger;
    
    /**
     * @var Container Container
     */
    private Container $container;
    
    /**
     * @var EventDispatcher EventDispatcher
     */
    private EventDispatcher $eventDispatcher;
    
    /**
     * Dohvaća singleton instancu ModuleManager-a
     * 
     * @param Logger $logger
     * @param Container $container
     * @param EventDispatcher $eventDispatcher
     * @return self
     */
    public static function getInstance(
        Logger $logger, 
        Container $container,
        ?EventDispatcher $eventDispatcher = null
    ): self {
        if (self::$instance === null) {
            // Ako event dispatcher nije dostavljen, pokušaj dohvatiti iz containera
            if ($eventDispatcher === null && $container->has('eventDispatcher')) {
                $eventDispatcher = $container->get('eventDispatcher');
            }
            
            // Ako još uvijek nema event dispatcher-a, kreiraj novi
            if ($eventDispatcher === null) {
                $eventDispatcher = new EventDispatcher($logger);
                $container->register('eventDispatcher', $eventDispatcher);
            }
            
            self::$instance = new self($logger, $container, $eventDispatcher);
        }
        
        return self::$instance;
    }
    
    /**
     * Privatni konstruktor za singleton
     * 
     * @param Logger $logger
     * @param Container $container
     * @param EventDispatcher $eventDispatcher
     */
    private function __construct(Logger $logger, Container $container, EventDispatcher $eventDispatcher)
    {
        $this->logger = $logger;
        $this->container = $container;
        $this->eventDispatcher = $eventDispatcher;
        
        // Inicijaliziraj ModuleRegistry
        ModuleRegistry::init($logger);
        
        $this->discoverModules();
    }
    
    /**
     * Otkriva instalirane module i učitava njihove metapodatke
     */
    private function discoverModules(): void
    {
        $modulesDir = \baseKRIZAN\Config\Config::get('paths.moduli') . '/';
        $enabledModules = explode(',', \baseKRIZAN\Config\Config::get('moduli'));
        $this->enabledModules = array_map('trim', $enabledModules);
        
        $this->logger->modules('Discovering modules from directory', [
            'modules_dir' => $modulesDir,
            'enabled_modules' => $this->enabledModules
        ]);
        
        if (!is_dir($modulesDir)) {
            $this->logger->modules('Modules directory not found', ['path' => $modulesDir]);
            return;
        }
        
        $dirEntries = scandir($modulesDir);
        if ($dirEntries === false) {
            $this->logger->modules('Failed to scan modules directory', ['path' => $modulesDir]);
            return;
        }
        
        foreach ($dirEntries as $entry) {
            if ($entry === '.' || $entry === '..') {
                continue;
            }
            
            $modulePath = $modulesDir . $entry;
            if (!is_dir($modulePath)) {
                continue;
            }
            
            $this->logger->modules('Found potential module directory', [
                'name' => $entry,
                'path' => $modulePath
            ]);
            
            // Provjeri module.json
            $moduleInfoFile = $modulePath . '/module.json';
            if (file_exists($moduleInfoFile)) {
                $moduleInfo = $this->loadModuleInfo($moduleInfoFile);
                if ($moduleInfo) {
                    $this->moduleMetadata[$entry] = $moduleInfo;
                    
                    // Registriraj u ModuleRegistry
                    ModuleRegistry::register($entry, $moduleInfo);
                    
                    $this->logger->modules('Registered module with info file', [
                        'name' => $entry,
                        'version' => $moduleInfo['version'] ?? 'unknown',
                        'enabled' => $moduleInfo['enabled'] ? 'yes' : 'no'
                    ]);
                }
            } else {
                // Legacy modul bez module.json
                $moduleInfo = [
                    'id' => $entry,
                    'name' => $entry,
                    'enabled' => in_array($entry, $this->enabledModules),
                    'version' => '1.0.0',
                    'description' => 'Legacy module',
                    'dependencies' => []
                ];
                
                $this->moduleMetadata[$entry] = $moduleInfo;
                
                // Registriraj u ModuleRegistry
                ModuleRegistry::register($entry, $moduleInfo);
                
                $this->logger->modules('Registered legacy module without info file', [
                    'name' => $entry,
                    'enabled' => in_array($entry, $this->enabledModules) ? 'yes' : 'no'
                ]);
            }
        }
        
        $this->logger->modules('Module discovery completed', [
            'count' => count($this->moduleMetadata),
            'modules' => array_keys($this->moduleMetadata)
        ]);
    }
    
    /**
     * Učitava informacije o modulu iz module.json datoteke
     * 
     * @param string $infoFile Putanja do module.json datoteke
     * @return array|null
     */
    private function loadModuleInfo(string $infoFile): ?array
    {
        $this->logger->modules('Loading module info file', ['file' => $infoFile]);
        
        $content = file_get_contents($infoFile);
        if ($content === false) {
            $this->logger->modules('Failed to read module info file', ['file' => $infoFile]);
            return null;
        }
        
        $info = json_decode($content, true);
        if ($info === null) {
            $this->logger->modules('Invalid module info file', [
                'file' => $infoFile,
                'error' => json_last_error_msg()
            ]);
            return null;
        }
        
        $moduleName = basename(dirname($infoFile));
        
        // Ako id polje nije postavljeno, koristi ime direktorija
        if (!isset($info['id'])) {
            $info['id'] = $moduleName;
        }
        
        // Ako name polje nije postavljeno, koristi ime direktorija
        if (!isset($info['name'])) {
            $info['name'] = $moduleName;
        }
        
        // Postavi enabled status prema config-u
        $info['enabled'] = in_array($moduleName, $this->enabledModules);
        
        $this->logger->modules('Module info loaded successfully', [
            'id' => $info['id'],
            'name' => $info['name'],
            'enabled' => $info['enabled'] ? 'yes' : 'no',
            'version' => $info['version'] ?? 'unknown',
            'description' => $info['description'] ?? 'No description'
        ]);
        
        return $info;
    }
    
    /**
     * Učitava i aktivira module na temelju ovisnosti
     */
    public function loadModules(): void
    {
        $this->logger->modules('Starting to load enabled modules');
        
        // Sortiraj module prema ovisnostima
        $sortedModules = $this->sortModulesByDependencies();
        
        $this->logger->modules('Modules sorted by dependencies', [
            'order' => $sortedModules
        ]);
        
        foreach ($sortedModules as $moduleId) {
            if (!isset($this->moduleMetadata[$moduleId]) || !$this->moduleMetadata[$moduleId]['enabled']) {
                $this->logger->modules('Skipping disabled module', ['id' => $moduleId]);
                continue;
            }
            
            $this->logger->modules('Loading module', ['id' => $moduleId]);
            
            // Provjeri jesu li sve ovisnosti učitane
            $dependencies = $this->moduleMetadata[$moduleId]['dependencies'] ?? [];
            $missingDependencies = [];
            
            foreach ($dependencies as $dependency) {
                if (!isset($this->modules[$dependency]) || !$this->modules[$dependency]->isEnabled()) {
                    $missingDependencies[] = $dependency;
                }
            }
            
            if (!empty($missingDependencies)) {
                $this->logger->modules('Module has missing dependencies', [
                    'id' => $moduleId,
                    'missing_dependencies' => $missingDependencies
                ]);
                continue;
            }
            
            if ($this->loadAndActivateModule($moduleId)) {
                $this->logger->modules('Module loaded and activated successfully', ['id' => $moduleId]);
            } else {
                $this->logger->modules('Failed to load or activate module', ['id' => $moduleId]);
            }
        }
        
        $this->logger->modules('Module loading completed', [
            'count' => count($this->modules),
            'loaded_modules' => array_keys($this->modules)
        ]);
        
        // Emitiraj događaj da su svi moduli učitani
        $this->eventDispatcher->dispatch('modules.all_loaded', [
            'modules' => array_keys($this->modules)
        ]);
    }
    
    /**
     * Učitava i aktivira modul prema ID-u
     * 
     * @param string $moduleId ID modula
     * @return bool
     */
    private function loadAndActivateModule(string $moduleId): bool
    {
        $modulesDir = \baseKRIZAN\Config\Config::get('paths.moduli') . '/';
        $modulePath = $modulesDir . $moduleId;
        
        if (!isset($this->moduleMetadata[$moduleId])) {
            $this->logger->modules('Module metadata not found', ['id' => $moduleId]);
            return false;
        }
        
        $metadata = $this->moduleMetadata[$moduleId];
        
        // Provjeri ima li modul vlastitu implementaciju ModuleInterface
        $moduleClassName = '\\' . $moduleId . '\\Module';
        if (class_exists($moduleClassName) && in_array(ModuleInterface::class, class_implements($moduleClassName))) {
            $this->logger->modules('Using module class implementation', [
                'id' => $moduleId,
                'class' => $moduleClassName
            ]);
            
            $moduleInstance = new $moduleClassName($modulePath, $metadata);
        } else {
            // Generički modul
            $moduleInstance = new GenericModule($modulePath, $metadata);
        }
        
        // Registriraj modul
        if (!$moduleInstance->register($this->container, $this->logger, $this->eventDispatcher)) {
            $this->logger->error('Failed to register module', ['id' => $moduleId]);
            return false;
        }
        
        // Učitaj dependencije modula
        $this->loadModuleDependencies($moduleId);
        
        // Učitaj rute modula ako je RouteBuilder dostupan
        if ($this->container->has('routeBuilder')) {
            $this->loadModuleRoutes($moduleId);
        }
        
        // Aktiviraj modul ako je omogućen
        if ($metadata['enabled'] && !$moduleInstance->activate($this->container)) {
            $this->logger->error('Failed to activate module', ['id' => $moduleId]);
            return false;
        }
        
        // Dodaj instancu u internu mapu
        $this->modules[$moduleId] = $moduleInstance;
        
        return true;
    }

    /**
     * Učitava zavisnosti za modul
     * 
     * @param string $moduleId ID modula
     * @return bool True ako je uspješno, false inače
     */
    private function loadModuleDependencies(string $moduleId): bool
    {
        $modulesDir = \baseKRIZAN\Config\Config::get('paths.moduli') . '/';
        $modulePath = $modulesDir . $moduleId;
        $dependenciesFile = $modulePath . '/' . $moduleId . '_dependencies.php';
        
        if (!file_exists($dependenciesFile)) {
            $this->logger->modules('Module dependencies file not found', [
                'id' => $moduleId,
                'file' => $dependenciesFile
            ]);
            return false;
        }
        
        try {
            $this->logger->modules('Loading module dependencies', [
                'id' => $moduleId,
                'file' => $dependenciesFile
            ]);
            
            $loader = include $dependenciesFile;
            
            if (is_callable($loader)) {
                $loader($this->container);
                
                $this->logger->modules('Module dependencies loaded successfully', [
                    'id' => $moduleId
                ]);
                
                return true;
            } else {
                $this->logger->modules('Module dependencies file did not return a callable function', [
                    'id' => $moduleId,
                    'file' => $dependenciesFile
                ]);
                return false;
            }
        } catch (\Throwable $e) {
            $this->logger->error('Failed to load module dependencies', [
                'id' => $moduleId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Učitava rute za modul
     * 
     * @param string $moduleId ID modula
     * @return bool True ako je uspješno, false inače
     */
    private function loadModuleRoutes(string $moduleId): bool
    {
        // Osiguraj da je RouteBuilder dostupan
        if (!$this->container->has('routeBuilder')) {
            $this->logger->modules('RouteBuilder not available, cannot load module routes', [
                'id' => $moduleId
            ]);
            return false;
        }
        
        $routeBuilder = $this->container->get('routeBuilder');
        $modulesDir = \baseKRIZAN\Config\Config::get('paths.moduli') . '/';
        $modulePath = $modulesDir . $moduleId;
        $routesFile = $modulePath . '/' . $moduleId . '_routes.php';
        
        if (!file_exists($routesFile)) {
            $this->logger->modules('Module routes file not found', [
                'id' => $moduleId,
                'file' => $routesFile
            ]);
            return false;
        }
        
        try {
            $this->logger->modules('Loading module routes', [
                'id' => $moduleId,
                'file' => $routesFile
            ]);
            
            $container = $this->container;
            $builder = $routeBuilder;
            $logger = $this->logger;
            $currentModule = $moduleId; // Set module context
            
            // Load routes
            include($routesFile);
            
            $this->logger->modules('Module routes loaded successfully', [
                'id' => $moduleId
            ]);
            
            // Register routes with Router if available
            if ($this->container->has('routes')) {
                $router = $this->container->get('routes');
                if (method_exists($router, 'setRoutes')) {
                    $router->setRoutes($builder->getRoutes());
                    $this->logger->modules('Module routes registered with Router', [
                        'id' => $moduleId
                    ]);
                }
            }
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to load module routes', [
                'id' => $moduleId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }
    
    /**
     * Sortira module prema ovisnostima koristeći topološko sortiranje
     * 
     * @return array Sortirani niz ID-ova modula
     */
    private function sortModulesByDependencies(): array
    {
        $modules = array_keys($this->moduleMetadata);
        $dependencies = [];
        $sorted = [];
        $visited = [];
        $temp = [];
        
        // Pripremi graf ovisnosti
        foreach ($modules as $module) {
            $deps = $this->moduleMetadata[$module]['dependencies'] ?? [];
            $dependencies[$module] = $deps;
        }
        
        // Funkcija za topološko sortiranje
        $visit = function($module) use (&$visit, &$sorted, &$visited, &$temp, &$dependencies) {
            // Ako je modul već sortiran, preskočimo ga
            if (isset($visited[$module])) {
                return;
            }
            
            // Provjera cikličkih ovisnosti
            if (isset($temp[$module])) {
                $this->logger->error('Circular dependency detected', [
                    'module' => $module,
                    'dependencies' => $dependencies[$module] ?? []
                ]);
                return;
            }
            
            // Označavamo modul kao posjećen u trenutnom prolazu
            $temp[$module] = true;
            
            // Posjećujemo sve ovisnosti
            foreach ($dependencies[$module] ?? [] as $dependency) {
                if (isset($this->moduleMetadata[$dependency])) {
                    $visit($dependency);
                }
            }
            
            // Označavamo modul kao potpuno obrađen
            unset($temp[$module]);
            $visited[$module] = true;
            $sorted[] = $module;
        };
        
        // Posjeti sve module
        foreach ($modules as $module) {
            $visit($module);
        }
        
        return $sorted;
    }
    
    /**
     * Aktivira modul prema ID-u
     * 
     * @param string $moduleId ID modula
     * @return bool
     */
    public function activateModule(string $moduleId): bool
    {
        // Ako modul nije učitan, pokušaj ga učitati
        if (!isset($this->modules[$moduleId])) {
            if (!$this->loadAndActivateModule($moduleId)) {
                return false;
            }
            return true;
        }
        
        $module = $this->modules[$moduleId];
        
        // Aktiviraj modul
        return $module->activate($this->container);
    }
    
    /**
     * Deaktivira modul prema ID-u
     * 
     * @param string $moduleId ID modula
     * @return bool
     */
    public function deactivateModule(string $moduleId): bool
    {
        if (!isset($this->modules[$moduleId])) {
            $this->logger->modules('Cannot deactivate module - not loaded', ['id' => $moduleId]);
            return false;
        }
        
        $module = $this->modules[$moduleId];
        
        // Provjeri ovisnosti - ne možemo deaktivirati modul koji je potreban drugim aktivnim modulima
        $dependentModules = $this->getDependentModules($moduleId);
        if (!empty($dependentModules)) {
            $this->logger->modules('Cannot deactivate module - other modules depend on it', [
                'id' => $moduleId,
                'dependent_modules' => $dependentModules
            ]);
            return false;
        }
        
        // Deaktiviraj modul
        return $module->deactivate($this->container);
    }
    
    /**
     * Vraća listu modula koji ovise o određenom modulu
     * 
     * @param string $moduleId ID modula
     * @return array Lista ID-ova modula koji ovise o zadanom modulu
     */
    public function getDependentModules(string $moduleId): array
    {
        $dependent = [];
        
        foreach ($this->modules as $id => $module) {
            if ($module->isEnabled() && in_array($moduleId, $module->getDependencies())) {
                $dependent[] = $id;
            }
        }
        
        return $dependent;
    }
    
    /**
     * Vraća instancu modula prema ID-u
     * 
     * @param string $moduleId ID modula
     * @return ModuleInterface|null
     */
    public function getModule(string $moduleId): ?ModuleInterface
    {
        return $this->modules[$moduleId] ?? null;
    }
    
    /**
     * Vraća sve učitane module
     * 
     * @return array<string, ModuleInterface>
     */
    public function getModules(): array
    {
        return $this->modules;
    }
    
    /**
     * Vraća sve učitane i aktivirane module
     * 
     * @return array<string, ModuleInterface>
     */
    public function getEnabledModules(): array
    {
        return array_filter($this->modules, function (ModuleInterface $module) {
            return $module->isEnabled();
        });
    }
    
    /**
     * Vraća metapodatke o modulima
     * 
     * @return array
     */
    public function getMetadata(): array
    {
        return $this->moduleMetadata;
    }
    
    /**
     * Provjerava je li modul aktiviran
     * 
     * @param string $moduleId ID modula
     * @return bool
     */
    public function isModuleEnabled(string $moduleId): bool
    {
        return isset($this->modules[$moduleId]) && $this->modules[$moduleId]->isEnabled();
    }
    
    /**
     * Provjerava je li modul učitan
     * 
     * @param string $moduleId ID modula
     * @return bool
     */
    public function isModuleLoaded(string $moduleId): bool
    {
        return isset($this->modules[$moduleId]);
    }
}